//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2016. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.client;


/**
 * This interface solves the problem of start up and shutting
 * down from a non-eBus thread. When an application object is
 * opening feeds, it is possible that eBus will call the object
 * back before the object's start up method is completed.
 * Synchronization must then be used to prevent start up
 * interruption. But synchronization should not be necessary
 * due to eBus guaranteeing clients that it will access clients
 * in a single-threaded fashion.
 * <p>
 * The solution is to have an eBus thread trigger the object
 * start up and shutdown methods. While the object is opening
 * feeds, any and all callbacks triggered by the feed opening
 * will not be posted until after the start up method returns.
 * </p>
 * <p>
 * A similar issue exists with shutting down application objects
 * from a non-eBus thread: after a feed is closed, the feed may
 * still deliver messages to the application object. Shutting
 * down feeds from an eBus thread removes this issue since all
 * the object's pending message queue will be cleared.
 * </p>
 * <p>
 * An application can either override these methods or
 * provide a {@link Runnable} lambda expression as the start up
 * and/or shutdown methods to
 * {@link EFeed#register(EObject, String)}.
 * </p>
 *
 * @author <a href="mailto:rapp@cboe.com">Charles Rapp</a>
 */

public interface EObject
{
//---------------------------------------------------------------
// Member methods.
//

    /**
     * {@link EFeed#startup(EObject)} calls this method to
     * asynchronously start up an eBus client from an eBus
     * thread. This means that object start up processing will
     * not be interrupted by an eBus callback.
     * <p>
     * The default implementation does nothing. This method can
     * be "overridden" using a {@link Runnable} interface
     * implementation.
     * </p>
     *
     * @see #shutdown()
     */
    default void startup()
    {}

    /**
     * {@link EFeed#shutdown(EObject)} calls this method to
     * asynchronously shutdown an eBus client from an eBus
     * thread. This means that object shutdown processing will
     * not be interrupted by an eBus callback.
     * <p>
     * <strong>Note:</strong> having eBus shutdown an application
     * object results in <em>all</em> open eBus feeds associated
     * with that application object being closed. It is not
     * possible to partially shutdown the object's feeds. The
     * eBus shutdown functionality should be used only when the
     * object is no longer needed.
     * </p>
     * <p>
     * The default implementation does nothing. This method can
     * be "overridden" using a {@link Runnable} interface
     * implementation.
     * </p>
     *
     * @see #startup()
     */
    default void shutdown()
    {}
} // end of interface EObject

