//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2015, 2016, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client;

import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Abstract base class for {@link EPublishFeed} and
 * {@link ESubscribeFeed} classes. Tracks the current
 * {@link EFeedState feed state}.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public abstract class ENotifyFeed
    extends ESingleFeed
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Statics.
    //

    /**
     * Logging subsystem interface.
     */
    private static final Logger sLogger =
        Logger.getLogger(ENotifyFeed.class.getName());

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new notification feed instance.
     * @param client route eBus messages to this client.
     * @param scope whether this feed supports local and/or
     * remote eBus client feeds.
     * @param feedType the actual feed type.
     * @param subject interact with this subject.
     */
    protected ENotifyFeed(final EClient client,
                          final FeedScope scope,
                          final FeedType feedType,
                          final ESubject subject)
    {
        super (client, scope, feedType, subject);
    } // end of ENotifyFeed(...)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Abstract Method Declarations.
    //

    /**
     * Issues the appropriate client callback, informing the
     * client that the feed state is changed. For a publish
     * feed, issues a publish state callback; for a subscribe
     * feed, issues a subscribe state callback.
     * @param feedState the updated feed state.
     */
    /* package */ abstract void update(EFeedState feedState);

    //
    // end of Abstract Method Declarations.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Set Methods.
    //

    /**
     * Sets the initial activation count to the given value and
     * sets the feed state accordingly.
     * @param actCount the initial activation count.
     * @return the feed's state as a result of this activation
     * count.
     */
    /* package */ EFeedState updateActivate(final int actCount)
    {
        final EFeedState retval;

        mActivationCount += actCount;
        if (mActivationCount < 0)
        {
            sLogger.warning(
                String.format("%s client %d, feed %d: negative activation count (%d), setting to zero (%s).",
                    mEClient.location(),
                    mEClient.clientId(),
                    mFeedId,
                    key(),
                    mActivationCount,
                    mScope));

            mActivationCount = 0;
        }

        // If the activation count is zero, that means this feed
        // is down due to no contra-feeds. If the count is > 0,
        // then the feed is up.
        retval = (mActivationCount == 0 ?
                  EFeedState.DOWN :
                  EFeedState.UP);

        // Did the feed state change?
        if (retval != mFeedState)
        {
            // Yes. Update the feed.
            update(retval);
        }

        if (sLogger.isLoggable(Level.FINEST))
        {
            sLogger.finest(String.format("%s client %d, feed %d: %s feed state=%s, activation count=%d (%s).",
                    mEClient.location(),
                    mEClient.clientId(),
                    mFeedId,
                    key(),
                    retval,
                    mActivationCount,
                    mScope));
        }

        return (retval);
    } // end of updateActivation(int)

    //
    // end of Set Methods.
    //-----------------------------------------------------------
} // end of class ENotifyFeed
