//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2015, 2016. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.NoSuchElementException;
import java.util.logging.Level;
import java.util.logging.Logger;
import net.sf.eBus.client.EClient.ClientLocation;
import net.sf.eBus.client.EFeed.FeedScope;

/**
 * Stores feed instances based on
 * {@link ClientLocation client location}: local client and
 * remote client. Feeds are placed into the array based on their
 * location. This allows messages destined for one or more
 * zones to be forward efficiently without having to test the
 * client location each time.
 * <p>
 * Tracks the supporting zone indicies as well. The idea here is
 * that a local client/local-only feed subscriber may received
 * notifications from all local clients, both local-only and
 * local &amp; remote feeds. So local client/local-only feed is
 * supported by all local clients. This relationship also exists
 * for a publisher: a local client/local-only feed publisher
 * should start its feed if there are any local subscribers.
 * The relationships for all feed locations are:
 * </p>
 * <ul>
 *   <li>
 *     local client/local-only feed is supported by all local
 *     clients, both local-only and local &amp; remote feeds.
 *   </li>
 *   <li>
 *     local client/local &amp; remote feed is supported by all
 *     clients, both local and remote.
 *   </li>
 *   <li>
 *     remote client/local-only feed is supported by local
 *     client/local &amp; remote feed.
 *   </li>
 * </ul>
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

/* package */ class EFeedList<T extends ESingleFeed>
{
//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Tracks the begin and end indices of the local or remote
     * feeds within {@link EFeedList#_feeds feed lists}. Also
     * tracks the number of opposing feeds which support this
     * feed list zone (if this is a publish feed list, then
     * the subscribe feeds are opposing). This object is used
     * to create a {@link EFeedListIterator} to iterate over the
     * zone feeds.
     *
     * @param <T> specifies either {@link EPublishFeed} or
     * {@link ESubscribeFeed}.
     *
     * @see EFeedListIterator
     * @see ClientLocation
     */
    private static final class ListZone<T extends ESingleFeed>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * The feeds stored in {@link #mFeeds} are located in
         * this client location.
         */
        private final ClientLocation mLocation;

        /**
         * The {@link EFeedList#_feeds} reference.
         */
        private final List<T> mFeeds;

        /**
         * This zone starts at this index in {@link #mFeeds}.
         */
        private int mBeginIndex;

        /**
         * This zone ends at this index in {@link #mFeeds}. Set
         * to zero when the zone is empty.
         */
        private int mEndIndex;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a zone within {@link #mFeeds} for the given
         * {@link ClientLocation client location}. The zone size
         * is initially empty.
         * @param location this zone covers this client location.
         * @param feeds reference to {@link EFeedList#_feeds}.
         */
        private ListZone(final ClientLocation location,
                         final List<T> feeds)
        {
            mLocation = location;
            mFeeds = feeds;

            mBeginIndex = 0;
            mEndIndex = 0;
        } // end of ListZone(ClientLocation, List)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the list client location.
         * @return client location.
         */
        public ClientLocation location()
        {
            return (mLocation);
        } // end of location()

        /**
         * Returns {@code true} if this list zone is empty.
         * @return {@code true} if this list zone is empty.
         */
        public boolean isEmpty()
        {
            return (mEndIndex == mBeginIndex);
        } // end of isEmpty()

        /**
         * Returns the list zone beginning index.
         * @return list zone beginning index.
         */
        public int beginIndex()
        {
            return (mBeginIndex);
        } // end of beginIndex()

        /**
         * Returns the list zone ending index. Returns zero if
         * the zone is empty.
         * @return list zone ending index.
         */
        public int endIndex()
        {
            return (mEndIndex);
        } // end of endIndex()

        /**
         * Returns the activation count summation for all feeds
         * in this zone.
         * @return value &ge; zero.
         */
        public int activationCount()
        {
            int i;
            int retval = 0;

            for (i = mBeginIndex; i < mEndIndex; ++i)
            {
                retval += (mFeeds.get(i)).activationCount();
            }

            return (retval);
        } // end of activationCount()

        //
        // end of Get Methods.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Inserts the feed at the current end index and then
         * increments the end index by one. As a side effect,
         * sets the feed's index to its insertion point. Returns
         * the list zone size as a result of this insert.
         * @param feed insert this feed at the list zone end.
         * @return the resulting list zone size.
         *
         * @see #remove(net.sf.eBus.client.EFeed)
         */
        private int add(final T feed)
        {
            mFeeds.add(mEndIndex, feed);
            feed.feedIndex(mEndIndex);

            ++mEndIndex;

            return (mEndIndex - mBeginIndex);
        } // end of add(T)

        /**
         * Removes the feed from its specified index. Decrements
         * the end index and the feed indicies to the right
         * <em>within this location.</em> Returns the list zone
         * size as a result of this remove.
         * @param feed remove this feed.
         * @return the resulting list zone size.
         *
         * @see #add(net.sf.eBus.client.EFeed)
         */
        private int remove(final T feed)
        {
            int index = feed.feedIndex();

            // Is there a feed to remove?
            if (index >= 0 && index < mFeeds.size())
            {
                mFeeds.remove(index);
                --mEndIndex;

                // Update the index for the feeds to the right.
                for (; index < mEndIndex; ++index)
                {
                    (mFeeds.get(index)).feedIndex(index);
                }
            }

            return (mEndIndex - mBeginIndex);
        } // end of remove(T)

        /**
         * Updates the beginning and ending indicies for this
         * zone by adding the given increment (either 1 or -1).
         * Then updates the feed index for all feeds within this
         * zone.
         * @param increment add this value to the beginning and
         * ending indicies.
         */
        private void shift(final int increment)
        {
            int index;

            mBeginIndex += increment;
            mEndIndex += increment;

            for (index = mBeginIndex; index < mEndIndex; ++index)
            {
                (mFeeds.get(index)).feedIndex(index);
            }

            return;
        } // end of shift(int)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class ListZone

    /**
     * Used to iterate over a zone within
     * {@link EFeedList#_feeds} using a {@link ListIterator}.
     *
     * @param <T> specifies either {@link EPublishFeed} or
     * {@link ESubscribeFeed}.
     */
    public static final class EFeedListIterator<T extends ESingleFeed>
        implements Iterator<T>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * List iterator for underlying feeds list.
         */
        private final ListIterator<T> mIterator;

        /**
         * Initialized to the number of elements covered by the
         * iterator. Decremented each time {@code next()} is
         * called. When value reaches zero, the iterator is
         * done. This value is needed since the iterator may
         * cover fewer elements than in the list.
         */
        private int mRemaining;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private EFeedListIterator(final ListIterator<T> lit,
                                  final int feedCount)
        {
            mIterator = lit;
            mRemaining = feedCount;
        } // end of EFeedListIterator(ListIterator, int)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Iterator Interface Implementation.
        //

        @Override
        public boolean hasNext()
        {
            return (mRemaining > 0);
        } // end of hasNext()

        @Override
        public T next()
            throws NoSuchElementException
        {
            if (mRemaining == 0)
            {
                throw (new NoSuchElementException());
            }

            --mRemaining;

            return (mIterator.next());
        } // end of next()

        //
        // end of Iterator Interface Implementation.
        //-------------------------------------------------------
    } // end of class EFeeListIterator

//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * The number of eBus feed zones.
     */
    private static final int ZONE_COUNT =
        (ClientLocation.values()).length;

    /**
     * Set the feed array list initial size to 64 feeds.
     */
    private static final int FEED_LIST_INITIAL_CAPACITY = 64;

    //-----------------------------------------------------------
    // Statics.
    //

    /**
     * The subject class logger.
     */
    private static final Logger _logger =
        Logger.getLogger(EFeedList.class.getName());

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The registered feeds stored into
     * {@link ClientLocation locations}: local or remote.
     * This division allows for quick iteration over those
     * feeds which need to be updated by a changed in feed
     * state. Note that zone iteration is in sequence. There is
     * no need to skip over elements. That is why this design
     * works.
     */
    private final List<T> _feeds;

    /**
     * {@link #_feeds} is divided into separate zones.
     */
    private final ListZone<T>[] _zones;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new, empty feed list for both local and remote
     * clients.
     */
    @SuppressWarnings ({"unchecked", "rawtypes"})
    /* package */ EFeedList()
    {
        _feeds = new ArrayList<>(FEED_LIST_INITIAL_CAPACITY);
        _zones = new ListZone[ZONE_COUNT];

        final ClientLocation[] zones = ClientLocation.values();
        int index;

        for (index = 0; index < ZONE_COUNT; ++index)
        {
            _zones[index] = new ListZone<>(zones[index], _feeds);
        }
    } // end of EFeedList()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns the feed instance at the specified index.
     * @param index feed index.
     * @return the referenced feed.
     */
    /* package */ T feed(final int index)
    {
        return (_feeds.get(index));
    } // end of index;

    /**
     * Returns the current activation count for the given client
     * location.
     * @param loc client location (local or remote).
     * @return feed location activation count.
     */
    /* package */ int activationCount(final ClientLocation loc)
    {
        return (_zones[loc.ordinal()].activationCount());
    } // end of activationCount(ClientLocation)

    /**
     * Returns the number feeds supporting the given scope.
     * @param scope the supported scope.
     * @return integer &ge; zero.
     */
    /* package */ int supports(final FeedScope scope)
    {
        return (
            (int) _feeds.stream()
                        .filter(
                            (feed) ->
                                (feed.scope() ==
                                     FeedScope.LOCAL_AND_REMOTE ||
                                 feed.scope() ==
                                     FeedScope.REMOTE_ONLY))
                        .count());
    } // end of supports(FeedScope)

    /**
     * Returns the number of feeds supporting the given client
     * location. Note that this support is <em>not</em>
     * dependent on the feed being {@link EFeedState#UP}, only
     * that the feed is advertised.
     * @param location find support for this client location.
     * @return number of advertised feeds supporting
     * {@code location}.
     */
    /* package */ int supports(final ClientLocation location)
    {
        return (
            (int) _feeds.stream()
                        .filter(
                            (feed) ->
                                (feed.scope()).supports(
                                    location))
                        .count());
    } // end of supports(ClientLocation)

    /**
     * Returns the number of feeds supported by the given scope.
     * @param scope supporting scope.
     * @return integer &ge; zero.
     */
    /* package */ int isSupportedBy(final FeedScope scope)
    {
        final List<ClientLocation> locations = scope.locations();
        final int beginIndex =
            _zones[(locations.get(0)).ordinal()].beginIndex();
        final int endIndex =
            _zones[(locations.get(locations.size() - 1)).ordinal()].endIndex();

        return (endIndex - beginIndex);
    } // end of isSupportedBy(FeedScope)

    /**
     * Returns the current feed state for the given client
     * location.
     * @param loc client location
     * @return {@code UP} if there is at least one feed
     * supporting the given location.
     */
    /* package */ EFeedState feedState(final ClientLocation loc)
    {
        final Iterator<T> fit = _feeds.iterator();
        T feed;
        EFeedState retval = EFeedState.DOWN;

        // Continue until either all feeds are checked or until
        // an UP feed supporting this client is found.
        while (fit.hasNext() && retval == EFeedState.DOWN)
        {
            feed = fit.next();

            if ((feed.scope()).supports(loc))
            {
                retval = feed.feedState();
            }
        }

        return (retval);
    } // end of feedState(ClientLocation)

    /**
     * Returns a iterator for traversing the feed list based on
     * the feed scope.
     * @param scope feed scope
     * @return a feed list iterator over the feeds list.
     */
    /* package */ Iterator<T> iterator(final FeedScope scope)
    {
        final List<ClientLocation> locations = scope.locations();
        final int beginIndex =
            _zones[(locations.get(0)).ordinal()].beginIndex();
        final int endIndex =
            _zones[(locations.get(locations.size() - 1)).ordinal()].endIndex();
        final ListIterator<T> lit =
            _feeds.listIterator(beginIndex);
        final int size = (endIndex - beginIndex);

        return (new EFeedListIterator<>(lit, size));
    } // end of iterator(ClientLocation)

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Set Methods.
    //

    /**
     * Inserts the feed instance into this location feed list.
     * Returns the resulting feed list size.
     * @param feed insert this feed into the list.
     * @return feed list size resulting from the add.
     *
     * @see #remove(net.sf.eBus.client.EFeed)
     */
    /* package */ int add(final T feed)
    {
        int zoneIndex = (feed.location()).ordinal();
        final int retval = _zones[zoneIndex].add(feed);

        // Shift the following zones to the right.
        for (++zoneIndex; zoneIndex < ZONE_COUNT; ++zoneIndex)
        {
            _zones[zoneIndex].shift(1);
        }

        return (retval);
    } // end of add(T)

    /**
     * Removes the feed from this list. Returns the resulting
     * feed list size.
     * feeds and the supporting feed indicies. Returns the number
     * of feeds in the feed location as a result.
     * @param feed remove this feed.
     * @return feed list size resulting from the remove.
     *
     * @see #add(net.sf.eBus.client.EFeed)
     */
    /* package */ int remove(final T feed)
    {
        int zoneIndex = (feed.location()).ordinal();
        final int retval = _zones[zoneIndex].remove(feed);

        // Shift the following zones to the left.
        for (++zoneIndex; zoneIndex < ZONE_COUNT; ++zoneIndex)
        {
            _zones[zoneIndex].shift(-1);
        }

        return (retval);
    } // end of remove(T)

    /**
     * Either increments or decrements the activation count based
     * on whether {@code feedState} is up or down, respectively.
     * If the incremented count is one, then the feeds are
     * informed that the feed state is up. If the decremented
     * count is zero, then the feeds are informed that the feed
     * state is down.
     * @param feed the feed with the updated feed state.
     * @param fs the updated feed state.
     * @return value &gt; zero if feeds are activated, &lt; zero
     * if feeds are deactivated, and zero if no feeds impacted.
     */
    /* package */ int updateCount(final ESingleFeed feed,
                                  final EFeedState fs)
    {
        final Iterator<T> fit = this.iterator(feed.scope());
        final ClientLocation location = feed.location();
        int retval = 0;

        // Find those feeds which support the contra-feed's
        // location and update the contra-*zone* active count.
        while (fit.hasNext())
        {
            retval +=
                (fit.next()).updateActivation(location, fs);
        }

        if (_logger.isLoggable(Level.FINEST))
        {
            _logger.finest(
                String.format(
                    "FeedList.updateCount([id=%d, key=%s, location=%s, scope=%s], %s) -> %d",
                    feed.feedId(),
                    feed.key(),
                    location,
                    feed.scope(),
                    fs,
                    retval));
        }

        return (retval);
    } // end of updateCount(ENotifyFeed)

    //
    // end of Set Methods.
    //-----------------------------------------------------------
} // end of class EFeedList
