//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2014, 2016. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client;

import java.io.Serializable;
import java.net.InetSocketAddress;
import java.util.Formatter;
import java.util.List;
import java.util.Objects;
import net.sf.eBus.messages.EMessageKey;

/**
 * This message reports when a
 * {@link #remoteAddress remote connection} has either
 * {@link #state logged on or logged off}. If the remote
 * connection logged off, then <em>may</em> provide the
 * {@link #reason reason} why remote eBus logged off
 * ({@code reason} field may be {@code null}).
 * <p>
 * All connection messages have the subject
 * {@link AbstractEBusMessage#EBUS_SUBJECT "/eBus"}. Subscribe to
 * {@link #MESSAGE_KEY ConnectionMessage.class/eBus} to receive
 * this notification.
 * </p>
 * <p>
 * <strong>Note:</strong> this notification is published
 * locally only and is not sent to remote eBus applications.
 * </p>
 *
 * @see ServerMessage
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class ConnectionMessage
    extends AbstractEBusMessage
    implements Serializable
{

//---------------------------------------------------------------
// Enums.
//

    /**
     * A remote eBus is either logged on or logged off.
     */
    public enum ConnectionState
    {
        /**
         * A remote eBus connection is logged on.
         */
        LOGGED_ON,

        /**
         * A remote eBus connection is logged off.
         */
        LOGGED_OFF,

        /**
         * Remote eBus connection is paused.
         */
        PAUSED,

        /**
         * Remote eBus connection is resumed.
         */
        RESUMED
    } // end of enum ConnectionState

//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * The connection message key
     * {@code ConnectionMessage.class:/eBus}. Use this key when
     * {@link ESubscribeFeed#open(ESubscriber, EMessageKey, EFeed.FeedScope, ECondition)}
     * for this message.
     */
    public static final EMessageKey MESSAGE_KEY =
        new EMessageKey(ConnectionMessage.class, EBUS_SUBJECT);

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The remote eBus is either logged on or logged off.
     */
    public final ConnectionState state;

    /**
     * If {@link #state} is {@link ConnectionState#LOGGED_OFF}
     * and the log off was due to an exception, then the
     * {@link Throwable#getMessage() exception message} is stored
     * here. May be {@code null}.
     */
    public final String reason;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new connection message for the given remote eBus
     * address and connection state. {@link #reason} is set to
     * {@code null}.
     * @param addr remote eBus address.
     * @param serverPort connection is associated with this
     * {@link EServer} port; zero if not an accepted connection.
     * @param state connection state.
     * @throws NullPointerException
     * if either {@code addr} or {@code state} is {@code null}.
     *
     * @deprecated use {@link Builder} to create connection
     * message.
     */
    @Deprecated
    public ConnectionMessage(final InetSocketAddress addr,
                             final int serverPort,
                             final ConnectionState state)
    {
        super (addr, serverPort);

        Objects.requireNonNull(state, "null state");

        this.state = state;
        this.reason = null;
    } // end of ConnectionMessage(...)

    /**
     * Creates a new connection message for the given remote eBus
     * address, connection state, and reason.
     * @param addr remote eBus address.
     * @param serverPort connection is associated with this
     * {@link EServer} port; zero if not an accepted connection.
     * @param state connection state.
     * @param reason reason for a logged off state. May be
     * {@code null} or empty.
     * @throws NullPointerException
     * if either {@code addr} or {@code state} is {@code null}.
     *
     * @deprecated use {@link Builder} to create connection
     * message.
     */
    @Deprecated
    public ConnectionMessage(final InetSocketAddress addr,
                             final int serverPort,
                             final ConnectionState state,
                             final String reason)
    {
        super (addr, serverPort);

        Objects.requireNonNull(state, "null state");

        this.state = state;
        this.reason = reason;
    } // end of ConnectionMessage(...)

    /**
     * Creates a remote eBus connection update with the given
     * parameters.
     * @param subject the notification subject. Will always be
     * {@link AbstractEBusMessage#EBUS_SUBJECT "/eBus"}.
     * @param timestamp notification timestamp.
     * @param addr remote eBus address.
     * @param serverPort connection is associated with this
     * {@link EServer} port; zero if not an accepted connection.
     * @param state connection state.
     * @param reason the reason for a
     * {@link ConnectionState#LOGGED_OFF} connection state.
     * @throws NullPointerException
     * if either {@code addr} or {@code state} is {@code null}.
     *
     * @deprecated use {@link Builder} to create connection
     * message.
     */
    @Deprecated
    public ConnectionMessage(final String subject,
                             final long timestamp,
                             final InetSocketAddress addr,
                             final int serverPort,
                             final ConnectionState state,
                             final String reason)
    {
        super (subject, timestamp, addr, serverPort);

        Objects.requireNonNull(state, "null state");

        this.state = state;
        this.reason = reason;
    } // end of ConnectionMessage(...)

    /**
     * Creates a new connection message based on the builder
     * settings.
     * @param builder connection message builder.
     */
    private ConnectionMessage(final Builder builder)
    {
        super (builder);

        this.state = builder.mConnectionState;
        this.reason = builder.mReason;
    } // end of ConnectionMessage(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns the connection message as text.
     * @return textual representation of the connection message.
     */
    @Override
    public String toString()
    {
        final Formatter retval = new Formatter();

        retval.format("%s%n", super.toString());
        retval.format("       connection state: %s%n", state);
        retval.format(
            "                 reason: %s",
            (reason == null || reason.isEmpty() ?
             "(not set)" :
             reason));

        return (retval.toString());
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    /**
     * Returns a connection message builder instance.
     * @return message builder instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Use this builder to create {@code ConnectionMessage}
     * instances.
     */
    public static final class Builder
        extends AbstractEBusMessage.Builder<ConnectionMessage, Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private ConnectionState mConnectionState;
        private String mReason;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private Builder()
        {
            super (ConnectionMessage.class);
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        @Override
        protected ConnectionMessage buildImpl()
        {
            return (new ConnectionMessage(this));
        } // end of buildImpl()

        @Override
        protected void validate(final List<String> problems)
        {
            super.validate(problems);

            if (mConnectionState == null)
            {
                problems.add("connection state not state");
            }

            return;
        } // end of validate(List<>)

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        public Builder state(final ConnectionState state)
        {
            if (state == null)
            {
                throw (new IllegalArgumentException());
            }

            mConnectionState = state;

            return (this);
        } // end of state(ConnectionState)

        public Builder reason(final String reason)
        {
            mReason = reason;

            return (this);
        } // end of reason(String)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class ConnectionMessage
