//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2014 - 2016. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.client;

import java.io.Serializable;
import java.net.InetSocketAddress;
import java.util.List;
import java.util.Objects;
import net.sf.eBus.messages.EMessageObject;
import net.sf.eBus.messages.ENotificationMessage;

/**
 * Base class for eBus {@link ConnectionMessage} and
 * {@link ServerMessage} classes. Defines the common notification
 * subject used for all eBus client messages.
 * <p>
 Contains the remote TCP address and service port used by
 subclasses.
 *
 * @see ConnectionMessage
 * @see ServerMessage
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public abstract class AbstractEBusMessage
    extends ENotificationMessage
    implements Serializable
{

//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * "/eBus" is the subject used for all eBus
     * {@code ConnectionMessage} and {@code ServerMessage}
     * instances.
     * <p>
     * When subscribing for {@link ConnectionMessage} and
     * {@link ServerMessage}, always use the subject "/eBus".
     */
    public static final String EBUS_SUBJECT = "/eBus";

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The client address referenced in this message.
     */
    public final InetSocketAddress remoteAddress;

    /**
     * A TCP service port number.
     */
    public final int serverPort;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new eBus message instance with the subject set
     * to {@link #EBUS_SUBJECT} and the timestamp set to the
     * current time.
     * @param addr the remote eBus address.
     * @param serverPort connection is associated with this
     * {@link EServer} port; zero if not an accepted connection.
     * @throws NullPointerException
     * if {@code addr} is {@code null}.
     */
    protected AbstractEBusMessage(final InetSocketAddress addr,
                                  final int serverPort)
    {
        this (EBUS_SUBJECT,
              System.currentTimeMillis(),
              addr,
              serverPort);
    } // end of AbstractEBusMessage(InetSocketAddress, int)

    /**
     * Creates a new eBus message instance with the subject set
     * to {@link #EBUS_SUBJECT} and the timestamp set to the
     * given time.
     * @param timestamp the notification timestamp.
     * @param addr the remote eBus address.
     * @param serverPort connection is associated with this
     * {@link EServer} port; zero if not an accepted connection.
     * @throws NullPointerException
     * if {@code addr} is {@code null}.
     */
    protected AbstractEBusMessage(final long timestamp,
                                  final InetSocketAddress addr,
                                  final int serverPort)
    {
        this (EBUS_SUBJECT, timestamp, addr, serverPort);
    } // end of AbstractEBusMessage(long, InetSocketAddress, int)

    /**
     * Creates a new eBus client message instance for the given
     * message subject and timestamp.
     * @param subject the notification subject. Will always be
     * {@link #EBUS_SUBJECT "/eBus"}.
     * @param timestamp the notification timestamp.
     * @param addr the remote eBus address.
     * @param serverPort connection is associated with this
     * {@link EServer} port; zero if not an accepted connection.
     * @throws NullPointerException
     * if {@code addr} is {@code null}.
     */
    protected AbstractEBusMessage(final String subject,
                                  final long timestamp,
                                  final InetSocketAddress addr,
                                  final int serverPort)
    {
        super (subject, timestamp);

        Objects.requireNonNull(addr, "null addr");

        this.remoteAddress = addr;
        this.serverPort = serverPort;
    } // end of AbstractEBusMessage(...)

    protected AbstractEBusMessage(final Builder builder)
    {
        super (builder);

        this.remoteAddress = builder.mRemoteAddress;
        this.serverPort = builder.mServerPort;
    } // end of AbstractEBusMessage(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns the message as text.
     * @return textual representation of the message.
     */
    @Override
    public String toString()
    {
        return (
            String.format(
                "%s%n                address: %s%n                   port: %d%n             server port: %d",
                super.toString(),
                remoteAddress.getHostName(),
                remoteAddress.getPort(),
                serverPort));
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Base class builder for {@code AbstractEBusMessage}
     * builders.
     *
     * @param <M> target message class.
     * @param <B> Builder subclass.
     */
    public static abstract class Builder<M extends AbstractEBusMessage,
                                         B extends Builder<M, ?>>
        extends ENotificationMessage.Builder<M, B>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private InetSocketAddress mRemoteAddress;
        private int mServerPort;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        protected Builder(final Class<? extends EMessageObject> targetClass)
        {
            super (targetClass, EBUS_SUBJECT);

            mServerPort = -1;
        } // end of Builder(Class)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        @Override
        protected void validate(final List<String> problems)
        {
            super.validate(problems);

            if (mRemoteAddress == null)
            {
                problems.add("remote address not set");
            }

            if (mServerPort < 0)
            {
                problems.add("server port not set");
            }

            return;
        } // end of validate(List<>)

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        @SuppressWarnings ("unchecked")
        public final B remoteAddress(final InetSocketAddress address)
        {
            mRemoteAddress = address;

            return ((B) this);
        } // end of remoteAddress(InetSocketAddress)

        @SuppressWarnings ("unchecked")
        public final B serverPort(final int serverPort)
        {
            mServerPort = serverPort;

            return ((B) this);
        } // end of serverPort(int)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class AbstractEBusMessage
