//
// Copyright 2021 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.config;

import com.google.common.collect.ImmutableList;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigException;
import java.util.List;
import java.util.Objects;
import net.openhft.affinity.AffinityStrategies;

/**
 * This immutable class contains the necessary information needed
 * to create an affinity between a thread and a core using
 * <a href="https://github.com/OpenHFT/Java-Thread-Affinity" target="_blank">OpenHFT Thread Affinity Library</a>.
 * User is assumed to understand thread affinity and the
 * necessary operating system configuration needed to support it.
 * Please see the above link for more information on how to
 * use thread affinity and its correct use.
 * <p>
 * ThreadAffinityConfigure consist of the following properties:
 * </p>
 * <ul>
 *   <li>
 *     {@code affinityType}: Required. Defines how core is
 *     acquired for the thread. There are five acquisition types
 *     as defined by {@link AffinityType}:
 *     <ol>
 *       <li>
 *         {@link AffinityType#ANY_CORE ANY_CORE}: Use
 *         {@code AffinityLock.acquireCore()} to assign any free
 *         core to thread.
 *       </li>
 *       <li>
 *         {@link AffinityType#ANY_CPU ANY_CPU}: Use
 *         {@code AffinityLock.acquireLock()} to assign any free
 *         CPU to thread.
 *       </li>
 *       <li>
 *         {@link AffinityType#CPU_LAST_MINUS CPU_LAST_MINUS}:
 *         Use {@code AffinityLock.acquireLock(int cpuId} to
 *         assign a CPU with specified identifier to thread.
 *         Requires property {@link #CPU_OFFSET_KEY} be set.
 *       </li>
 *       <li>
 *         {@link AffinityType#CPU_ID CPU_ID}: Use
 *         {@code AffinityLock.acquireLock(int cpuId} to
 *         assign a CPU with specified identifier to thread.
 *         Requires property {@link #CPU_ID_KEY} be set.
 *       </li>
 *       <li>
 *         {@link AffinityType#CPU_STRATEGIES CPU_STRATEGIES}:
 *         Use {@code AffinityLock.acquireLock(AffinityStrategies...)}
 *         to assign a CPU for thread affinity. Selects a CPU for
 *         thread affinity based on the given selection
 *         strategies. Requires property {@link #STRATEGIES_KEY}
 *         be set.
 *         <p>
 *         Please note that this type may not be used by itself
 *         or as an initial CPU acquisition type. Rather there
 *         must be previous CPU allocation to this (for example a
 *         previous dispatcher configuration using thread
 *         affinity) which the strategy then uses to allocate the
 *         next CPU. Attempts to use this acquisition type either
 *         by itself or as the first strategy will result in an
 *         error and no CPU allocated for the thread.
 *         </p>
 *       </li>
 *     </ol>
 *   </li>
 *   <li>
 *     {@code bind}: Optional, default value is {@code false}.
 *     If {@code true}, then bind current thread to allocated
 *     {@code AffinityLock}.
 *   </li>
 *   <li>
 *     {@code wholeCore}: Optional, default value is
 *     {@code false}. If {@code true}, then bind current thread
 *     to allocated {@code AffinityLock} reserving the whole
 *     core. This property is used only when {@code bind}
 *     property is {@code true}.
 *   </li>
 *   <li>
 *     {@code cpuId}: Required when {@code affinityType} is set
 *     to {@code CPU_ID}. Specifies the allocated CPU by its
 *     identifier.
 *   </li>
 *   <li>
 *     {@code cpuStrategies}: Required when {@code affinityType}
 *     is set to {@code CPU_STRATEGIES}. Values are restricted to
 *     enum {@code net.openhft.affinity.AffinityStrategies}.
 *     <p>
 *     <strong>Note:</strong> strategy ordering is important.
 *     {@code AffinityStrategies.ANY} must appear as the last
 *     listed strategy. This allows any CPU to be selected in
 *     case none of the other strategies found an acceptable CPU.
 *     </p>
 *   </li>
 * </ul>
 * <p>
 * Users should be familiar with the OpenHFT Java Thread Affinity
 * library and how it works before using eBus thread affinity
 * configuration. This includes configuring the operating system
 * to isolate acquired CPUs from the operating system. This
 * prevents the OS from pre-empting the thread from its assigned
 * CPU which means the thread does not entirely own the CPU. That
 * said, isolating too many CPUs from the OS can lead to a kernel
 * panic. So using thread affinity is definitely an advanced
 * software technique, requiring good understanding of how an OS
 * functions.
 * <p>
 * The following example shows how to use thread affinity for
 * eBus dispatcher threads and especially the CPU_STRATEGIES
 * acquisition type.
 * </p>
 * <pre><code>"dispatchers" : [
    {
        "name" : "mdDispatcher"
        "numberThreads" : 1
        "runQueueType" : "spinning"
        "priority" : 9
        "quantum" : 10000
        "isDefault" : false
        "classes" : [ "com.acme.trading.MDHandler" ]
        "threadAffinity" {       // optional, selector thread core affinity
            affinityType : CPU_ID // required, core selection type.
            cpuId : 7             // required for CPU_ID affinity type
            bind : true           // optional, defaults to false
            wholeCore : true      // optional, defaults to false
        }
    },
    {
        "name" : "orderDispatcher"
        "numberThreads" : 1
        "runQueueType" : "spinning"
        "priority" : 9
        "quantum" : 10000
        "isDefault" : false
        "classes" : [ "com.acme.trading.OrderHandler" ]
        "threadAffinity" {       // optional, selector thread core affinity
            affinityType : CPU_STRATEGIES // required, core selection type.
            cpuStrategies : [             // required for CPU_STRATEGIES affinity type
                	SAME_CORE, SAME_SOCKET, ANY // Note: ANY must be last strategy.
            ]
            bind : true           // optional, defaults to false
            wholeCore : true      // optional, defaults to false
        }
    },
    {
        "name" : "defaultDispatcher"
        "numberThreads" : 8
        "runQueueType" : "blocking"
        "priority" : 4
        "quantum" : 100000
        "isDefault" : true
    }
]</code></pre>
 * </p>
 * <p>
 * eBus uses this configuration to optionally pin
 * {@code net.sf.eBus.net.SelectorThread} and
 * {@code net.sf.eBus.client.EClient.RQThread} instances to a
 * core or cores.
 * </p>
 *
 * @see ThreadAffinityConfigure.Builder
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public class ThreadAffinityConfigure
{
//---------------------------------------------------------------
// Member enums.
//

    /**
     * These affinity types map to a specific
     * {@code AffinityLock static} method used to acquire an
     * affinity lock.
     */
    public enum AffinityType
    {
        /**
         * Use {@code AffinityLock.acquireCore()} to assign any
         * free core to thread.
         */
        ANY_CORE,

        /**
         * Use {@code AffinityLock.acquireLock()} to assign any
         * free CPU to thread.
         */
        ANY_CPU,

        /**
         * Use {@code AffinityLock.acquireLockLastMinus(int n}
         * to allocate a CPU from the end of the core set based
         * on the given positive number. Requires property
         * {@link #CPU_OFFSET_KEY CPU index} is set.
         */
        CPU_LAST_MINUS,

        /**
         * Use {@code AffinityLock.acquireLock(int cpuId} to
         * assign a CPU with specified identifier to thread.
         * Requires CPU identifier is set.
         */
        CPU_ID,

        /**
         * Use {@code AffinityLock.acquireLock(AffinityStrategies...)}
         * to assign a CPU for thread affinity. Selects a CPU for
         * thread affinity based on the given selection
         * strategies. Requires selection
         * {@link AffinityStrategies strategy types} are set.
         * Strategy order is significant and
         * {@link AffinityStrategies#ANY}
         * <strong><em>must</em></strong> be the final strategy
         * type.
         */
        CPU_STRATEGIES
    } // end of enum AffinityType

//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    //
    // Property keys.
    //

    /**
     * Key {@value} contains an {@link AffinityType} value.
     */
    public static final String AFFINITY_TYPE_KEY =
        "affinityType";

    /**
     * Key {@value} contains the bind-thread-to-affinity lock
     * flag.
     */
    public static final String BIND_KEY = "bind";

    /**
     * Key {@value} contains reserve-whole-core flag. This
     * property is used only if {@link #BIND_KEY} is set to
     * {@code true}.
     */
    public static final String WHOLE_CORE_KEY = "wholeCore";

    /**
     * Key {@value} contains CPU offset used when
     * {@link #AFFINITY_TYPE_KEY} is set to
     * {@link AffinityType#CPU_LAST_MINUS}. Otherwise this
     * property is ignored for any other affinity type.
     */
    public static final String CPU_OFFSET_KEY = "lastMinusOffset";

    /**
     * Key {@value} contains CPU identifier used when
     * {@link #AFFINITY_TYPE_KEY} is set to
     * {@link AffinityType#CPU_ID}. Otherwise this property is
     * ignored for any other affinity type. Property value must
     * be &ge; zero.
     */
    public static final String CPU_ID_KEY = "cpuId";

    /**
     * Key {@value} contains CPU selection strategies array
     * used when {@link #AFFINITY_TYPE_KEY} is set to
     * {@link AffinityType#CPU_STRATEGIES}. Otherwise this
     * property is ignored for any other affinity type.
     * Property value must be a non-empty array where the final
     * element is {@link AffinityStrategies#ANY}. Note that
     * strategy ordering is significant.
     */
    public static final String STRATEGIES_KEY = "cpuStrategies";

    //-----------------------------------------------------------
    // Statics.
    //

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Defines how a CPU is chosen for thread affinity. Default
     * setting is {@link AffinityType#ANY_CPU}.
     */
    private final AffinityType mAffinityType;

    /**
     * Set to {@code true} when thread is bound to its
     * {@code AffinityLock}. Default setting is {@code false} -
     * no binding is done.
     *
     * @see #mWholeCore
     */
    private final boolean mBindFlag;

    /**
     * Set to {@code true} when affinity reserves the whole
     * core. This flag is used only when {@link #mBindFlag} is
     * set to {@code true}.
     *
     * @see #mBindFlag
     */
    private final boolean mWholeCore;

    /**
     * When affinity type is {@link AffinityType#CPU_ID} this is the
     * CPU identifier used for thread affinity. Default setting
     * is negative meaning no CPU selected.
     */
    private final int mCpuId;

    /**
     * When affinity type is {@link AffinityType#CPU_LAST_MINUS}
     * this is the offset used to select a CPU from the end of
     * the CPU set. Default setting is zero meaning no offset is
     * selected.
     */
    private final int mOffset;

    /**
     * When affinity type is {@link AffinityType#CPU_STRATEGIES}
     * these are the pre-defined strategies used to select
     * CPU which with the thread will have affinity. Default
     * setting is {@code null} if no strategies are defined.
     */
    private final List<AffinityStrategies> mStrategies;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new thread affinity configuration based on the
     * given builder's settings.
     * @param builder contains thread affinity settings.
     */
    private ThreadAffinityConfigure(final Builder builder)
    {
        mAffinityType = builder.mAffinityType;
        mBindFlag = builder.mBindFlag;
        mWholeCore = builder.mWholeCore;
        mCpuId = builder.mCpuId;
        mOffset = builder.mOffset;
        mStrategies = builder.mStrategies;
    } // end of ThreadAffinityConfigure(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (this == o);

        if (!retcode && o instanceof ThreadAffinityConfigure)
        {
            final ThreadAffinityConfigure tac =
                (ThreadAffinityConfigure) o;

            retcode =
                (mAffinityType == tac.mAffinityType &&
                 mBindFlag == tac.mBindFlag &&
                 mWholeCore == tac.mWholeCore &&
                 mCpuId == tac.mCpuId &&
                 mOffset == tac.mOffset &&
                 Objects.equals(
                     this.mStrategies, tac.mStrategies));
        }

        return (retcode);
    } // end of equals(Object)

    @Override
    public int hashCode()
    {
        return (Objects.hash(mAffinityType,
                             mBindFlag,
                             mWholeCore,
                             mCpuId,
                             mOffset,
                             mStrategies));
    } // end of hashCode()

    @Override
    public String toString()
    {
        final StringBuilder retval = new StringBuilder();

        retval.append("[type=").append(mAffinityType)
              .append(", bind=").append(mBindFlag);

        if (mBindFlag)
        {
            retval.append(", core=").append(mWholeCore);
        }

        switch (mAffinityType)
        {
            case CPU_ID:
                retval.append(", cpu ID=").append(mCpuId);
                break;

            case CPU_LAST_MINUS:
                retval.append(", offset=").append(mOffset);
                break;

            case CPU_STRATEGIES:
                String sep = "";

                retval.append(", strategies={");
                for (AffinityStrategies s : mStrategies)
                {
                    retval.append(sep).append(s);
                    sep = ",";
                }
                retval.append('}');
                break;

            default:
                // No other data to append.
        }

        retval.append(']');

        return (retval.toString());
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns affinity type used for creating thread affinity to
    selected CPU_ID.
     * @return CPU_ID selection type.
     */
    public AffinityType affinityType()
    {
        return (mAffinityType);
    } // end of affinityType()


    /**
     * Returns the bind-thread-to-affinity lock setting. If
     * {@code true} then thread is bound to the lock.
     * @return {@code true} if thread is bound to the affinity
     * lock.
     *
     * @see #wholeCoreFlag()
     */
    public boolean bindFlag()
    {
        return (mBindFlag);
    } // end of bindFlag()

    /**
     * Returns whole core reservation bind settings. If
     * {@code true} then thread reserves entire core and does not
     * allow hyper-threading. This value is used only if
     * {@link #bindFlag()} returns {@code true}; otherwise it is
     * ignored.
     * @return whole core reservation flag.
     *
     * @see #bindFlag()
     */
    public boolean wholeCoreFlag()
    {
        return (mWholeCore);
    } // end of wholeCoreFlag()

    /**
     * Returns CPU_ID identifier used for {@link AffinityType#CPU_ID}
     * affinity type. Set to a negative number for any other
     * affinity type.
     * @return CPU_ID identifier or &lt; zero if no identifier
     * specified.
     */
    public int cpuId()
    {
        return (mCpuId);
    } // end of cpuId()

    /**
     * Returns CPU offset used for
     * {@link AffinityType#CPU_LAST_MINUS} affinity type. Set to
     * zero for any other affinity type.
     * @return CPU offset or zero if no offset specified.
     */
    public int cpuOffset()
    {
        return (mOffset);
    } // end of cpuOffset()

    /**
     * Returns immutable list of CPU_ID selection strategies used
 for {@link AffinityType#CPU_STRATEGIES} affinity type. Set to
     * {@code null} for any other affinity type.
     * @return CPU_ID selection strategies.
     */
    public List<AffinityStrategies> strategies()
    {
        return (mStrategies);
    } // end of strategies()

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    /**
     * Returns a thread affinity configuration loaded from a
     * given JSON configuration under property key.
     * @param key affinity key is stored under this property.
     * @param config JSON configuration.
     * @return thread affinity configuration.
     * @throws ConfigException
     * if {@code config} is missing a required property or a
     * property value is invalid.
     */
    public static ThreadAffinityConfigure loadAffinity(final String key,
                                                       final Config config)
    {
        ThreadAffinityConfigure retval = null;

        if (config.hasPath(key))
        {
            retval = loadAffinityImpl(config.getConfig(key));
        }

        return (retval);
    } // end of loadAffinity(String, Config)

    /**
     * Returns list of thread affinity configurations loaded from
     * given JSON configuration under property key.
     * @param key affinity keys stored under this property.
     * @param config JSON configuration.
     * @return list of thread affinity configurations.
     * @throws ConfigException
     * if {@code config} is missing a required property or a
     * property value is invalid.
     */
    public static List<ThreadAffinityConfigure> loadAffinities(final String key,
                                                               final Config config)
    {
        final ImmutableList.Builder<ThreadAffinityConfigure> retval =
            ImmutableList.builder();

        if (config.hasPath(key))
        {
            config.getConfigList(key)
                  .forEach(c -> retval.add(loadAffinityImpl(c)));
        }

        return (retval.build());
    } // end of loadAffinities(String, Config)

    /**
     * Returns new {@code ThreadAffinityConfigure} builder
     * instance. {@link Builder} is the only way to create a
     * thread affinity configuration. It is recommended that a
     * new builder is used to create each new thread affinity
     * configuration.
     * @return thread affinity configuration builder.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

    /**
     * Returns thread affinity configuration extracted from
     * given typesafe configuration.
     * @param config extract affinity from this configuration.
     * @return thread affinity configuration.
     */
    private static ThreadAffinityConfigure loadAffinityImpl(final Config config)
    {
        final AffinityType type =
            config.getEnum(
                AffinityType.class, AFFINITY_TYPE_KEY);
        final Builder builder = new Builder();

        builder.affinityType(type);

        if (config.hasPath(BIND_KEY))
        {
            builder.bind(config.getBoolean(BIND_KEY));

            if (config.hasPath(WHOLE_CORE_KEY))
            {
                builder.wholeCore(
                    config.getBoolean(WHOLE_CORE_KEY));
            }
        }

        switch (type)
        {
            case CPU_ID:
                setCpuId(builder, config);
                break;

            case CPU_LAST_MINUS:
                setOffset(builder, config);
                break;

            case CPU_STRATEGIES:
                setStrategies(builder, config);
                break;

            default:
                // No required properties for other affinity
                // types.
        }

        return (builder.build());
    } // end of loadAffinityImpl(Config)

    /**
     * Retrieves CPU identifier from JSON configuration and
     * places into builder. Converts builder exception into
     * {@code ConfigException.BadValue}.
     * @param builder place CPU identifier into this builder.
     * @param config retrieve CPU identifier from this JSON
     * configuration.
     * @throws ConfigException
     * if CPU identifier is invalid.
     */
    private static void setCpuId(final Builder builder,
                                 final Config config)
    {
        try
        {
            builder.cpuId(config.getInt(CPU_ID_KEY));
        }
        catch (Exception jex)
        {
            throw (
                new ConfigException.BadValue(
                    CPU_ID_KEY, jex.getMessage()));
        }
    } // end of setCpuId(Builder, Config)

    /**
     *
     * Retrieves CPU offset from JSON configuration and
     * places into builder. Converts builder exception into
     * {@code ConfigException.BadValue}.
     * @param builder place CPU offset into this builder.
     * @param config retrieve CPU offset from this JSON
     * configuration.
     * @throws ConfigException
     * if CPU offset is invalid.
     */
    private static void setOffset(final Builder builder,
                                  final Config config)
    {
        try
        {
            builder.lastMinusOffset(
                config.getInt(CPU_OFFSET_KEY));
        }
        catch (Exception jex)
        {
            throw (
                new ConfigException.BadValue(
                    CPU_OFFSET_KEY, jex.getMessage()));
        }
    } // end of setOffset(Builder, Config)

    /**
     *
     * Retrieves CPU strategies from JSON configuration and
     * places into builder. Converts builder exception into
     * {@code ConfigException.BadValue}.
     * @param builder place CPU strategies into this builder.
     * @param config retrieve CPU strategies from this JSON
     * configuration.
     * @throws ConfigException
     * if CPU strategies is invalid.
     */
    private static void setStrategies(final Builder builder,
                                      final Config config)
    {
        try
        {
            builder.strategies(
                config.getEnumList(
                    AffinityStrategies.class, STRATEGIES_KEY));
        }
        catch (Exception jex)
        {
            throw (
                new ConfigException.BadValue(
                    STRATEGIES_KEY, jex.getMessage()));
        }
    } // end of setStrategies(Builder, Config)

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Constructs a {@link ThreadAffinityConfigure} instance
     * based on the builder settings. {@code Builder} is the
     * only means available to create a thread affinity
     * configuration and the only way to acquire a builder
     * instance is via {@link ThreadAffinityConfigure#builder()}.
     * <p>
     * The builder default setting is for an unbound
     * {@link AffinityType#ANY_CPU}.
     * </p>
     * <p>
     * The only required setting is
     * {@link #affinityType(ThreadAffinityConfigure.AffinityType) affinity type}. That
     * said, if affinity type is set certain values, then other
     * properties are then required:
     * , then the
     *  is required.
     * </p>
     * <ul>
     *   <li>
     *     {@link AffinityType#CPU_ID CPU_ID}: requires
     *     {@link #cpuId(int) cpu ID} be set.
     *   </li>
     *   <li>
     *     {@link AffinityType#CPU_LAST_MINUS CPU_LAST_MINUS}:
     *     requires {@link #lastMinusOffset(int) cpu offset} be
     *     set.
     *   </li>
     *   <li>
     *     {@link AffinityType#CPU_STRATEGIES CPU_STRATEGIES}:
     *     requires {@link #strategies(net.openhft.affinity.AffinityStrategies...) strateies}
     *     be set.
     *   </li>
     * </ul>
     * <p>
     * Please see {@link ThreadAffinityConfigure} for detailed
     * discussion on how to configure thread affinity.
     * </p>
     * <h1>Example building a {@code ThreadAffinityConfigure} instance</h1>
     * <p>
     * Creates a thread affinity configuration in which the
     * affinity is for a particular CPU identifier, the thread is
     * bound to the affinity lock, and the whole core is
     * reserved for the thread.
     * </p>
     * <pre><code>final ThreadAffinityConfigure.Builder builder = ThreadAffinityConfigure.builder();
final ThreadAffinityConfigure taConfig = builder.affinityType(ThreadAffinityConfigure.AffinityType.CPU_ID)
                                                .cpuId(7)
                                                .bind(true)
                                                .wholeCore(true)
                                                .build();</code></pre>
     *
     * @see ThreadAffinityConfigure
     */
    public static final class Builder
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private AffinityType mAffinityType;
        private boolean mBindFlag;
        private boolean mWholeCore;
        private int mCpuId;
        private int mOffset;
        private List<AffinityStrategies> mStrategies;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Constructor is private since the only way to create
         * a builder instance is by calling
         * {@link ThreadAffinityConfigure#builder()}.
         */
        private Builder()
        {
            mAffinityType = AffinityType.ANY_CPU;
            mBindFlag = false;
            mWholeCore = false;
            mCpuId = -1;
            mOffset = 0;
            mStrategies = null;
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets affinity type to the given value overriding
         * default setting {@link AffinityType#ANY_CPU}.
         * @param type desired thread affinity type.
         * @return {@code this Builder} instance to allow for
         * method chaining.
         * @throws NullPointerException
         * if {@code type} is {@code null}.
         */
        public Builder affinityType(final AffinityType type)
        {
            mAffinityType =
                Objects.requireNonNull(type, "type is null");

            return (this);
        } // end of affinityType(AffinityType)

        /**
         * Sets flag for binding thread to affinity lock
         * overriding default setting {@code false}.
         * @param flag if {@code true} then bind thread to
         * affinity lock.
         * @return {@code this Builder} instance to allow for
         * method chaining.
         */
        public Builder bind(final boolean flag)
        {
            mBindFlag = flag;

            return (this);
        } // end of bind(boolean)

        /**
         * Sets flag reserving the entire core and not allowing
         * hyper-threading on that core. Default settings is
         * {@code false}. This flag is used only when
         * {@link #bind(boolean)} is set to {@code true};
         * otherwise it is ignored.
         * @param flag if {@code true} then binding reserves
         * whole core.
         * @return {@code this Builder} instance to allow for
         * method chaining.
         */
        public Builder wholeCore(final boolean flag)
        {
            mWholeCore = flag;

            return (this);
        } // end of wholeCore(boolean)

        /**
         * Sets CPU_ID identifier used by
         * {@link AffinityType#CPU_ID} affinity type. This value
         * must be set when using {@code CPU_ID} affinity type and
         * is ignored for all others.
         * <p>
         * <strong>Note:</strong> {@code id} is <em>not</em>
         * checked to see if it is a valid CPU_ID identifier
         * beyond making sure it is &ge; zero.
         * </p>
         * @param id CPU_ID identifier.
         * @return {@code this Builder} instance to allow for
         * method chaining.
         * @throws IllegalArgumentException
         * if {@code id} is &lt; zero.
         */
        public Builder cpuId(final int id)
        {
            if (id < 0)
            {
                throw (
                    new IllegalArgumentException("id < zero"));
            }

            mCpuId = id;

            return (this);
        } // end of cpuId(int)

        /**
         * Sets positive integer offset used by
         * {@link AffinityType#CPU_LAST_MINUS} affinity type.
         * This value must be set when using
         * {@code CPU_LAST_MINUS} affinity type and is ignored
         * for all others.
         * <p>
         * <strong>Note:</strong> {@code n} is <em>not</em>
         * checked to see if it is a valid CPU offset beyond
         * making sure it is &gt; zero.
         * @param n offset from end used to select CPU.
         * @return {@code this Builder} instance to allow for
         * method chaining.
         * @throws IllegalArgumentException
         * if {@code n} is &le; zero.
         */
        public Builder lastMinusOffset(final int n)
        {
            if (n <= 0)
            {
                throw (
                    new IllegalArgumentException("n <= zero"));
            }

            mOffset = n;

            return (this);
        } // end of lastMinusOffset(int)

        /**
         * Sets CPU selection strategies used by
         * {@link AffinityType#CPU_STRATEGIES} affinity type.
         * This list must be set when using
         * {@code CPU_STRATEGIES} affinity type and is ignored
         * for all others.
         * <p>
         * <strong>Note:</strong> strategy ordering is important.
         * The {@code ANY} strategy <em>must</em> appear as the
         * last listed strategy. This allows any CPU to be
         * selected in case none of the other strategies found
         * an acceptable CPU.
         * </p>
         * @param strategies CPU selection strategies list.
         * @return {@code this Builder} instance to allow for
         * method chaining.
         * @throws NullPointerException
         * if {@code strategies} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code strategies} is an empty list or the final
         * element is not {@link AffinityStrategies#ANY}.
         *
         * @see #strategies(List)
         */
        public Builder strategies(final AffinityStrategies... strategies)
        {
            if (strategies.length == 0)
            {
                throw (
                    new IllegalArgumentException(
                        "strategies is empty"));
            }

            final AffinityStrategies last =
                strategies[strategies.length - 1];

            if (last != AffinityStrategies.ANY)
            {
                throw (
                    new IllegalArgumentException(
                        "final strategy is not ANY"));
            }

            mStrategies = ImmutableList.copyOf(strategies);

            return (this);
        } // end of strategies(AffinityStrategies...)

        /**
         * Sets CPU selection strategies used by
         * {@link AffinityType#CPU_STRATEGIES} affinity type.
         * This list must be set when using
         * {@code CPU_STRATEGIES} affinity type and is ignored
         * for all others.
         * <p>
         * <strong>Note:</strong> strategy ordering is important.
         * The {@code ANY} strategy <em>must</em> appear as the
         * last listed strategy. This allows any CPU to be
         * selected in case none of the other strategies found
         * an acceptable CPU.
         * </p>
         * @param strategies CPU selection strategies list.
         * @return {@code this Builder} instance to allow for
         * method chaining.
         * @throws NullPointerException
         * if {@code strategies} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code strategies} is an empty list or the final
         * element is not {@link AffinityStrategies#ANY}.
         *
         * @see #strategies(AffinityStrategies...)
         */
        public Builder strategies(final List<AffinityStrategies> strategies)
        {
            if (Objects.requireNonNull(
                    strategies,
                    "strategies is null").isEmpty())
            {
                throw (
                    new IllegalArgumentException(
                        "strategies is empty"));
            }

            final AffinityStrategies last =
                strategies.get(strategies.size() - 1);

            if (last != AffinityStrategies.ANY)
            {
                throw (
                    new IllegalArgumentException(
                        "final strategy is not ANY"));
            }

            mStrategies = ImmutableList.copyOf(strategies);

            return (this);
        } // end of strategies(AffinityStrategies[])

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Returns new {@link ThreadAffinityConfigure} instance
         * created from {@code this Builder}'s settings.
         * @return new {@code ThreadAffinityConfigure} instance.
         * @throws ConfigException
         * if builder settings are invalid. This happens when
         * required settings are not provided for the affinity
         * type.
         */
        public ThreadAffinityConfigure build()
        {
            validate();

            return (new ThreadAffinityConfigure(this));
        } // end of build()

        /**
         * This method validates the thread affinity
         * configuration and returns if the configuration passes.
         * Otherwise throws an {@code ConfigException}. This
         * method is called for effect only.
         * @throws ConfigException
         * if thread affinity configuration is invalid.
         */
        private void validate()
        {
            if (mAffinityType == AffinityType.CPU_ID && mCpuId < 0)
            {
                throw (
                    new ConfigException.BadValue(
                        CPU_ID_KEY,
                        "CPU identifier not configured"));
            }

            if (mAffinityType == AffinityType.CPU_LAST_MINUS &&
                mOffset == 0)
            {
                throw (
                    new ConfigException.BadValue(
                        CPU_OFFSET_KEY,
                        "CPU offset not configured"));
            }

            if (mAffinityType == AffinityType.CPU_STRATEGIES &&
                mStrategies == null)
            {
                throw (
                    new ConfigException.BadValue(
                        STRATEGIES_KEY,
                        "CPU offset not configured"));
            }
        } // end of validate()
    } // end of class ThreadAffinityBuilder
} // end of class ThreadAffinityConfigure
