//
// Copyright 2009 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.config;

import java.io.Serializable;
import java.net.InetSocketAddress;
import java.util.Comparator;

/**
 * Compares two {@code java.net.InetSocketAddress} instances
 * based on the address bytes and the TCP port. Uses
 * {@code net.sf.eBus.net.InetAddressComparator} to compare
 * the socket addresses.
 *
 * @author crapp
 */

public class InetSocketAddressComparator
    implements Comparator<InetSocketAddress>,
               Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Java serialization identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Use this comparator for the InetAddress.
     */
    private final InetAddressComparator mAddressComparator;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new instance of InetSocketAddressComparator.
     */
    public InetSocketAddressComparator()
    {
        mAddressComparator = new InetAddressComparator();
    } // end of InetSocketAddressComparator()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Comparable Interface Implementation.
    //

    /**
     * Returns an integer &lt;, equal to or &gt; zero if
     * {@code a1} is &lt;, equal to or &gt; {@code a2}.
     * The {@code InetAddress}es are compared first. If equal,
     * the ports are compared.
     * @param a1 First compared address.
     * @param a2 Second compared address.
     * @return an integer &lt;, equal to or &gt; zero if
     * {@code a1} is &lt;, equal to or &gt; {@code a2}.
     * @see InetAddressComparator
     */
    @Override
    public int compare(final InetSocketAddress a1,
                       final InetSocketAddress a2)
    {
        int retval =
            mAddressComparator.compare(
                a1.getAddress(), a2.getAddress());

        if (retval == 0)
        {
            retval = (a1.getPort() - a2.getPort());
        }

        return (retval);
    } // end of compare(InetSocketAddress, InetSocketAddress)

    //
    // end of Comparable Interface Implementation.
    //-----------------------------------------------------------
} // end of class InetSocketAddressComparator
