//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2009. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.config;

import java.io.Serializable;
import java.net.InetAddress;
import java.util.Comparator;

/**
 * Compares two {@code InetAddress} addresses using the address
 * bytes.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public class InetAddressComparator
    implements Comparator<InetAddress>,
               Serializable
{
//--------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Java serialization identifier.
     */
    private static final long serialVersionUID = 0x050200L;

//--------------------------------------------------------------
// Member methods.
//

    //----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new instance of InetAddressComparator.
     */
    public InetAddressComparator()
    {}

    //
    // end of Constructors.
    //----------------------------------------------------------

    //----------------------------------------------------------
    // Comparator Interface Implementation.
    //

    /**
     * Returns an integer &lt;, equal to or &gt; zero if
     * {@code a1} is &lt;, equal to or &gt; {@code a2}. The
     * comparison is based on the raw IP address octets. If
     * {@code a1} has fewer bytes or more bytes than {@code a2},
     * then a negative or positve integer is returned,
     * respectively. Otherwise, the IP address octets are
     * compared until the first non-zero result is found.
     * @param a1 First compared address.
     * @param a2 Second compared address.
     * @return an integer &lt;, equal to or &gt; zero if
     * {@code a1} is &lt;, equal to or &gt; {@code a2}.
     */
    @Override
    public int compare(final InetAddress a1,
                       final InetAddress a2)
    {
        final byte[] b1 = a1.getAddress();
        final byte[] b2 = a2.getAddress();
        int index;
        final int size =
            (b1.length < b2.length ? b1.length : b2.length);
        int retval = (b1.length - b2.length);

        for (index = 0; index < size && retval == 0; ++index)
        {
            retval = (b1[index] - b2[index]);
        }

        return (retval);
    } // end of compare(InetAddress, InetAddress)

    //
    // end of Comparator Interface Implementation.
    //----------------------------------------------------------
} // end of class InetAddressComparator
