//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2008 - 2013, 2015 - 2018. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.config;

import com.google.common.base.Strings;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigException;
import com.typesafe.config.ConfigObject;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.ProtocolFamily;
import java.net.StandardProtocolFamily;
import java.net.UnknownHostException;
import java.nio.ByteOrder;
import java.text.ParseException;
import java.time.Duration;
import java.util.Arrays;
import java.util.Formatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.Objects;
import java.util.Properties;
import java.util.function.Consumer;
import java.util.prefs.Preferences;
import javax.net.ssl.SSLContext;
import static net.sf.eBus.config.ENetConfigure.ANY_PORT;
import static net.sf.eBus.config.ENetConfigure.DEFAULT_BUFFER_SIZE;
import static net.sf.eBus.config.ENetConfigure.MAX_PORT;
import static net.sf.eBus.config.ENetConfigure.MIN_PORT;

/**
 * This immutable class contains eBus services, remote
 * connection, dispatcher, and pause configuration. This
 * configuration can be specified:
 * <ul>
 *   <li>
 *     By building {@link EConfigure.Service} and
 *     {@link EConfigure.RemoteConnection} instances and passing
 *     these objects to
 *     {@code EServer.configure(EConfigure)} and
 *     {@code ERemoteApp.configure(EConfigure)}, respectively.
 *     {@link EConfigure.Dispatcher} has a private constructor
 *     and cannot be instantiated by the application.
 *   </li>
 *   <li>
 *     From properties.
 *   </li>
 *   <li>
 *     From preferences.
 *   </li>
 *   <li>
 *     From <a href="https://github.com/lightbend/config">typesafe</a>
 *     JSON-based configuration file.
 *   </li>
 * </ul>
 * <p>
 * Also provides methods for storing this configuration to
 * {@link #store(java.util.Properties) properties} or
 * {@link #store(Preferences) preferences}.
 * </p>
 * <p>
 * Please note that eBus configuration requires no service or
 * connections to be specified. If this is the case, remote
 * connections are neither accepted nor established.
 * </p>
 * <p>
 * eBus release 4.7.0 supports secure TCP connections based on
 * SSL/TLS using {@link SSLContext}. This connection type is
 * <em>not</em> supported using properties/preferences
 * configuration as this would require placing sensitive
 * information in an insecure manner. Therefore, creating
 * secure TCP services and connections can only be done
 * using the API.
 * </p>
 * <p>
 * Dispatcher threads are instantiated on JVM start-up, the
 * dispatcher configuration is taken from the properties file
 * specified by the command line parameter
 * {@code -Dnet.sf.eBus.config.file=}&lt;<em>property file path</em>&gt;
 * or
 * {@code -Dnet.sf.eBus.config.jsonFile=}&lt;<em>typesafe json file path</em>&gt;.
 * eBus loads the {@code EClientDispatcher} configuration from
 * this properties file and creates the dispatcher threads
 * according to this configuration. If no configuration file is
 * specified or is incorrectly configured, eBus defaults to a
 * single, blocking run queue. The run queue thread count equals
 * {@link Runtime#availableProcessors()}.
 * </p>
 * <p>
 * The eBus configuration uses the following keys for both
 * properties and preferences. JSON configuration keys are
 * similar and shown <a href="json-config">here</a>. As of
 * eBus v. 5.1.0, eBus connections may be paused which is
 * described in detail <a href="connection-pause">here</a>.
 * </p>
 * <ul>
 *   <li>
 *     {@code eBus.servers}: a comma-separated list of
 *     eBus local server names. This key is optional or may
 *     be an empty string. The names must be unique (no
 *     duplicates). The server name is used to retrieve the
 *     following keys:
 *     <ol>
 *       <li>
 *         <code>eBus.service.<em>name</em>.port</code>:
 *         the service is opened on this TCP port.
 *         <p>
 *         This property is required.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.addressFilter</code>:
 *         the {@link AddressFilter} limiting which clients may
 *         connect to this eBus application. This is a positive
 *         filter meaning that is specifies which clients may
 *         connect to this application and not a negative filter
 *         which allows all connections except those specified.
 *         <p>
 *         This property is a string which must satisfy
 *         {@link AddressFilter#parse(String)}.
 *         </p>
 *         <p>
 *         This property is optional and may be set to an empty
 *         string.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.inputBufferSize</code>:
 *         the socket input buffer size in bytes per connection.
 *         Must be an integer value &gt; zero.
 *         <p>
 *         The default setting is
 *         {@code ERemoteApp.DEFAULT_BUFFER_SIZE}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.outputBufferSize</code>:
 *         the socket output buffer size in bytes per connection.
 *         Must be an integer value &gt; zero.
 *         <p>
 *         The default setting is
 *         {@code ERemoteApp.DEFAULT_BUFFER_SIZE}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.byteOrder</code>:
 *         the socket input and output
 *         {@link java.nio.ByteBuffer}s are set to this byte
 *         order.
 *         <p>
 *         The default setting is
 *         {@link java.nio.ByteOrder#LITTLE_ENDIAN}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.messageQueueSize</code>:
 *         when the output buffer is full, outgoing messages are
 *         queued until the output buffer is no longer full.
 *         This value specifies the maximum number of messages
 *         which may enqueued at any one time. Must be an integer
 *         &ge; zero where zero means the message queue size is
 *         unlimited.
 *         <p>
 *         The default setting is an unlimited message queue.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.server.<em>name</em>.serviceSelector</code>:
 *         the named selector thread used to monitor the server
 *         socket channel.
 *         <p>
 *         The default setting is
 *         {@code net.sf.eBus.net.AsyncChannel.defaultSelector}
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.server.<em>name</em>.connectionSelector</code>:
 *         the named selector thread used to monitor the accepted
 *         client socket channels.
 *         <p>
 *         The default setting is
 *         {@code net.sf.eBus.net.AsyncChannel.defaultSelector}
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.heartbeatDelay</code>:
 *         the accepted connection sends heartbeat messages this
 *         many milliseconds after the last received message.
 *         Every time a message is received, this timer is reset.
 *         Must be an integer &ge; zero where zero means that
 *         no heartbeat message is sent.
 *         <p>
 *         The default is not to send heartbeat message.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.heartbeatReplyDelay</code>:
 *         wait this many milliseconds for a reply to a heartbeat
 *         message. If a reply is <i>not</i> received in this
 *         time limit, close the connection. Must be an integer
 *         value &ge; zero where zero means no time limit.
 *         <p>
 *         The default is no time limit because heart beating is
 *         off by default.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.canPause</code>:
 *         this boolean value specifies whether accepted
 *         connections may be paused by the remote connection
 *         initiator. If {@code true}, then the following two
 *         properties are used to limit how long the connection
 *         may be paused and the number of messages that may
 *         be queued up waiting for transmission when the
 *         connection is resumed.
 *         <p>
 *         The default is {@code false}, accepted connections
 *         may not be paused.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.pause.pauseTime</code>:
 *         the maximum allowed pause time for accepted
 *         connections. The value is in milliseconds.
 *         <p>
 *         This value is required if {@code canPause} is
 *         {@code true}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.pause.maxBacklogSize</code>:
 *         when a connection is paused, at most this many
 *         messages are queued up for transmit when the
 *         connection is resumed. When the limit is reached,
 *         messages are discarded based on the client-specified
 *         {@link net.sf.eBus.config.EConfigure.DiscardPolicy}.
 *         <p>
 *         The default value is zero which means there is no
 *         limit to the message backlog queue.
 *         </p>
 *       </li>
 *     </ol>
 *   </li>
 *   <li>
 *     {@code eBus.connections}: a comma-separated list of
 *     eBus remote connection names. This key is optional or may
 *     be an empty string. The names must be unique (no
 *     duplicates). The connection name is used to retrieve the
 *     following keys:
 *     <ul>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.host</code>:
 *         the connection is made to this host. May be either a
 *         host name or dotted notation IP address.
 *         <p>
 *         This setting is required.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.port</code>:
 *         the connection is made to this TCP port.
 *         <p>
 *         This setting is required.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.bindPort</code>:
 *         the connection's local end is bound to this TCP port.
 *         <p>
 *         The default setting is to bind the local port is
 *         OS-dependent. The OS will bind the local port to what
 *         is considered the next available TCP port.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.inputBufferSize</code>:
 *         the socket input buffer size in bytes. Must be an
 *         integer value &gt; zero.
 *         <p>
 *         The default setting is
 *         {@code ERemoteApp.DEFAULT_BUFFER_SIZE}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.outputBufferSize</code>:
 *         the socket output buffer size in bytes. Must be an
 *         integer value &gt; zero.
 *         <p>
 *         The default setting is
 *         {@code ERemoteApp.DEFAULT_BUFFER_SIZE}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.byteOrder</code>:
 *         the socket input and output
 *         {@link java.nio.ByteBuffer}s are set to this byte
 *         order.
 *         <p>
 *         The default setting is
 *         {@link java.nio.ByteOrder#LITTLE_ENDIAN}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.messageQueueSize</code>:
 *         when the output buffer is full, outgoing messages are
 *         queued until the output buffer is no longer full.
 *         This value specifies the maximum number of messages
 *         which may enqueued at any one time. Must be an integer
 *         &ge; zero where zero means the queue size is
 *         unlimited.
 *         <p>
 *         The default setting is an unlimited message queue
 *         size.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.selector</code>:
 *         the named selector thread used to monitor this socket
 *         channel.
 *         <p>
 *         The default setting is
 *         {@code net.sf.eBus.net.AsyncChannel.defaultSelector}
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.reconnect</code>:
 *         boolean value which specifies whether the connection
 *         is automatically reconnected if lost if true. Should
 *         be {@code true} or {@code false}.
 *         <p>
 *         The default setting is {@code false}. This default
 *         value is used if the key is either missing, has no
 *         value, or the value cannot be decoded.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.reconnectTime</code>:
 *         the number of milliseconds between reconnect attempts.
 *         This is an 8-byte, signed integer value and must be
 *         &gt; zero.
 *         <p>
 *         The default setting is
 *         {@code ERemoteApp.DEFAULT_RECONNECT_TIME}. This value
 *         is ignored if reconnect is {@code false}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.heartbeatDelay</code>:
 *         send a heartbeat message after this many milliseconds
 *         after the last received message. Every time a message
 *         is received, this timer is reset. This integer value
 *         must be &ge; zero where zero means that heartbeat
 *         messages are not sent.
 *         <p>
 *         The default setting is not to send a heartbeat
 *         message.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.heartbeatReplyDelay</code>:
 *         wait this many milliseconds for a reply to a heartbeat
 *         message. If a reply is <i>not</i> received in this
 *         time limit, close the connection. This integer value
 *         must be &ge; zero where zero means there is no
 *         heartbeat reply time limit.
 *         <p>
 *         The default setting is no time limit because
 *         heart beating is off by default.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.canPause</code>:
 *         this boolean value specifies whether the connection
 *         will pause or not based on the pause properties. If
 *         {@code true}, then the following properties are used
 *         to decide when the connection is paused and when the
 *         connection is resumed.
 *         <p>
 *         The default is {@code false}, the connection will not
 *         pause.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.pause.pauseTime</code>:
 *         the connection will pause for at most this much time.
 *         If the remote end has a shorter maximum pause time,
 *         then that shorted time period is used. The value is
 *         in milliseconds.
 *         <p>
 *         This value is required if {@code canPause} is
 *         {@code true}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.pause.maxBacklogSize</code>:
 *         when a connection is paused, at most this many
 *         messages are queued up for transmit when the
 *         connection is resumed. When the limit is reached,
 *         messages are discarded based on the client-specified
 *         {@link net.sf.eBus.config.EConfigure.DiscardPolicy}.
 *         If the remote end supports a lesser value, then that
 *         lesser value is used.
 *         <p>
 *         The default value is zero which means there is no
 *         limit to the message backlog queue.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.pause.discardPolicy</code>:
 *         when the pending message queue reaches the
 *         {@code maxBacklogSize}, messages are discard based on
 *         either
 *         {@link net.sf.eBus.config.EConfigure.DiscardPolicy#YOUNGEST_FIRST}
 *         or
 *         {@link net.sf.eBus.config.EConfigure.DiscardPolicy#OLDEST_FIRST}
 *         <p>
 *         The default value is {@code OLDEST_FIRST}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.pause.idleTime</code>:
 *         pause the connection if no messages are sent or
 *         received on this connection for this duration. The
 *         value is in milliseconds. If specified this value
 *         <em>should</em> be &lt; {@code maxConnectTime}. This
 *         rule is not enforced.
 *         <p>
 *         This property is optional and defaults to zero
 *         milliseconds which means idle time is turned off.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.pause.maxConnectTime</code>:
 *         a connection may remain up for at most this many
 *         milliseconds regardless of connection being busy.
 *         When this limit is reached the connection is
 *         immediately paused.
 *         <p>
 *         This property is required if {@code canPause} is
 *         {@code true}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.pause.resumeOnBacklogSize</code>:
 *         when the message backlog reaches this limit,
 *         automatically resume the connection. If specified,
 *         this value <em>should</em> be &lt;
 *         {@code maxBacklogSize} to allow time for the
 *         connection to be resumed and the pending messages sent
 *         before the maximum backlog limit is reached and
 *         messages are discarded. This rule is not enforced.
 *         <p>
 *         This property is optional and defaults to zero which
 *         means this capability is disabled.
 *         </p>
 *       </li>
 *     </ul>
 *   </li>
 *   <li>
 *     {@code eBus.dispatchers}: a comma-separated list of eBus
 *     dispatcher groups. This key is optional and may be set to
 *     an empty string. The names must be unique with no
 *     duplicates. The dispatcher name is used to retrieve the
 *     following keys:
 *     <br>
 *     (<strong>Note:</strong> Dispatcher configuration must be placed into a
 *     properties file and the {@code java} command line
 *     parameter
 *     {@code -Dnet.sf.eBus.config.file=}&lt;{@code properties file path}&gt;
 *     must be set because dispatchers are started by the
 *     {@code EClient} class initializer block.)
 *     <ul>
 *       <li>
 *         <code>eBus.dispatcher.<em>name</em>.isDefault</code>:
 *         Set to "true" if this is the default dispatcher. The
 *         default dispatcher is used for all client classes not
 *         explicitly assigned to a dispatcher. If multiple
 *         dispatchers are designated as the default, then it
 *         cannot be determined which selector will be used as
 *         default.
 *         <p>
 *         The default value for this setting is {@code false}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.dispatcher.<em>name</em>.runQueueType</code>:
 *         Defines how the dispatcher thread removes clients from
 *         the run queue: {@code blocking}, {@code spinning},
 *         {@code spin+park}, {@code spin+yield}. Blocking is the
 *         most fair since it allows other threads to use the
 *         core while waiting a client to be offered to the run
 *         queue; but blocking is also the slowest. Spinning
 *         means that the dispatcher is repeatedly calling
 *         {@link java.util.Queue#poll()} until a client is
 *         returned. Spinning is the fastest extraction technique
 *         but monopolizes the core, preventing other threads
 *         from using it. The spin+park and spin+yield types are
 *         a compromise between blocking and spinning. The
 *         dispatcher calls {@code Queue.poll} a configured
 *         number of times, trying to extract a client. If no
 *         client is returned after that configured spin limit is
 *         reached, the dispatcher either parks for a specified
 *         nanosecond time or yields the core. When the
 *         dispatcher returns from the park or yield, the spin
 *         limit is reset to the configured limit and the poll
 *         spin begins again.
 *         <p>
 *         The default value for this setting is {@code blocking}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.dispatcher.<em>name</em>.spinLimit</code>:
 *         If {@code runQueueType} is set to {@code spin+park} or
 *         {@code spin+yield}, then this is the
 *         {@code Queue.poll} spin limit.
 *         <p>
 *         The default value for this setting is
 *         {@link #DEFAULT_SPIN_LIMIT}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.dispatcher.<em>name</em>.parkTime</code>:
 *         If {@code runQueueType} is set to {@code spin+park},
 *         then this is the nanosecond park time.
 *         <p>
 *         The default value for this setting is
 *         {@link #DEFAULT_PARK_TIME}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.dispatcher.<em>name</em>.priority</code>:
 *         The dispatcher threads are run at this thread
 *         priority. The specified value must be
 *         &ge; {@link Thread#MIN_PRIORITY} and
 *         &le; {@link Thread#MAX_PRIORITY}.
 *         <p>
 *         The default value for this setting is
 *         {@link Thread#NORM_PRIORITY}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.dispatcher.<em>name</em>.quantum</code>:
 *         This is the maximum number of nanoseconds a client may
 *         consume on a dispatcher thread before it musts yield
 *         to another consumer. Note that this is <em>not</em> a
 *         preemptive yield. The dispatcher thread tracks how
 *         long a client takes processing each message, adding up
 *         total time spent. When the accumulated time equals or
 *         exceeds the quantum, the client is posted back to the
 *         run queue if it has more messages to process. The
 *         quantum is reset each time the client is acquired by a
 *         dispatcher thread.
 *         <p>
 *         <strong>Note:</strong> it is possible for a client to
 *         monopolize a dispatcher thread, denying the thread to
 *         other clients associated with the dispatcher.
 *         </p>
 *         <p>
 *         The default value for this setting is
 *         {@link #DEFAULT_QUANTUM}
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.dispatcher.<em>name</em>.classes</code>:
 *         specifies the client classes associated with this
 *         dispatcher. A class should appear in at most one
 *         dispatcher {@code classes} list. eBus does not detect
 *         if a class is assigned to multiple dispatchers and it
 *         is unknown to which dispatcher this class will be
 *         assigned.
 *         <p>
 *          This property is ignored if {@code isDefault} is
 *          {@code true}.
 *         </p>
 *         <p>
 *         This property is required if this not a default
 *         dispatcher.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.dispatcher.<em>name</em>.numberThreads</code>:
 *         The number of threads in this dispatcher. Must be
 *         &gt; zero.
 *         <p>
 *         The default value is {@link #DEFAULT_NUMBER_THREADS}.
 *         </p>
 *       </li>
 *     </ul>
 *   </li>
 * </ul>
 * See {@link EConfigure.Dispatcher} for information about two
 * special, pre-defined {@code Dispatcher} types:
 * {@link DispatcherType#SWING} and {@link DispatcherType#JAVAFX}
 * which deliver eBus messages via the GUI thread.
 * <p>
 * See the eBus overview section on connecting eBus applications
 * for a detailed description in configuring eBus connections.
 * </p>
 * <h2>
 *   <a name="connection-pause">Connection Pause</a>
 * </h2>
 * <p>
 * eBus release 5.1.0 introduces the ability to automatically
 * pause a connection when it has been idle or continuously
 * connected for a given time. When paused, messages are kept on
 * a backlog for later transmission when the connection is
 * resumed. The message backlog queue size may be limited. Once
 * the queue size reaches that limit, then messages are discarded
 * based on the specified discard policy.
 * </p>
 * <p>
 * The connection is resumed when the pause time is reached.
 * This pause time is always used.
 * Optionally a resume-on-backlog-size may also be set. When
 * the backlog size reaches this size while paused, the
 * connection is resumed even if the pause time is not yet
 * reached.
 * </p>
 * <p>
 * The connection pause time and maximum backlog size are
 * negotiated between the two connection ends. The lesser pause
 * time and backlog size values are used by both ends.
 * </p>
 * <p>
 * This feature is added with an eye towards mobile devices which
 * cannot support keeping connections up for an extended time due
 * to battery limitations.
 * </p>
 * <h2>
 *   <a name="json-config">eBus JSON Configuration</a>
 * </h2>
 * <p>
 * eBus now supports
 * <a href="https://github.com/lightbend/config">typesafe</a>
 * JSON configuration. This represents a subset of JSON known as
 * <a href="https://github.com/lightbend/config/blob/master/HOCON.md">HOCON (Human-Optimized Config Object Notation).</a>
 * The following is an eBus configuration in HOCON. Going forward
 * typesafe HOCON will be the preferred method for configuring
 * eBus with {@code java.util.Properties} and
 * {@code java.util.prefs.Preferences} being deprecated.
 * </p>
 * <pre><code>selectors : [
    {
        name : selector1
        type : "spin+park"
        isDefault : true
        priority : 7
        spinLimit : 1000000
        parkTime : 500
    }
]

services : [
     {
        name : service1
        port : 12345
        addressFilter : [
            "127.0.0.1",
            "127.0.0.1:54321"
        ]
        serviceSelector : selector1
        connectionSelector : selector1
        byteOrder : LITTLE_ENDIAN
        inputBufferSize : 8192
        outputBufferSize : 65536
        messageQueueSize : 100
        canPause : true

        // Connection pause/resume configuration.
        pause : {
            pauseTime : 10m
            maxBacklogSize : 50
        }
    }
]

connections : [
     {
        name : conn1
        host : "127.0.0.1"
        port : 12346
        bindPort : 0
        byteOrder : BIG_ENDIAN
        selector : selector1
        inputBufferSize : 8192
        outputBufferSize : 65536
        messageQueueSize : 100
        reconnect : true
        reconnectTime : 500
        canPause : true

        // Connection pause/resume configuration.
        pause : {
            pauseTime : 5m
            maxBacklogSize : 100
            discardPolicy : YOUNGEST_FIRST
            idleTime : 1m
            maxConnectTime : 2m
            resumeOnBacklogSize : 10
        }
    }
]

dispatchers : [
    {
        name : d1
        runQueueType : "spin+park"
        spinLimit : 2500000
        parkTime : 500
        isDefault : true
        priority : 5
        quantum : 1000
        numberOfThreads : 4
    }
]</code></pre>
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class EConfigure
{
//---------------------------------------------------------------
// Enums.
//

    /**
     * Enumerates the channel types supported by eBus
     * {@code ERemoteApp}. This is currently limited to either
     * TCP or secure TCP (TLS - Transport Layer Security).
     */
    public enum ConnectionType
    {
        /**
         * Remote application connection is a TCP socket.
         */
        TCP,

        /**
         * Remote application connection is a TCP socket with
         * TLS (Transport Layer Security).
         */
        SECURE_TCP,

        /*
         * Remote application connection is a UDP socket.
         */
        UDP
    } // end of ConnectionType

    /**
     * Enumerates the supported eBus dispatcher thread types.
     * There are effectively two dispatcher types: eBus and
     * GUI with GUI divided between Swing and JavaFX.
     */
    public enum DispatcherType
    {
        /**
         * Default eBus dispatcher based on a run queue, so the
         * dispatcher method is
         * {@code EClient.doDispatch(Runnable)}.
         */
        EBUS (false, null),

        /**
         * Posts a task to the JavaFX GUI thread using
         * {@code javafx.application.Platform.runLater(Runnable)}.
         */
        JAVAFX (true,
                task ->
                {
                    // If currently running in the JavaFX GUI
                    // thread, then execute the task immediately.
                    if (javafx.application.Platform.isFxApplicationThread())
                    {
                        try
                        {
                            task.run();
                        }
                        catch (Exception jex)
                        {
                            System.err.println(
                                jex.getLocalizedMessage());
                        }

                    }
                    else
                    {
                        javafx.application.Platform.runLater(task);
                    }
                }),

        /**
         * Posts a task to the Swing GUI thread using
         * {@code javax.swing.SwingUtilities.invokeLater(Runnable)}.
         */
        SWING (true,
               task ->
               {
                   // If currently running in the Swing GUI
                   // thread, then execute the task immediately.
                   if (javax.swing.SwingUtilities.isEventDispatchThread())
                   {
                       try
                       {
                           task.run();
                       }
                       catch (Exception jex)
                       {
                            System.err.println(
                                jex.getLocalizedMessage());
                       }
                   }
                   else
                   {
                       javax.swing.SwingUtilities.invokeLater(task);
                   }
               });

    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Special dispatchers may only be marked as the default
         * dispatcher. All other properties are ignored.
         */
        private final boolean mSpecial;

        /**
         * References the method used to dispatch a task. Will be
         * {@code null} if {@code mRunQueue} is not {@code null}.
         */
        private final Consumer<Runnable> mDispatchHandle;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new dispatcher type for the given run queue
         * and handle.
         * @param special marks this as a special dispatcher
         * which may not be configured.
         * @param handle dispatcher method handle. Must be
         * {@code null} when {@code handle} is not {@code null}.
         */
        private DispatcherType(final boolean special,
                               final Consumer<Runnable> handle)
        {
            mSpecial = special;
            mDispatchHandle = handle;
        } // end of DispatcherType(boolean, Consumer<>)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns {@code true} if this dispatcher type is a
         * special, non-configurable dispatcher.
         * @return {@code true} if a special dispatcher.
         */
        public boolean isSpecial()
        {
            return (mSpecial);
        } // end of isSpecial()

        /**
         * Returns the task dispatch method handle. May be
         * {@code null}.
         * @return dispatch method handle.
         */
        public Consumer<Runnable> dispatchHandle()
        {
            return (mDispatchHandle);
        } // end of dispatchHandle()

        /**
         * Returns the {@link DispatcherType} for the given
         * {@code name} using case-insensitive string matching.
         * If {@code name} is either {@code null} or an empty
         * string, then returns {@link DispatcherType#EBUS} by
         * default.
         * @param name find dispatcher type with the given name
         * using case-insensitive matching.
         * @return matching dispatcher type; defaults to
         * {@link #EBUS}.
         */
        public static DispatcherType findType(final String name)
        {
            final DispatcherType[] types =
                DispatcherType.values();
            final int numTypes = types.length;
            int index;
            DispatcherType retval = null;

            for (index = 0;
                 index < numTypes && retval == null;
                 ++index)
            {
                if (name.equalsIgnoreCase(types[index].name()))
                {
                    retval = types[index];
                }
            }

            if (retval == null)
            {
                retval = EBUS;
            }

            return (retval);
        } // end of findType(String)

        //
        // end of Get Methods.
        //-------------------------------------------------------
    } // end of enum DispatcherType

    /**
     * When a connection is paused and the server message backlog
     * breaches the maximum allowed message backlog limit, this
     * policy specifies which messages should be discarded to
     * keep the backlog at the limit.
     */
    public enum DiscardPolicy
    {
        /**
         * Discard the oldest messages first (FIFO).
         */
        OLDEST_FIRST,

        /**
         * Discard the youngest messages first (LIFO).
         */
        YOUNGEST_FIRST
    } // end of enum DiscardPolicy

    /**
     * Defines remote connection side, either acceptor or
     * initiator.
     */
    public enum ConnectionRole
    {
        /**
         * This side initiated the connection.
         */
        INITIATOR,

        /**
         * This side accepted the connection.
         */
        ACCEPTOR
    } // end of enum ConnectionRole

//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Use {@code -D} to set system property {@value} to point to
     * the eBus configuration property file. eBus will configure
     * itself as per the file's properties.
     * <p>
     * <strong>Please note:</strong> either
     * {@code -Dnet.sf.eBus.config.file} or
     * {@code -Dnet.sf.eBus.config.jsonFile} or neither is
     * defined but not both. The application will exit if that
     * is the case since eBus will not attempt to merge the
     * two configurations.
     * </p>
     *
     * @see #JSON_FILE_ENV
     */
    public static final String CONFIG_FILE_ENV =
        "net.sf.eBus.config.file";

    /**
     * Use {@code -D} to set system property {@value} to point to
     * the eBus JSON configuration file. eBus will configure
     * itself as per the file.
     * <p>
     * <strong>Please note:</strong> either
     * {@code -Dnet.sf.eBus.config.file} or
     * {@code -Dnet.sf.eBus.config.jsonFile} or neither is
     * defined but not both. The application will exit if that
     * is the case since eBus will not attempt to merge the
     * two configurations.
     * </p>
     *
     * @see #CONFIG_FILE_ENV
     */
    public static final String JSON_FILE_ENV =
        "net.sf.eBus.config.jsonFile";

    //
    // System property keys.
    //

    // Property keys.

    /**
     * All properties keys are prefixed with {@value}.
     */
    public static final String EBUS_KEY = "eBus";

    /**
     * Key {@value} is used to extract the configuration object's
     * name.
     */
    public static final String NAME_KEY = "name";

    /**
     * The key {@value} contains a comma-separated list of local
     * service names.
     */
    public static final String SERVICES_KEY = "services";

    /**
     * Service keys are prefixed by {@value} and followed by the
     * service name.
     */
    public static final String SERVICE_PREFIX =
        EBUS_KEY + ".service.";

    /**
     * The key {@value} contains a comma-separated list of remote
     * connection names.
     */
    public static final String CONNECTIONS_KEY =
        "connections";

    /**
     * Remote connection keys are prefixed by {@value} and
     * followed by the connection name as found in
     * {@link #CONNECTIONS_KEY}.
     */
    public static final String CONNECTION_PREFIX =
        EBUS_KEY + ".connection.";

    /**
     * The key {@value} contains a comma-separated list of
     * unique, client dispatcher names.
     */
    public static final String DISPATCHERS_KEY =
        "dispatchers";

    /**
     * Dispatcher keys are prefixed by {@value} and followed by
     * the dispatcher name as found in {@link #DISPATCHERS_KEY}.
     */
    public static final String DISPATCHER_PREFIX =
        "dispatcher.";

    /**
     * Both service and connection definitions use the {@value}
     * to define the connection protocol type.
     */
    public static final String CONN_TYPE_KEY = "connectionType";

    /**
     * Both the service and connection definitions use the
     * {@value} key suffix. Must be an integer &gt; zero.
     *
     */
    public static final String PORT_KEY = "port";

    /**
     * Both the service and connection definitions use the
     * {@value} key suffix. Must be an integer &gt; zero.
     */
    public static final String INBUFFER_SIZE_KEY =
        "inputBufferSize";

    /**
     * Both the service and connection definitions use the
     * {@value} key suffix. Must be an integer &gt; zero.
     */
    public static final String OUTBUFFER_SIZE_KEY =
        "outputBufferSize";

    /**
     * Both the service and connection definitions use the
     * {@value} key suffix. Must be either
     * {@link ByteOrder#BIG_ENDIAN} or
     * {@link ByteOrder#LITTLE_ENDIAN}.
     * <p>
     * The default is {@link ByteOrder#LITTLE_ENDIAN}.
     * </p>
     */
    public static final String BYTE_ORDER_KEY =
        "byteOrder";

    /**
     * Both the service and connection definitions use the
     * {@value} key suffix. Must be an integer &gt; zero.
     */
    public static final String MSG_QUEUE_SIZE_KEY =
        "messageQueueSize";

    /**
     * The service definition uses the key suffix
     * {@value} to define the selector used to monitor the
     * service socket. Must be a known selector. If not
     * specified, then defaults to
     * {@code AsyncChannel.defaultSelector}.
     */
    public static final String SVC_SELECTOR_KEY =
        "serviceSelector";

    /**
     * The service definition uses the key suffix
     * {@value} to define the selector used to monitor TCP socket
     * channels accepted by the service socket. Must be a known
     * selector. If not specified, then defaults to
     * {@code AsyncChannel.defaultSelector}.
     */
    public static final String CONN_SELECTOR_KEY =
        "connectionSelector";

    /**
     * Both the service and connection definitions use the
     * {@value} key suffix. Must be a known selector. If not
     * specified, then defaults to
     * {@code AsyncChannel.defaultSelector}.
     */
    public static final String SELECTOR_KEY = "selector";

    /**
     * The connection definition uses the {@value} key suffix.
     * Must be either a valid host name or IP address in dotted
     * notation.
     */
    public static final String HOST_KEY = "host";

    /**
     * The connection definition uses the {@value} key suffix.
     * Must be an integer &ge; zero.
     */
    public static final String BIND_PORT_KEY = "bindPort";

    /**
     * The connection definition uses the {@value} key suffix.
     * Must be a boolean.
     */
    public static final String RECONNECT_KEY = "reconnect";

    /**
     * The connection definition uses the {@value} key suffix.
     * Must be an integer &ge; zero.
     */
    public static final String RECONNECT_DELAY_KEY =
        "reconnectTime";

    /**
     * The connection definition uses the {@value} key suffix.
     * Must be an integer &ge; zero.
     */
    public static final String HB_DELAY_KEY = "heartbeatDelay";

    /**
     * The connection definition uses the {@value} key suffix.
     * Must be an integer &ge; zero.
     */
    public static final String HB_REPLY_DELAY_KEY =
        "heartbeatReplyDelay";

    /**
     * The service definition uses the {@value} key suffix. Must
     * be a valid address filter list.
     */
    public static final String FILTER_KEY = "addressFilter";

    /**
     * The boolean {@value} property is used to specify when this
     * dispatcher should be used as the default dispatcher. At
     * most one dispatcher should be designated as the default
     * dispatcher. If more than one is so designated, the
     * subsequent {@code .isDefault} keys are ignored.
     * <p>
     * If no user-defined dispatcher is marked as default, then
     * a {@link ThreadType#BLOCKING blocking},
     * {@link Thread#NORM_PRIORITY normal priority} dispatcher
     * named {@code EClient.DEFAULT_DISPATCHER} is used as
     * the default dispatcher.
     * </p>
     * <p>
     * The default value is {@code false}
     * (not the default selector).
     * </p>
     */
    public static final String DEFAULT_KEY = "isDefault";

    /**
     * The {@link DispatcherType} {@value} property is used to
     * specify the dispatcher thread type. This property allows
     * eBus tasks to be executed on supported third-party
     * threads.
     * <p>
     * The default value is {@link DispatcherType#EBUS}.
     * </p>
     */
    public static final String DISPATCHER_TYPE_KEY =
        "dispatcherType";

    /**
     * The {@link ThreadType run queue} {@value} property is used
     * to specify how the dispatcher thread acquires the next
     * available {@code EClient} from the run queue by either
     * blocking, spinning, spin+park, or spin+yield.
     * <p>
     * The default value is {@link #DEFAULT_THREAD_TYPE}.
     * </p>
     */
    public static final String THREAD_TYPE_KEY = "runQueueType";

    /**
     * If the thread type is
     * {@link ThreadType#SPINPARK spin+park} or
     * {@link ThreadType#SPINYIELD}, then the
     * integer {@code .spinLimit} property may be specified. This
     * setting defines the number of times the Dispatcher thread
     * may call
     * {@link java.util.Queue#poll()} before parking or yielding.
     * <p>
     * This value must be &gt; zero.
     * </p>
     * <p>
     * Default values is {@link #DEFAULT_SPIN_LIMIT}.
     * </p>
     * <p>
     * This property is ignored is the thread type is not
     * spin+park or spin+yield.
     * </p>
     */
    public static final String SPIN_LIMIT_KEY = "spinLimit";

    /**
     * If the thread type is
     * {@link ThreadType#SPINPARK spin+park}, then the
     * integer {@code .parkTime} property may be specified. This
     * setting specifies the <em>nanosecond</em> park time taken
     * between {@link java.util.Queue#poll()} spin cycles.
     * <p>
     * This value must be &gt; zero.
     * </p>
     * <p>
     * Default values is {@link #DEFAULT_PARK_TIME}.
     * </p>
     * <p>
     * This property is ignored is the thread type is not
     * spin+park.
     * </p>
     */
    public static final String PARK_TIME_KEY = "parkTime";

    /**
     * The integer {@value} property is used to set the dispatcher
     * thread priority.
     * <p>
     * The default value is {@link Thread#NORM_PRIORITY}.
     * </p>
     */
    public static final String PRIORITY_KEY = "priority";

    /**
     * The key {@value} defines the run quantum each
     * client is initially assigned. If a client exhausts its
     * quantum, the client is placed back on the run queue.
     */
    public static final String QUANTUM_KEY = "quantum";

    /**
     * The key {@value} defines the classes which are
     * posted to this dispatcher. This key is ignored if
     * {@link #DEFAULT_KEY .isDefault} is set to {@code true}.
     */
    public static final String CLASSES_KEY = "classes";

    /**
     * The dispatcher definition uses {@value}
     * key suffix to specify the number of threads in the
     * dispatcher.
     */
    public static final String NUM_THREADS_KEY =
        "numberThreads";

    /**
     * If a remote client connection may be paused, then this
     * property is defined and set to {@code true}. If not set
     * then assumed to be {@code false} and the connection cannot
     * be paused. This key may be defined for both acceptor and
     * initiator connections.
     */
    public static final String CAN_PAUSE_KEY = "canPause";

    /**
     * If {@link #CAN_PAUSE_KEY} is set to {@code true}, then
     * the connection pause parameters are stored under the
     * {@value} key. This key may be defined for both acceptor
     * and initiator connections.
     */
    public static final String PAUSE_KEY = "pause";

    /**
     * The pause duration is stored in {@value}. This is a
     * sub-key under {@link #PAUSE_KEY}. This key may be defined
     * for both acceptor and initiator connections.
     */
    public static final String PAUSE_DURATION_KEY = "pauseTime";

    /**
     * Maximum allowed message backlog when paused. This key may
     * be defined for both acceptor and initiator connections.
     */
    public static final String MAX_BACKLOG_SIZE_KEY =
        "maxBacklogSize";

    /**
     * When the maximum message backlog limit is breached, then
     * this policy defines which messages should be discarded.
     * This key is defined for initiator connections only.
     */
    public static final String DISCARD_POLICY_KEY =
        "discardPolicy";

    /**
     * When a connection is idle (no messages sent or received)
     * for a given time, that connection is automatically paused.
     * This value is set in the {@value} property.
     */
    public static final String IDLE_TIME_KEY = "idleTime";

    /**
     * Maximum allowed time between pauses, irrespective of
     * connection being busy. This value is set in the the
     * {@value} property.
     */
    public static final String MAX_CONNECT_TIME_KEY =
        "maxConnectTime";

    /**
     * Resumes the client connection automatically when the
     * transmit queue reaches this size (application messages
     * only). This trigger is turned off when set to zero which
     * is the default value. This value should be less than the
     * maximum backlog size to be effective.
     */
    public static final String RESUME_ON_BACKLOG_SIZE_KEY =
        "resumeOnBacklogSize";

    /**
     * {@value} key is used for {@code ConfigException} purposes
     * only.
     */
    private static final String SSL_CONTEXT_KEY = "sslContext";

    /**
     * Array key separator as a character.
     */
    private static final char KEY_IFS = ',';

    /**
     * Array key separator as a string.
     */
    private static final String KEY_SEP = ",";

    //
    // Default values.
    //

    /**
     * Unfortunately, {@link ByteOrder} is <em>not</em> an
     * {@code enum}, so we need to define strings for
     * comparison purposes.
     */
    private static final String BIGENDIAN = "BIG_ENDIAN";

    /**
     * Unfortunately, {@link ByteOrder} is <em>not</em> an
     * {@code enum}, so we need to define strings for
     * comparison purposes.
     */
    private static final String LITTLEENDIAN = "LITTLE_ENDIAN";

    /**
     * The default byte order is {@link ByteOrder#LITTLE_ENDIAN}.
     */
    private static final ByteOrder DEFAULT_BYTE_ORDER =
        ByteOrder.LITTLE_ENDIAN;

    /**
     * The default Dispatcher thread type is
     * {@link ThreadType#BLOCKING}.
     */
    public static final String DEFAULT_THREAD_TYPE =
        (ThreadType.BLOCKING).textName();

    /**
     * The default spin limit is {@value} calls to
     * {@link java.util.Queue#poll()} before parking or yielding.
     */
    public static final int DEFAULT_SPIN_LIMIT = 2_500_000;

    /**
     * The default park time after spinning is {@value}
     * nanoseconds.
     */
    public static final int DEFAULT_PARK_TIME = 1_000;

    /**
     * The default dispatcher thread priority is {@value}.
     */
    public static final int DEFAULT_PRIORITY =
        Thread.NORM_PRIORITY;

    /**
     * The default run quantum is {@value} nanoseconds.
     */
    public static final int DEFAULT_QUANTUM = 500_000;

    /**
     * The default dispatcher thread count is 4.
     */
    public static final int DEFAULT_NUMBER_THREADS = 4;

    /**
     * This value specifies that there is no limit to the
     * transmit queue size.
     */
    public static final int DEFAULT_QUEUE_SIZE = 0;

    /**
     * By default a lost connection is <i>not</i> reconnected.
     */
    public static final boolean DEFAULT_RECONNECT_FLAG = false;

    /**
     * The default reconnect time is 5 seconds. This value is
     * used only if reconnection is set to {@code true} when the
     * connection was
     * {@code ERemoteApp.openConnection(EConfigure.RemoteConnection) opened}.
     */
    public static final long DEFAULT_RECONNECT_TIME = 5000;

    /**
     * By default heart-beating is turned off.
     */
    public static final long DEFAULT_HEARTBEAT_DELAY = 0L;

    /**
     * By default wait indefinitely for a heartbeat reply.
     */
    public static final long DEFAULT_HEARTBEAT_REPLY_DELAY = 0L;

    /**
     * Used to detect that either no port or an invalid port
     * was specified.
     */
    private static final int NO_PORT = -2;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Open local services on the given ports. Maps the server
     * name to the local service.
     */
    private final Map<String, Service> mServices;

    /**
     * Open connections to the specified remote eBus instances.
     * Maps the connection name to the remote connection.
     */
    private final Map<String, RemoteConnection> mRemoteConnections;

    /**
     * The task dispatcher thread configurations.
     */
    private final Map<String, Dispatcher> mDispatchers;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates an eBus configuration for the given services and
     * remote connections. Each local service and remote
     * connection must have a unique name in {@code connections}.
     * @param services local eBus services.
     * @param conns the remote eBus connections.
     * @param dispatchers dispatcher thread configurations.
     * @see RemoteConnection
     * @see Service
     */
    private EConfigure(final Map<String, Service> services,
                       final Map<String, RemoteConnection> conns,
                       final Map<String, Dispatcher> dispatchers)
    {
        mServices = services;
        mRemoteConnections = conns;
        mDispatchers = dispatchers;
    } // end of EConfigure(Map<>, Map<>, Map<>)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns {@code true} if {@code o} is a non-{@code null}
     * {@code EConfigure} instance with the same settings;
     * returns {@code false} otherwise.
     * @param o Test equals with this object.
     * @return {@code true} if {@code o} is a non-{@code null}
     * {@code EConfigure} instance with the same settings;
     * returns {@code false} otherwise.
     */
    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (this == o);

        if (!retcode && o instanceof EConfigure)
        {
            final EConfigure config = (EConfigure) o;

            retcode =
                (mServices.equals(config.mServices) &&
                 mRemoteConnections.equals(
                     config.mRemoteConnections) &&
                 mDispatchers.equals(config.mDispatchers));
        }

        return (retcode);
    } // end of equals(Object)

    /**
     * Returns the configuration hash code.
     * @return the configuration hash code.
     */
    @Override
    public int hashCode()
    {
        return (
            Objects.hash(
                mServices, mRemoteConnections, mDispatchers));
    } // end of hashCode()

    /**
     * Returns the configuration text representation.
     * @return the configuration text representation.
     */
    @Override
    public String toString()
    {
        final Formatter retval = new Formatter();

        retval.format("         services:");
        if (mServices.isEmpty())
        {
            retval.format(" none%n");
        }
        else
        {
            (mServices.values())
                .stream()
                .forEach((service) ->
                {
                    retval.format("%n    %s%n", service);
                });
        }
        retval.format("       connections:");
        if (mRemoteConnections.isEmpty())
        {
            retval.format(" none%n");
        }
        else
        {
            (mRemoteConnections.values())
                .stream()
                .forEach((conn) ->
                {
                    retval.format("%n    %s", conn);
                });
        }

        return (retval.toString());
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns {@code true} if at least one  service is
     * configured and {@code false} otherwise.
     * @return {@code true} if there is a service.
     */
    public boolean hasServices()
    {
        return (!mServices.isEmpty());
    } // end of hasServices()

    /**
     * Returns {@code true} if there is an eBus service defined
     * with the given name.
     * @param name service name.
     * @return {@code true} if service exists.
     */
    public boolean hasService(final String name)
    {
        return (mServices.containsKey(name));
    } // end of hasService(String)

    /**
     * Returns the eBus services map.
     * @return the eBus services map.
     */
    public Map<String, Service> services()
    {
        return (mServices);
    } // end of services()

    /**
     * Returns the eBus service for the given name. Returns
     * {@code null} if there is no service for {@code name}.
     * @param name service name.
     * @return eBus service with the given name.
     */
    public Service service(final String name)
    {
        return (mServices.get(name));
    } // end of service(String)

    /**
     * Returns {@code true} if remote connections are configured
     * and {@code false} if not.
     * @return {@code true} if there are remote connections.
     */
    public boolean hasRemoteConnections()
    {
        return (!mRemoteConnections.isEmpty());
    } // end of hasRemoteConnections()

    /**
     * Returns {@code true} if there is a remote connection
     * defined with the given name.
     * @param name connection name.
     * @return {@code true} if connection exists.
     */
    public boolean hasRemoteConnection(final String name)
    {
        return (mRemoteConnections.containsKey(name));
    } // end of hasRemoteConnection(String)

    /**
     * Returns the set of remote eBus connections.
     * @return the set of remote eBus connections.
     */
    public Map<String, RemoteConnection> remoteConnections()
    {
        return (mRemoteConnections);
    } // end of remoteConnections()

    /**
     * Returns the remote connection configuration with the
     * given name. Returns {@code null} if there is no such
     * named remote connection configuration.
     * @param name remote connection configuration name.
     * @return remote connection configuration instance or
     * {@code null}.
     */
    public RemoteConnection connection(final String name)
    {
        return (mRemoteConnections.get(name));
    } // end of connection(String)

    /**
     * Returns {@code true} if dispatchers are configured and
     * {@code false} if not.
     * @return {@code true} if there are dispatchers configured.
     */
    public boolean hasDispatchers()
    {
        return (!mDispatchers.isEmpty());
    } // end of hasDispatchers()

    /**
     * Returns {@code true} if there is a dispatcher defined
     * with the given name.
     * @param name dispatcher name.
     * @return {@code true} if dispatcher exists.
     */
    public boolean hasDispatcher(final String name)
    {
        return (mDispatchers.containsKey(name));
    } // end of hasDispatcher(String)

    /**
     * Returns the set of client dispatcher configurations.
     * @return client dispatcher configurations.
     */
    public Map<String, Dispatcher> dispatchers()
    {
        return (mDispatchers);
    } // end of dispatchers()

    /**
     * Returns the dispatcher configuration with the given name.
     * Returns {@code null} if there is no such named dispatcher.
     * @param name dispatcher configuration name.
     * @return dispatcher configuration instance or {@code null}.
     */
    public Dispatcher dispatcher(final String name)
    {
        return (mDispatchers.get(name));
    } // end of dispatcher(String)

    //
    // end of Get methods.
    //-----------------------------------------------------------

    /**
     * Creates an eBus configuration for the given service and
     * remote connections. Each remote connection must have a
     * unique name in {@code connections}.
     * @param services the local eBus services.
     * @param connections the remote eBus connections.
     * @param dispatchers dispatcher thread configurations.
     * @return an eBus configuration based on the given
     * parameters.
     * @see RemoteConnection
     * @see Service
     */
    public static EConfigure
        create(final Map<String, Service> services,
               final Map<String, RemoteConnection> connections,
               final Map<String, Dispatcher> dispatchers)
    {
        return (new EConfigure(services,
                               connections,
                               dispatchers));
    } // end of load(Service, Set<>, Set<>)

    /**
     * Returns a new {@code ServerBuilder} instance used to
     * construct an {@code EServer} programmatically.
     * @return new {@code ServerBuilder} instance.
     */
    public static ServerBuilder serverBuilder()
    {
        return (new ServerBuilder());
    } // end of serverBuilder()

    /**
     * Returns a new {@link ConnectionBuilder} instance used to
     * construct a {@code ERemoteApp} programmatically.
     * @return new {@code ConnectionBuilder} instance.
     */
    public static ConnectionBuilder connectionBuilder()
    {
        return (new ConnectionBuilder());
    } // end of connectionBuilder()

    /**
     * Returns a new {@link DispatcherBuilder} instance used to
     * construct a {@link Dispatcher}.
     * @return new {@code DispatcherBuilder} instance.
     */
    public static DispatcherBuilder dispatcherBuilder()
    {
        return (new DispatcherBuilder());
    } // end of dispatcherBuilder()

    /**
     * Returns a new {@link PauseBuilder} instance used to
     * construct a {@link PauseConfig}.
     * @param role build pause configuration for this connection
     * role.
     * @return new {@link PauseBuilder} instance.
     */
    public static PauseBuilder pauseBuilder(final ConnectionRole role)
    {
        return (new PauseBuilder(role));
    } // end of pauseBuilder(ConnectionRole)

    /**
     * Returns the eBus service, remote connection, and
     * dispatcher configurations found in the given properties.
     * There are four top-level keys: {@code net},
     * {@code eBus.messageLoaders}, optional {@code eBus.server}
     * and {@code ebus.remoteConnections}.
     * Both keys have a comma-separated list of unique
     * names as the value. Each name is used to extract one
     * server or remote connection configuration.
     * <p>
     * <b>Note:</b> Each name must be unique within the remote
     * connection list.
     * </p>
     * <p>
     * The optional server configuration keys are:
     * </p>
     * <ul>
     *   <li>
     *     {@code eBus.service.port}: The TCP
     *     service port integer. Must be &gt; zero and &lt; 65,535.
     *   </li>
     *   <li>
     *     {@code eBus.service.socketBufferSize}:
     *     The accepted connection's output buffer size.
     *   </li>
     * </ul>
     * <p>
     * The remote configuration keys are:
     * </p>
     * <ul>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .host}:
     *     The remote eBus host name or dotted notation IP
     *     address.
     *   </li>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .port}: The
     *     remote eBus TCP service port.
     *   </li>
     *   <li>
     *     {@code eBus.connection}.<i>name</i>{@code .bindPort}:
     *     the connection's local end is bound to this TCP port.
     *     If this key is unspecified or zero, then the local
     *     end is bound to any available TCP port.
     *   </li>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .socketBufferSize}:
     *     The maximum number of outbound bytes that may be
     *     waiting for transmission at a time.
     *   </li>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .reconnect}:
     *     Either {@code true} of {@code false}. If {@code true},
     *     then a lost connection will be re-established.
     *   </li>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .reconnectTime}:
     *     When reconnecting, wait this many milliseconds between
     *     connect attempts.
     *   </li>
     * </ul>
     * <p>
     * The dispatcher configuration keys are:
     * </p>
     * <ul>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .runQueueType}:
     *     the dispatcher uses either a blocking, spinning, or
     *     spin+park run queue.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .isDefault}:
     *     marks this as the default run queue used by all eBus
     *     clients not explicitly assigned a run queue. There
     *     should be only one dispatcher marked as default.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .priority}:
     *     dispatcher threads are all assigned this priority.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .quantum}:
     *     the maximum run quantum assigned to clients.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .classes}:
     *     Instances of these client classes are assigned to this
     *     dispatcher. A class should appear in at most one
     *     dispatcher classes list.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .numberThreads}:
     *     The number of threads in this dispatcher.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .spinLimit}:
     *     If {@code runQueueType} is spin+park, then this is the
     *     spin limit.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .parkTime}:
     *     If {@code runQueueType} is spin+park, then this is the
     *     nanosecond park time.
     *   </li>
     * </ul>
     * @param props extract the eBus configuration information
     * from these properties.
     * @return eBus configuration settings.
     * @throws ConfigException
     * if a required property is missing or contains an invalid
     * value.
     */
    public static EConfigure load(final java.util.Properties props)
    {
        return (new EConfigure(loadServices(props),
                               loadConnections(props),
                               loadDispatchers(props)));
    } // end of load(Properties)

    /**
     * Extracts the services from the properties, returning the
     * service name to configuration map.
     * @param props eBus configuration properties.
     * @return the eBus TCP services map.
     * @throws ConfigException
     * if a service property is missing or contains an invalid
     * value.
     */
    public static Map<String, Service> loadServices(final java.util.Properties props)
    {
        final net.sf.eBus.util.Properties p = properties(props);
        final String[] names =
            p.getArrayProperty(
                EBUS_KEY + "." + SERVICES_KEY, KEY_IFS);
        ServerBuilder builder;
        String keyPrefix;
        String key;
        final Map<String, Service> retval = new HashMap<>();

        for (String name: names)
        {
            keyPrefix = SERVICE_PREFIX + name + ".";
            builder = new ServerBuilder();

            builder.name(name);

            key = keyPrefix + CONN_TYPE_KEY;
            builder.connectionType(getConnectionType(key,
                                                     ConnectionType.TCP,
                                                     p));

            key = keyPrefix + PORT_KEY;
            builder.port(getPort(key, NO_PORT, p));

            key = keyPrefix + FILTER_KEY;
            builder.addressFilter(getFilter(key, p));

            key = keyPrefix +  INBUFFER_SIZE_KEY;
            builder.inputBufferSize(getSize(key,
                                            DEFAULT_BUFFER_SIZE,
                                            0,
                                            p));

            key = keyPrefix +  OUTBUFFER_SIZE_KEY;
            builder.outputBufferSize(getSize(key,
                                             DEFAULT_BUFFER_SIZE,
                                             0,
                                             p));

            key = keyPrefix + BYTE_ORDER_KEY;
            builder.byteOrder(
                getByteOrder(key,
                             DEFAULT_BYTE_ORDER,
                             p));

            key = keyPrefix + MSG_QUEUE_SIZE_KEY;
            builder.messageQueueSize(getSize(key,
                                             DEFAULT_QUEUE_SIZE,
                                             0,
                                             p));

            key = keyPrefix + SVC_SELECTOR_KEY;
            builder.serviceSelector(p.getProperty(key,
                    (ENetConfigure.defaultSelector()).name()));

            key = keyPrefix + CONN_SELECTOR_KEY;
            builder.connectionSelector(p.getProperty(key,
                    (ENetConfigure.defaultSelector()).name()));

            key = keyPrefix + HB_DELAY_KEY;
            builder.heartbeatDelay(p.getIntProperty(key, 0));

            key = keyPrefix + HB_REPLY_DELAY_KEY;
            builder.heartbeatReplyDelay(p.getIntProperty(key, 0));

            key = keyPrefix + CAN_PAUSE_KEY;
            builder.canPause(p.getBooleanProperty(key, false));

            // Do the accepted connections support pausing?
            if (builder.mCanPause)
            {
                final PauseBuilder pauseBuilder =
                    new PauseBuilder(ConnectionRole.ACCEPTOR);

                // Create the pause prefix.
                keyPrefix = keyPrefix + PAUSE_KEY + ".";

                key = keyPrefix + PAUSE_DURATION_KEY;
                pauseBuilder.duration(
                    Duration.ofMillis(p.getIntProperty(key)));

                key = keyPrefix + MAX_BACKLOG_SIZE_KEY;
                pauseBuilder.maxBacklogSize(
                    p.getIntProperty(key));

                builder.pauseConfig(pauseBuilder.build());
            }

            retval.put(name, builder.build());
        }

        return (retval);
    } // end of loadServices(Properties)

    /**
     * Returns the remote eBus connections found in the
     * properties. If there are no remote connections then
     * returns an empty set.
     * @param props the eBus configuration properties.
     * @return the remote eBus connections.
     * @throws ConfigException
     * if there are properties missing for a specified remote
     * connection or a property contains an invalid value.
     */
    public static Map<String, RemoteConnection>
        loadConnections(final java.util.Properties props)
    {
        final net.sf.eBus.util.Properties p = properties(props);
        final String[] names =
            p.getArrayProperty(
                EBUS_KEY + "." + CONNECTIONS_KEY, KEY_IFS);
        ConnectionBuilder builder;
        String keyPrefix;
        String key;
        final Map<String, RemoteConnection> retval =
            new HashMap<>();

        for (String name: names)
        {
            keyPrefix = CONNECTION_PREFIX + name + ".";
            builder = new ConnectionBuilder();

            builder.name(name)
                   .connectionType(
                       getConnectionType(
                           keyPrefix + CONN_TYPE_KEY,
                           ConnectionType.TCP,
                           p))
                   .address(getAddress(name, keyPrefix, p))
                   .bindPort(
                       getPort(keyPrefix + BIND_PORT_KEY,
                               ANY_PORT,
                               p))
                   .inputBufferSize(
                       getSize(keyPrefix + INBUFFER_SIZE_KEY,
                               DEFAULT_BUFFER_SIZE,
                               0,
                               p))
                   .outputBufferSize(
                       getSize(keyPrefix + OUTBUFFER_SIZE_KEY,
                               DEFAULT_BUFFER_SIZE,
                               0,
                               p))
                   .byteOrder(
                       getByteOrder(keyPrefix + BYTE_ORDER_KEY,
                                    DEFAULT_BYTE_ORDER,
                                    p))
                   .messageQueueSize(
                       getSize(keyPrefix + MSG_QUEUE_SIZE_KEY,
                               DEFAULT_QUEUE_SIZE,
                               0,
                               p))
                   .selector(p.getProperty(keyPrefix + SELECTOR_KEY,
                           (ENetConfigure.defaultSelector()).name()));

            key = keyPrefix + RECONNECT_KEY;
            builder.reconnect(p.getBooleanProperty(key, false));
            if (!builder.mReconnectFlag)
            {
                builder.reconnectDelay(0L);
            }
            else
            {
                key = keyPrefix + RECONNECT_DELAY_KEY;
                builder.reconnectDelay(
                    getDelay(
                        key,
                        (int) DEFAULT_RECONNECT_TIME,
                        0L,
                        p));
            }

            builder.heartbeatDelay(
                p.getIntProperty(
                    keyPrefix + HB_DELAY_KEY,
                    (int) DEFAULT_HEARTBEAT_DELAY))
                   .heartbeatReplyDelay(
                p.getIntProperty(
                    keyPrefix + HB_REPLY_DELAY_KEY,
                    (int) DEFAULT_HEARTBEAT_REPLY_DELAY));

            key = keyPrefix + CAN_PAUSE_KEY;
            builder.canPause(p.getBooleanProperty(key, false));

            if (builder.mCanPause)
            {
                final PauseBuilder pauseBuilder =
                    new PauseBuilder(ConnectionRole.INITIATOR);

                keyPrefix = keyPrefix + PAUSE_KEY + ".";
                key = keyPrefix + PAUSE_DURATION_KEY;
                pauseBuilder.duration(
                    Duration.ofMillis(p.getIntProperty(key)));

                key = keyPrefix + MAX_BACKLOG_SIZE_KEY;
                pauseBuilder.maxBacklogSize(p.getIntProperty(key));

                key = keyPrefix + DISCARD_POLICY_KEY;
                pauseBuilder.discardPolicy(
                    DiscardPolicy.valueOf(
                        p.getProperty(
                            key,
                            (DiscardPolicy.OLDEST_FIRST).name())));

                key = keyPrefix + IDLE_TIME_KEY;
                pauseBuilder.idleTime(
                    Duration.ofMillis(p.getIntProperty(key, 0)));

                key = keyPrefix + MAX_CONNECT_TIME_KEY;
                pauseBuilder.maxConnectionTime(
                    Duration.ofMillis(p.getIntProperty(key)));

                key = keyPrefix + RESUME_ON_BACKLOG_SIZE_KEY;
                pauseBuilder.resumeOnBacklogSize(
                    p.getIntProperty(key, 0));

                builder.pauseConfig(pauseBuilder.build());
            }

            retval.put(name, builder.build());
        }

        return (retval);
    } // end of loadConnections(Properties)

    /**
     * Returns the eBus dispatcher configuration found in the
     * given properties. If this is no dispatcher configuration
     * found in the properties, then returns an empty map.
     * @param props eBus configuration properties.
     * @return eBus dispatcher configurations.
     * @throws ConfigException
     * if there are properties missing for a specified remote
     * connection or a property is invalid.
     */
    public static Map<String, Dispatcher>
        loadDispatchers(final java.util.Properties props)
    {
        final net.sf.eBus.util.Properties p = properties(props);
        final String key = EBUS_KEY + "." + DISPATCHERS_KEY;
        final String[] names = p.getArrayProperty(key, KEY_IFS);
        DispatcherBuilder builder;
        final Map<String, Dispatcher> retval = new HashMap<>();

        for (String name : names)
        {
            builder = new DispatcherBuilder();
            builder.name(name);
            builder.dispatcherType(
                DispatcherType.findType(name));

            // If this dispatcher type is special, then only
            // certain properties apply.
            if ((builder.mType).isSpecial())
            {
                loadSpecialDispatcher(builder, p);
            }
            else
            {
                loadDispatcher(builder, p);
            }

            retval.put(name, builder.build());
        }

        return (retval);
    } // end of loadDispatchers(Properties)

    /**
     * Returns an eBus configuration extracted from the
     * preferences. There are four top-level keys: {@code net},
     * {@code eBus.messageLoaders}, optional {@code eBus.server}
     * and {@code ebus.remoteConnections}.
     * Both keys have a comma-separated list of unique
     * names as the value. Each name is used to extract one
     * server or remote connection configuration.
     * <p>
     * <b>Note:</b> Each name must be unique with the remote
     * remote connection list.
     * </p>
     * <p>
     * The optional server configuration keys are:
     * </p>
     * <ul>
     *   <li>
     *     {@code eBus.service.port}: The TCP
     *     service port integer. Must be &gt; zero and &lt; 65,535.
     *   </li>
     *   <li>
     *     {@code eBus.service.socketBufferSize}:
     *     The accepted connection's output buffer size.
     *   </li>
     * </ul>
     * <p>
     * The remote configuration keys are:
     * </p>
     * <ul>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .host}:
     *     The remote eBus host name or dotted notation IP
     *     address.
     *   </li>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .port}: The
     *     remote eBus TCP service port.
     *   </li>
     *   <li>
     *     {@code eBus.connection}.<i>name</i>{@code .bindPort}:
     *     the connection's local end is bound to this TCP port.
     *     If this key is unspecified or zero, then the local
     *     end is bound to any available TCP port.
     *   </li>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .socketBufferSize}:
     *     The maximum number of outbound bytes that may be
     *     waiting for transmission at a time.
     *   </li>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .reconnect}:
     *     Either {@code true} of {@code false}. If {@code true},
     *     then a lost connection will be re-established.
     *   </li>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .reconnectTime}:
     *     When reconnecting, wait this many milliseconds between
     *     connect attempts.
     *   </li>
     * </ul>
     * <p>
     * The dispatcher configuration keys are:
     * </p>
     * <ul>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .runQueueType}:
     *     the dispatcher uses either a blocking, spinning, or
     *     spin+park run queue.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .isDefault}:
     *     marks this as the default run queue used by all eBus
     *     clients not explicitly assigned a run queue. There
     *     should be only one dispatcher marked as default.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .priority}:
     *     dispatcher threads are all assigned this priority.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .quantum}:
     *     the maximum run quantum assigned to clients.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .classes}:
     *     Instances of these client classes are assigned to this
     *     dispatcher. A class should appear in at most one
     *     dispatcher classes list.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .numberThreads}:
     *     The number of threads in this dispatcher.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .spinLimit}:
     *     If {@code runQueueType} is spin+park, then this is the
     *     spin limit.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .parkTime}:
     *     If {@code runQueueType} is spin+park, then this is the
     *     nanosecond park time.
     *   </li>
     * </ul>
     * @param prefs Read in the eBus configuration from these
     * preferences.
     * @return an eBus configuration loaded from the preferences.
     * @exception MissingResourceException
     * if a required property is missing.
     * @throws IllegalStateException
     * if a property contains an invalid value.
     * @deprecated Preferences-based configuration will be
     * removed at a future date. Please consider moving to
     * the JSON-based configuration file supported by
     * {@link #load(com.typesafe.config.Config)}.
     *
     * @see #load(Properties)
     * @see #load(com.typesafe.config.Config)
     */
    @Deprecated
    public static EConfigure load(final Preferences prefs)
    {
        return (new EConfigure(loadServices(prefs),
                               loadConnections(prefs),
                               loadDispatchers(prefs)));
    } // end of load(Preferences)

    /**
     * Returns the eBus TCP services. Returns an empty map if
     * there are no eBus services.
     * @param p eBus configuration preferences.
     * @return the eBus TCP service.
     * @throws MissingResourceException
     * if a service property is missing.
     * @throws IllegalStateException
     * if a service property is invalid.
     * @deprecated Preferences-based configuration will be
     * removed at a future date. Please consider moving to
     * the JSON-based configuration file supported by
     * {@link #load(com.typesafe.config.Config)}.
     *
     * @see #load(Preferences)
     */
    @Deprecated
    public static Map<String, Service> loadServices(final Preferences p)
    {
        final String[] names =
            (p.get(EBUS_KEY + "." + SERVICES_KEY, "")).split(",");
        ServerBuilder builder;
        String keyPrefix;
        String key;
        final Map<String, Service> retval = new HashMap<>();

        for (String name: names)
        {
            keyPrefix = SERVICE_PREFIX + name + ".";
            builder = new ServerBuilder();

            builder.name(name);

            key = keyPrefix + CONN_TYPE_KEY;
            builder.connectionType(
                getConnectionType(key,
                                  ConnectionType.TCP,
                                  p));

            key = keyPrefix + PORT_KEY;
            builder.port(getPort(key, 0, p));

            key = keyPrefix + FILTER_KEY;
            builder.addressFilter(getFilter(key, p));

            key = keyPrefix +  INBUFFER_SIZE_KEY;
            builder.inputBufferSize(
                getSize(key,
                        DEFAULT_BUFFER_SIZE,
                        0,
                        p));

            key = keyPrefix + OUTBUFFER_SIZE_KEY;
            builder.outputBufferSize(
                getSize(key,
                        DEFAULT_BUFFER_SIZE,
                        0,
                        p));

            key = keyPrefix + BYTE_ORDER_KEY;
            builder.byteOrder(
                getByteOrder(
                    key, DEFAULT_BYTE_ORDER, p));

            key = keyPrefix + MSG_QUEUE_SIZE_KEY;
            builder.messageQueueSize(
                getSize(key,
                        DEFAULT_QUEUE_SIZE,
                        0,
                        p));

            key = keyPrefix + SVC_SELECTOR_KEY;
            builder.serviceSelector(
                p.get(key,
                      (ENetConfigure.defaultSelector()).name()));

            key = keyPrefix + SELECTOR_KEY;
            builder.connectionSelector(
                p.get(key,
                      (ENetConfigure.defaultSelector()).name()));

            key = keyPrefix + HB_DELAY_KEY;
            builder.heartbeatDelay(p.getLong(key, 0L));

            key = keyPrefix + HB_REPLY_DELAY_KEY;
            builder.heartbeatReplyDelay(p.getLong(key, 0L));

            retval.put(name, builder.build());
        }

        return (retval);
    } // end of loadServices(Preferences)

    /**
     * Returns the remote eBus connections found in the
     * preferences. If there are no remote connections then
     * returns an empty set.
     * @param p the eBus configuration preferences.
     * @return the remote eBus connections.
     * @throws MissingResourceException
     * if there are properties missing for a specified remote
     * connection.
     * @throws IllegalStateException
     * if a property is invalid.
     * @deprecated Preferences-based configuration will be
     * removed at a future date. Please consider moving to
     * the JSON-based configuration file supported by
     * {@link #load(com.typesafe.config.Config)}.
     *
     * @see #load(Preferences)
     */
    @Deprecated
    public static Map<String, RemoteConnection>
        loadConnections(final Preferences p)
    {
        final String[] names =
            (p.get(EBUS_KEY + "." + CONNECTIONS_KEY, "")).split(",");
        String keyPrefix;
        ConnectionBuilder builder;
        String key;
        final Map<String, RemoteConnection> retval =
            new HashMap<>();

        for (String name: names)
        {
            keyPrefix = CONNECTION_PREFIX + name + ".";
            builder = new ConnectionBuilder();

            builder.name(name);

            key = keyPrefix + CONN_TYPE_KEY;
            builder.connectionType(
                        getConnectionType(
                            key, ConnectionType.TCP, p))
                   .address(getAddress(keyPrefix, p));

            key = keyPrefix + BIND_PORT_KEY;
            builder.bindPort(
                getPort(key, ANY_PORT, p));

            // The output buffer size property is not required.
            builder.inputBufferSize(
                getSize(keyPrefix + INBUFFER_SIZE_KEY,
                        DEFAULT_BUFFER_SIZE,
                        0,
                        p));
            builder.outputBufferSize(
                getSize(keyPrefix + OUTBUFFER_SIZE_KEY,
                        DEFAULT_BUFFER_SIZE,
                        0,
                        p));
            builder.byteOrder(
                getByteOrder(keyPrefix + BYTE_ORDER_KEY,
                             DEFAULT_BYTE_ORDER,
                             p));
            builder.messageQueueSize(
                getSize(keyPrefix + MSG_QUEUE_SIZE_KEY,
                        DEFAULT_QUEUE_SIZE,
                        0,
                        p));
            builder.selector(
                p.get(keyPrefix + SELECTOR_KEY,
                      (ENetConfigure.defaultSelector()).name()));

            key = keyPrefix + RECONNECT_KEY;
            builder.reconnect(p.getBoolean(key, false));
            if (!builder.mReconnectFlag)
            {
                builder.reconnectDelay(0L);
            }
            else
            {
                key = keyPrefix + RECONNECT_DELAY_KEY;
                builder.reconnectDelay(
                    getDelay(
                        key,
                        (int) DEFAULT_RECONNECT_TIME,
                        0L,
                        p));
            }

            key = keyPrefix + HB_DELAY_KEY;
            builder.heartbeatDelay(
                p.getLong(
                    key,
                    DEFAULT_HEARTBEAT_DELAY));

            key = keyPrefix + HB_REPLY_DELAY_KEY;
            builder.heartbeatReplyDelay(
                p.getLong(
                    key,
                    DEFAULT_HEARTBEAT_REPLY_DELAY));

            retval.put(name, builder.build());
        }

        return (retval);
    } // end of loadConnections()

    /**
     * Returns the eBus dispatcher configuration found in the
     * given preferences. If this is no dispatcher configuration
     * found in the preferences, then returns an empty set.
     * @param p the eBus configuration preferences.
     * @return eBus dispatcher configurations.
     * @throws MissingResourceException
     * if there are properties missing for a specified remote
     * connection.
     * @deprecated Preferences-based configuration will be
     * removed at a future date. Please consider moving to
     * the JSON-based configuration file supported by
     * {@link #load(com.typesafe.config.Config)}.
     *
     * @see #load(Preferences)
     */
    @Deprecated
    public static Map<String, Dispatcher> loadDispatchers(final Preferences p)
    {
        final String key = EBUS_KEY + "." + DISPATCHERS_KEY;
        final String[] names = (p.get(key, "")).split(",");
        DispatcherBuilder builder;
        final Map<String, Dispatcher> retval = new HashMap<>();

        for (String name : names)
        {
            builder = new DispatcherBuilder();
            builder.name(name);
            builder.dispatcherType(
                DispatcherType.findType(name));

            // If this dispatcher type is special, then only
            // certain properties apply.
            if ((builder.mType).isSpecial())
            {
                loadSpecialDispatcher(builder, p);
            }
            else
            {
                loadDispatcher(builder, p);
            }

            retval.put(name, builder.build());
        }

        return (retval);
    } // end of loadDispatchers(Preferences)

    /**
     * Returns the eBus service, remote connection, and
     * dispatcher configurations found in the given JSON
     * configuration.
     * <p>
     * An example of the eBus JSON configuration:
     * </p>
     * <p>
     * <b>Note:</b> Each name must be unique within the selectors,
     * services, and connections lists.
     * </p>
     * <pre><code>"selectors" : [
    {
        "name" : "mdSelector"
        "type" : "spinning"
        "isDefault" : "false"
        "priority" : 9
    },
    {
        "name" : "stockSelector"
        "type" : "spin+park"
        "isDefault" : "false"
        "priority" : 6
        "spinLimit" : 3000000
        "parkTime" : 1000
    }
]

"services" : [
     {
        "name" : "stockService"
        "port" : 12345
        "addressFilter" : [
            "198.168.3.2",
            "198.168.3.3:55001"
        ]
        "serviceSelector" : "stockSelector"
        "connectionSelector" : "stockSelector"
        "byteOrder" : "BIG_ENDIAN"
        "inputBufferSize" : 8192
        "outputBufferSize" : 65536
        "messageQueueSize" : 100
        "heartbeatDelay" : 30000
        "heartbeatReply" : 500
        "canPause" : false
    }
]

"connections" : [
     {
         "name" : "marketData"
         "host" : "192.168.3.4"
         "port" : 10000
         "inputBufferSize" : 65536
         "outputBufferSize" : 8192
         "byteOrder" : "BIG_ENDIAN"
         "messageQueueSize" : 0
         "selector" : "mdSelector"
         "reconnect" : true
         "reconnectTime" : 500
         "canPause" : false
     },
     {
        "name" : "stockOrders"
        "host" : "StocksRUs.com"
        "port" : 10001
        "bindPort" : 55000
        "inputBufferSize" : 8192
        "outputBufferSize" : 65536
        "byteOrder" : "BIG_ENDIAN"
        "messageQueueSize" : 10
        "selector" : "stockSelector"
        "reconnect" : true
        "reconnectTime" : 500
        "canPause" : false
    }
]</code></pre>
     * @param config extract the eBus configuration from these
     * JSON properties.
     * @return eBus configuration settings.
     * @throws ConfigException
     * if {@code config} contains an invalid or incomplete eBus
     * configuration.
     */
    public static EConfigure load(final Config config)
    {
        return (new EConfigure(loadServices(config),
                               loadConnections(config),
                               loadDispatchers(config)));
    } // end of load(Config)

    /**
     * Returns the eBus TCP services. Returns an empty map if
     * there are no eBus services.
     * @param config {@code typesafe.config} HOCON configuration
     * containing service definition.
     * @return eBus service.
     * @throws ConfigException
     * if a service definition is invalid.
     *
     * @see #load(Config)
     */
    public static Map<String, Service> loadServices(final Config config)
    {
        Service service;
        final Map<String, Service> retval = new HashMap<>();

        // Are there any defined services?
        if (config.hasPath(SERVICES_KEY))
        {
            // Yes. Load each service in turn.
            for (ConfigObject co :
                    config.getObjectList(SERVICES_KEY))
            {
                service = loadService(co.toConfig());
                retval.put(service.name(), service);
            }
        }
        // Else return an empty hash map.

        return (retval);
    } // end of loadServices(Config)

    /**
     * Returns the remote eBus connections found in the
     * preferences. If there are no remote connections then
     * returns an empty set.
     * @param config {@code typesafe.config} HOCON configuration
     * containing connection definition.
     * @return the remote eBus connections.
     * @throws ConfigException
     * if a remote connection definition is invalid.
     *
     * @see #load(Preferences)
     */
    public static Map<String, RemoteConnection>
        loadConnections(final Config config)
    {
        RemoteConnection connection;
        final Map<String, RemoteConnection> retval =
            new HashMap<>();

        // Are there any connections to load?
        if (config.hasPath(CONNECTIONS_KEY))
        {
            // Yes. Load each connection in turn.
            for (ConfigObject co : config.getObjectList(CONNECTIONS_KEY))
            {
                connection = loadConnection(co.toConfig());
                retval.put(connection.name(), connection);
            }
        }

        return (retval);
    } // end of loadConnections(Config)

    /**
     * Returns the eBus dispatcher configuration found in the
     * given JSON configuration. If this is no dispatcher
     * configuration found, then returns an empty map.
     * @param config JSON configuration.
     * @return map from dispatcher name to its configuration.
     * @throws ConfigException
     * if {@code config} contains an invalid dispatcher
     * configuration.
     */
    public static Map<String, Dispatcher>
        loadDispatchers(final Config config)
    {
        Dispatcher dispatcher;
        final Map<String, Dispatcher> retval = new HashMap<>();

        // Are there any dispatchers to load?
        if (config.hasPath(DISPATCHERS_KEY))
        {
            // Yes. Load each dispatcher in turn.
            for (ConfigObject co : config.getObjectList(DISPATCHERS_KEY))
            {
                dispatcher = loadDispatcher(co.toConfig());
                retval.put(dispatcher.name(), dispatcher);
            }
        }

        return (retval);
    } // end of loadDispatchers(Config)

    /**
     * Stores the eBus configuration to the given properties.
     * @param props store the configuration to this properties
     * map.
     * @see EConfigure#load(java.util.Properties)
     * @see EConfigure#load(java.util.prefs.Preferences)
     * @see EConfigure#store(java.util.prefs.Preferences)
     */
    public void store(final java.util.Properties props)
    {
        storeService(props);
        storeConnections(props);
        storeDispatchers(props);
    } // end of store(Properties)

    /**
     * Stores the eBus configuration to the given preferences.
     * @param prefs store the configuration to these preferences.
     *
     * @see EConfigure#load(Preferences)
     * @see EConfigure#load(java.util.Properties)
     * @see EConfigure#store(java.util.Properties)
     */
    @Deprecated
    public void store(final Preferences prefs)
    {
        storeService(prefs);
        storeConnections(prefs);
        storeDispatchers(prefs);
    } // end of store(Preferences)

    /**
     * Returns the given {@link java.util.Properties} as
     * {@link net.sf.eBus.util.Properties}. If {@code props} is
     * an eBus properties, then performs a typecast. Otherwise,
     * creates a new eBus properties instance based on the the
     * Java properties.
     * @param props convert to eBus properties.
     * @return eBus properties.
     */
    private static net.sf.eBus.util.Properties properties(final java.util.Properties props)
    {
        net.sf.eBus.util.Properties retval;

        if (props instanceof net.sf.eBus.util.Properties)
        {
            retval = (net.sf.eBus.util.Properties) props;
        }
        else
        {
            retval = new net.sf.eBus.util.Properties(props);
        }

        return (retval);
    } // end of properties(Properties)

    /**
     * Returns the named {@link DispatcherType#EBUS eBus}
     * dispatcher loaded in from the given properties.
     * @param name eBus dispatcher name.
     * @param p properties containing the dispatcher
     * configuration.
     * @return eBus dispatcher.
     * @throws MissingResourceException
     * if {@code p} is missing a property required for the
     * eBus dispatcher definition.
     * @throws IllegalStateException
     * if there are unsupported keys for this special dispatcher.
     */
    private static void loadDispatcher(final DispatcherBuilder builder,
                                       final net.sf.eBus.util.Properties p)
    {
        String keyName =
            EBUS_KEY + "." + DISPATCHER_PREFIX + builder.mName + ".";
        String key;

        key = keyName + THREAD_TYPE_KEY;
        builder.threadType(
            ThreadType.find(
                p.getProperty(key, DEFAULT_THREAD_TYPE)));

        if (builder.mRunQueueType == ThreadType.SPINPARK ||
            builder.mRunQueueType == ThreadType.SPINYIELD)
        {
            key = keyName + SPIN_LIMIT_KEY;
            builder.spinLimit(
                p.getIntProperty(key, DEFAULT_SPIN_LIMIT));
        }

        if (builder.mRunQueueType == ThreadType.SPINPARK)
        {
            key = keyName + PARK_TIME_KEY;
            builder.parkTime(
                p.getIntProperty(key, DEFAULT_PARK_TIME));
        }

        key = keyName + PRIORITY_KEY;
        builder.priority(p.getIntProperty(key, DEFAULT_PRIORITY));

        key = keyName + QUANTUM_KEY;
        builder.quantum(p.getIntProperty(key, DEFAULT_QUANTUM));

        key = keyName + NUM_THREADS_KEY;
        builder.numberThreads(
            p.getIntProperty(key, DEFAULT_NUMBER_THREADS));

        key = keyName + DEFAULT_KEY;
        builder.isDefault(p.getBooleanProperty(key, false));

        // Is this the default dispatcher?
        if (builder.mIsDefault)
        {
            // Yes. Then use an empty class array.
            builder.classes(new Class<?>[0]);
        }
        // No. Read in the class names and create the class
        // array from that.
        else
        {
            key = keyName + CLASSES_KEY;
            builder.classes(getClasses(key, p));
        }
    } // end of loadDispatcher(String, Properties)

    /**
     * Returns the named special dispatcher loaded from the given
     * properties. Special dispatchers only use the "isDefault"
     * and "classes" attributes.
     * @param builder place special dispatcher configuration into
     * this builder.
     * @param p read in dispatcher configuration from here.
     * @return special dispatcher.
     * @throws MissingResourceException
     * if {@code p} is missing a property required for the
     * special dispatcher definition.
     */
    private static void loadSpecialDispatcher(final DispatcherBuilder builder,
                                              final net.sf.eBus.util.Properties p)
    {
        String keyName =
            EBUS_KEY + "." + DISPATCHER_PREFIX + builder.mName + ".";
        String key;

        key = keyName + DEFAULT_KEY;
        builder.isDefault(p.getBooleanProperty(key, false));

        // Is this the default dispatcher?
        if (builder.mIsDefault)
        {
            // Yes. Then use an empty class array.
            builder.classes(new Class<?>[0]);
        }
        // No. Read in the class names and create the class
        // array from that.
        else
        {
            key = keyName + CLASSES_KEY;
            builder.classes(getClasses(key, p));
        }

        // Set the priority, quantum, and numThreads to zero
        // since special Dispatcher threads are 3rd party and
        // not configured by eBus. Uses a blocking thread type.
        builder.threadType(ThreadType.BLOCKING)
               .spinLimit(0L)
               .parkTime(0L)
               .priority(0)
               .quantum(0L)
               .numberThreads(0);
    } // end of loadSpecialDispatcher(...)

    /**
     * Returns the named {@link DispatcherType#EBUS eBus}
     * dispatcher loaded in from the given preferences.
     * @param builder place dispatcher configuration into this
     * builder.
     * @param p preferences containing the dispatcher
     * configuration.
     * @return eBus dispatcher.
     * @throws MissingResourceException
     * if {@code p} is missing a key required for the
     * eBus dispatcher definition.
     */
    private static void loadDispatcher(final DispatcherBuilder builder,
                                       final Preferences p)
    {
        String keyName =
            EBUS_KEY + "." + DISPATCHER_PREFIX + builder.mName + ".";
        String key;

        key = keyName + THREAD_TYPE_KEY;
        builder.threadType(
            ThreadType.find(p.get(key, DEFAULT_THREAD_TYPE)));

        if (builder.mRunQueueType == ThreadType.SPINPARK ||
            builder.mRunQueueType == ThreadType.SPINYIELD)
        {
            key = keyName + SPIN_LIMIT_KEY;
            builder.spinLimit(p.getInt(key, DEFAULT_SPIN_LIMIT));
        }

        if (builder.mRunQueueType == ThreadType.SPINPARK)
        {
            key = keyName + PARK_TIME_KEY;
            builder.parkTime(p.getInt(key, DEFAULT_PARK_TIME));
        }

        key = keyName + PRIORITY_KEY;
        builder.priority(p.getInt(key, DEFAULT_PRIORITY));

        key = keyName + QUANTUM_KEY;
        builder.quantum(p.getInt(key, DEFAULT_QUANTUM));

        key = keyName + NUM_THREADS_KEY;
        builder.numberThreads(
            p.getInt(key, DEFAULT_NUMBER_THREADS));

        key = keyName + DEFAULT_KEY;
        builder.isDefault(p.getBoolean(key, false));

        // Is this the default dispatcher?
        if (builder.mIsDefault)
        {
            // Yes. Then use an empty class array.
            builder.classes(new Class<?>[0]);
        }
        // No. Read in the class names and create the class
        // array from that.
        else
        {
            builder.classes(getClasses(key, p));
        }
    } // end of loadDispatcher(DispatcherBuilder, Preferences)

    /**
     * Returns the named special dispatcher loaded from the given
     * preferences. Special dispatchers only use the ".isDefault"
     * and ".classes" attributes.
     * @param name special dispatcher name.
     * @param dType special dispatcher type.
     * @param p read in dispatcher configuration from here.
     * @return special dispatcher.
     * @throws MissingResourceException
     * if {@code p} is missing a property required for the
     * special dispatcher definition.
     */
    private static void loadSpecialDispatcher(final DispatcherBuilder builder,
                                              final Preferences p)
    {
        String keyName =
            EBUS_KEY + "." + DISPATCHER_PREFIX + builder.mName + ".";
        String key;

        key = keyName + DEFAULT_KEY;
        builder.isDefault(p.getBoolean(key, false));

        // Is this the default dispatcher?
        if (builder.mIsDefault)
        {
            // Yes. Then use an empty class array.
            builder.classes(new Class<?>[0]);
        }
        // No. Read in the class names and create the class
        // array from that.
        else
        {
            builder.classes(getClasses(key, p));
        }

        // Set the priority, quantum, and numThreads to zero
        // since special Dispatcher threads are 3rd party and
        // not configured by eBus. Use a blocking thread type.
        builder.threadType(ThreadType.BLOCKING)
               .spinLimit(0L)
               .parkTime(0L)
               .priority(0)
               .quantum(0)
               .numberThreads(0);
    } // end of loadSpecialDispatcher(...)

    /**
     * Returns a {@link Service} instance based on the given
     * JSON configuration.
     * @param config JSON configuration.
     * @return eBus service instance.
     * @throws ConfigException
     * if {@code config} is either missing required settings or
     * existing settings are invalid.
     */
    private static Service loadService(final Config config)
        throws ConfigException
    {
        final ConnectionType connType =
            (config.hasPath(CONN_TYPE_KEY) ?
             config.getEnum(ConnectionType.class, CONN_TYPE_KEY) :
             ConnectionType.TCP);
        final ServerBuilder builder = new ServerBuilder();

        builder.loaderFlag(true)
               .name(config.getString(NAME_KEY))
               .connectionType(connType)
               .port(config.getInt(PORT_KEY))
               .addressFilter(
                   AddressFilter.load(config, FILTER_KEY))
               .inputBufferSize(
                   config.hasPath(INBUFFER_SIZE_KEY) ?
                   config.getInt(INBUFFER_SIZE_KEY) :
                   DEFAULT_BUFFER_SIZE)
               .outputBufferSize(
                   config.hasPath(OUTBUFFER_SIZE_KEY) ?
                   config.getInt(OUTBUFFER_SIZE_KEY) :
                   DEFAULT_BUFFER_SIZE)
               .byteOrder(loadByteOrder(config, BYTE_ORDER_KEY))
               .messageQueueSize(
                   config.hasPath(MSG_QUEUE_SIZE_KEY) ?
                   config.getInt(MSG_QUEUE_SIZE_KEY) :
                   DEFAULT_QUEUE_SIZE)
               .serviceSelector(
                   config.hasPath(SVC_SELECTOR_KEY) ?
                   config.getString(SVC_SELECTOR_KEY) :
                   (ENetConfigure.defaultSelector()).name())
               .connectionSelector(
                   config.hasPath(CONN_SELECTOR_KEY) ?
                   config.getString(CONN_SELECTOR_KEY) :
                   (ENetConfigure.defaultSelector()).name())
               .heartbeatDelay(
                   config.hasPath(HB_DELAY_KEY) ?
                   config.getLong(HB_DELAY_KEY) :
                   0L)
               .heartbeatReplyDelay(
                   config.hasPath(HB_REPLY_DELAY_KEY) ?
                   config.getLong(HB_REPLY_DELAY_KEY) :
                   0L)
               .canPause(config.hasPath(CAN_PAUSE_KEY) ?
                         config.getBoolean(CAN_PAUSE_KEY) :
                         false);

            if (builder.mCanPause)
            {
                final Config pauseConfig =
                    config.getConfig(PAUSE_KEY);
                final PauseBuilder pauseBuilder =
                    new PauseBuilder(ConnectionRole.ACCEPTOR);

                pauseBuilder.duration(pauseConfig.getDuration(PAUSE_DURATION_KEY))
                            .maxBacklogSize(pauseConfig.getInt(MAX_BACKLOG_SIZE_KEY));

                builder.pauseConfig(pauseBuilder.build());
            }

        return (builder.build());
    } // end of loadService(Config)

    /**
     * Returns a {@link RemoteConnection} instance based on the
     * given JSON configuration.
     * @param config extract connection from this JSON
     * configuration.
     * @return remote connection.
     * @throws ConfigException
     * if {@code config} contains an invalid connection
     * configuration.
     */
    private static RemoteConnection loadConnection(final Config config)
        throws ConfigException
    {
        final ConnectionType connType =
            (config.hasPath(CONN_TYPE_KEY) ?
             config.getEnum(ConnectionType.class, CONN_TYPE_KEY) :
             ConnectionType.TCP);
        final ConnectionBuilder builder =
            new ConnectionBuilder();

        // Only plain text TCP connections are supported
        // because supporting a secure TCP connection
        // requires storing the passphrase in config.
        // This can be done in a somewhat safe manner but
        // there are still other security issues.
        builder.loaderFlag(true)
               .name(config.getString(NAME_KEY))
               .connectionType(connType)
               .address(loadAddress(config))
               .bindPort(
                   config.hasPath(BIND_PORT_KEY) ?
                   config.getInt(BIND_PORT_KEY) :
                   ANY_PORT)
               .inputBufferSize(
                   config.hasPath(INBUFFER_SIZE_KEY) ?
                   config.getInt(INBUFFER_SIZE_KEY) :
                   DEFAULT_BUFFER_SIZE)
               .outputBufferSize(
                   config.hasPath(OUTBUFFER_SIZE_KEY) ?
                   config.getInt(OUTBUFFER_SIZE_KEY) :
                   DEFAULT_BUFFER_SIZE)
               .byteOrder(loadByteOrder(config, BYTE_ORDER_KEY))
               .messageQueueSize(
                   config.hasPath(MSG_QUEUE_SIZE_KEY) ?
                   config.getInt(MSG_QUEUE_SIZE_KEY) :
                   DEFAULT_QUEUE_SIZE)
               .selector(config.hasPath(SELECTOR_KEY) ?
                   config.getString(SELECTOR_KEY) :
                   (ENetConfigure.defaultSelector()).name())
               .reconnect(
                   config.hasPath(RECONNECT_KEY) ?
                   config.getBoolean(RECONNECT_KEY) :
                   false)
               .reconnectDelay(
                   builder.mReconnectFlag ?
                   config.getLong(RECONNECT_DELAY_KEY) :
                   0L)
               .heartbeatDelay(
                   config.hasPath(HB_DELAY_KEY) ?
                   config.getLong(HB_DELAY_KEY) :
                   DEFAULT_HEARTBEAT_DELAY)
               .heartbeatReplyDelay(
                   config.hasPath(HB_REPLY_DELAY_KEY) ?
                   config.getLong(HB_REPLY_DELAY_KEY) :
                   DEFAULT_HEARTBEAT_REPLY_DELAY)
               .canPause(config.hasPath(CAN_PAUSE_KEY) ?
                         config.getBoolean(CAN_PAUSE_KEY) :
                         false);

        if (builder.mCanPause)
        {
            final Config pauseConfig =
                config.getConfig(PAUSE_KEY);
            final PauseBuilder pauseBuilder =
                new PauseBuilder(ConnectionRole.INITIATOR);

            pauseBuilder.duration(pauseConfig.getDuration(PAUSE_DURATION_KEY))
                        .maxBacklogSize(pauseConfig.getInt(MAX_BACKLOG_SIZE_KEY))
                        .discardPolicy(pauseConfig.getEnum(DiscardPolicy.class,DISCARD_POLICY_KEY))
                        .idleTime(pauseConfig.getDuration(IDLE_TIME_KEY))
                        .maxConnectionTime(pauseConfig.getDuration(MAX_CONNECT_TIME_KEY));

            if (pauseConfig.hasPath(RESUME_ON_BACKLOG_SIZE_KEY))
            {
                pauseBuilder.resumeOnBacklogSize(pauseConfig.getInt(RESUME_ON_BACKLOG_SIZE_KEY));
            }

            builder.pauseConfig(pauseBuilder.build());
        }

        return (builder.build());
    } // end of loadConnection(Config)

    /**
     * Returns a dispatcher configured based on the given JSON
     * properties.
     * @param config JSCON configuration.
     * @return dispatcher
     * @throws ConfigException
     * if {@code config} contains an invalid dispatcher
     * configuration.
     */
    private static Dispatcher loadDispatcher(final Config config)
        throws ConfigException
    {
        DispatcherBuilder builder;

        builder = new DispatcherBuilder();
        builder.name(config.getString(NAME_KEY))
               .dispatcherType(
                   DispatcherType.findType(builder.mName));

        // If this dispatcher type is special, then only
        // certain properties apply.
        if ((builder.mType).isSpecial())
        {
            loadSpecialDispatcher(builder, config);
        }
        else
        {
            loadDispatcher(builder, config);
        }

        return (builder.build());
    } // end of loadDispatcher(Config)

    /**
     * Returns the named {@link DispatcherType#EBUS eBus}
     * dispatcher loaded in from the given JSON configuration.
     * @param builder place dispatcher configuration into this
     * builder.
     * @param config JSON configuration.
     * @throws ConfigException
     * if {@code config} contains an invalid dispatcher
     * configuration.
     */
    @SuppressWarnings ("fallthrough")
    private static void loadDispatcher(final DispatcherBuilder builder,
                                       final Config config)
        throws ConfigException
    {
        builder.threadType(config.hasPath(THREAD_TYPE_KEY) ?
                           ThreadType.find(config.getString(THREAD_TYPE_KEY)) :
                           ThreadType.find(DEFAULT_THREAD_TYPE));

        switch (builder.mRunQueueType)
        {
            case SPINPARK:
                builder.parkTime(config.hasPath(PARK_TIME_KEY) ?
                                 config.getLong(PARK_TIME_KEY) :
                                 DEFAULT_PARK_TIME);
                // Continue on through to the next case.

            case SPINYIELD:
                builder.spinLimit(config.hasPath(SPIN_LIMIT_KEY) ?
                                  config.getLong(SPIN_LIMIT_KEY) :
                                  DEFAULT_SPIN_LIMIT);
                break;

            default:
                // Do nothing for the remaining cases.
        }

        builder.priority(config.hasPath(PRIORITY_KEY) ?
                         config.getInt(PRIORITY_KEY) :
                         DEFAULT_PRIORITY)
               .quantum(config.hasPath(QUANTUM_KEY) ?
                        config.getLong(QUANTUM_KEY) :
                        DEFAULT_QUANTUM)
               .numberThreads(config.hasPath(NUM_THREADS_KEY) ?
                              config.getInt(NUM_THREADS_KEY) :
                              DEFAULT_NUMBER_THREADS)
               .isDefault(config.hasPath(DEFAULT_KEY) ?
                          config.getBoolean(DEFAULT_KEY) :
                          false)
               .classes(builder.mIsDefault ?
                        new Class<?>[0] :
                        loadClasses(config));
    } // end of loadDispatcher(DipatcherBuilder, Config)

    /**
     * Returns the named special dispatcher loaded from the given
     * configuration. Special dispatchers only use the
     * "isDefault" and "classes" keys.
     * @param builder place special dispatcher configuration into
     * this builder.
     * @param config JSON configuration containing special
     * dispatcher configuration.
     * @throws ConfigException
     * if {@code config} contains an invalid special dispatcher
     * configuration.
     */
    private static void loadSpecialDispatcher(final DispatcherBuilder builder,
                                              final Config config)
        throws ConfigException
    {
        builder.isDefault(config.hasPath(DEFAULT_KEY) ?
                          config.getBoolean(DEFAULT_KEY) :
                          false)
               .classes(builder.mIsDefault ?
                        new Class<?>[0] :
                        loadClasses(config))
               // Set the priority, quantum, and numThreads to
               // zero since special Dispatcher threads are 3rd
               // party and not configured by eBus. Uses a
               // blocking thread type.
               .threadType(ThreadType.BLOCKING)
               .spinLimit(0L)
               .parkTime(0L)
               .priority(0)
               .quantum(0L)
               .numberThreads(0);
    } // end of loadSpecialDispatcher(DispatcherBuilder, Config)

    private static ConnectionType getConnectionType(final String key,
                                                    final ConnectionType defaultType,
                                                    final net.sf.eBus.util.Properties p)
    {
        final String name = p.getProperty(key);
        final ConnectionType retval;

        if (name == null)
        {
            retval = defaultType;
        }
        else
        {
            try
            {
                retval = ConnectionType.valueOf(name);
            }
            catch (IllegalArgumentException argex)
            {
                throw (
                    new ConfigException.BadValue(
                        key,
                        "\"" + name + "\" is an invalid ConnectionType",
                        argex));
            }
        }

        return (retval);
    } // end of getConnectionType(String, Properties)

    private static ConnectionType getConnectionType(final String key,
                                                    final ConnectionType defaultType,
                                                    final Preferences p)
    {
        final String name = p.get(key, null);
        final ConnectionType retval;

        if (name == null)
        {
            retval = defaultType;
        }
        else
        {
            try
            {
                retval = ConnectionType.valueOf(name);
            }
            catch (IllegalArgumentException argex)
            {
                throw (
                    new ConfigException.BadValue(
                        key,
                        "\"" + name + "\" is an invalid ConnectionType",
                        argex));
            }
        }

        return (retval);
    } // end of getConnectionType(String, Preferences)

    private static int getPort(
        final String key,
        final int defaultPort,
        final net.sf.eBus.util.Properties p)
    {
        final int retval = p.getIntProperty(key, defaultPort);

        if ((retval < 0 && retval != ANY_PORT) ||
            retval > MAX_PORT)
        {
            final String badValue = p.getProperty(key);

            throw (
                new ConfigException.BadValue(
                    key,
                    key + " invalid port (" + badValue + ")"));
        }

        return (retval);
    } // end of getPort(String, Properties)

    private static int getPort(final String key,
                               final int defaultPort,
                               final Preferences p)
    {
        final int retval = p.getInt(key, defaultPort);

        if ((retval < 0 && retval != ANY_PORT) ||
            retval > MAX_PORT)
        {
            throw (
                new IllegalStateException(
                    key +
                    " invalid port (" +
                    Integer.toString(retval) +
                    ")"));
        }


        return (retval);
    } // end of getPort(String, int, Preferences)

    private static AddressFilter getFilter(
        final String key,
        final net.sf.eBus.util.Properties p)
    {
        AddressFilter retval;

        try
        {
            retval = AddressFilter.parse(p.getProperty(key));
        }
        catch (ParseException parsex)
        {
            final String badValue = p.getProperty(key);

            throw (
                new ConfigException.BadValue(
                    key,
                    key + " invalid address filter (" + badValue + ")",
                    parsex));
        }

        return (retval);
    } // end of getFilter(String, Properties)

    private static AddressFilter getFilter(final String key,
                                           final Preferences p)
    {
        AddressFilter retval;

        try
        {
            retval = AddressFilter.parse(p.get(key, null));
        }
        catch (ParseException parsex)
        {
            throw (new IllegalStateException(parsex));
        }

        return (retval);
    } // end of getFilter(String, Preferences)

    private static InetSocketAddress getAddress(final String name,
                                                final String keyPrefix,
                                                final net.sf.eBus.util.Properties p)
    {
        String key;
        String host;
        int port;
        InetSocketAddress retval;

        key = keyPrefix + HOST_KEY;
        host = p.getProperty(key);
        if (Strings.isNullOrEmpty(host))
        {
            throw (
                new ConfigException.BadValue(
                    key, "host is null or empty"));
        }

        key = keyPrefix + PORT_KEY;
        port = getPort(key, -1, p);

        try
        {
            retval =
                new InetSocketAddress(
                    InetAddress.getByName(host), port);
        }
        catch (UnknownHostException hostex)
        {
            throw (
                new ConfigException.BadValue(
                    key,
                    name + " invalid address \"" + host + "\"",
                    hostex));
        }

        return (retval);
    } // end of getAddress(String, String, Properties)

    private static InetSocketAddress getAddress(final String keyPrefix,
                                                final Preferences p)
    {
        String key;
        String host;
        int port;
        InetSocketAddress retval;

        key = keyPrefix + HOST_KEY;
        host = p.get(key, null);
        if (Strings.isNullOrEmpty(host))
        {
            throw (
                new ConfigException.BadValue(
                    key, "host is null or empty"));
        }

        key = keyPrefix + PORT_KEY;
        port = getPort(key, ANY_PORT, p);

        try
        {
            retval =
                new InetSocketAddress(
                    InetAddress.getByName(host), port);
        }
        catch (UnknownHostException hostex)
        {
            throw (
                new ConfigException.BadValue(
                    key,
                    host,
                    hostex));
        }

        return (retval);
    } // end of getAddress(String, String, Preferences)

    private static InetSocketAddress loadAddress(final Config config)
        throws ConfigException
    {
        final String host = config.getString(HOST_KEY);
        final int port = config.getInt(PORT_KEY);
        final InetSocketAddress retval;

        try
        {
            retval =
                new InetSocketAddress(
                    InetAddress.getByName(host), port);
        }
        catch (UnknownHostException hostex)
        {
            throw (
                new ConfigException.BadValue(
                    HOST_KEY,
                    "\"" + host + "\" is not a valid address",
                    hostex));
        }

        return (retval);
    } // end of loadAddress(Config)

    private static int getSize(final String key,
                               final int defaultValue,
                               final int minValue,
                               final net.sf.eBus.util.Properties p)
    {
        final int retval = p.getIntProperty(key, defaultValue);

        if (retval < minValue)
        {
            final String badValue = p.getProperty(key);

            throw (
                new ConfigException.BadValue(
                    key, badValue + " < " + minValue));
        }

        return (retval);
    } // end of getSize(String, int, int, Properties)

    /**
     * Returns the size value stored in the given key and
     * preferences.
     * @param key the preference key.
     * @param defaultValue this value is returned if {@code key}
     * is not in the preferences.
     * @param minValue the minimum accepted values.
     * @param p the user preferences.
     * @return the size value retrieved from the preferences.
     * @throws IllegalStateException
     * if {@code key} does not contain a valid size.
     */
    private static int getSize(final String key,
                               final int defaultValue,
                               final int minValue,
                               final Preferences p)
    {
        final int retval = p.getInt(key, defaultValue);

        if (retval < minValue)
        {
            throw (
                new IllegalStateException(
                    key + " < " + Integer.toString(minValue)));
        }

        return (retval);
    } // end of getSize(String, int, Preferences)

    /**
     * Returns the buffer byte order stored in the given key and
     * properties. If not defined in the preferences, then
     * returns {@code defaultValue}.
     * @param key the property key.
     * @param defaultValue the default byte order.
     * @param p the properties.
     * @return the buffer byte order.
     * @throws IllegalStateException
     * if {@code key} contains an invalid byte order.
     */
    private static ByteOrder getByteOrder(
        final String key,
        final ByteOrder defaultValue,
        final net.sf.eBus.util.Properties p)
    {
        final String value = p.getProperty(key);
        ByteOrder retval;

        // Was a big endian order selected?
        if (BIGENDIAN.equalsIgnoreCase(value))
        {
            // Yes.
            retval = ByteOrder.BIG_ENDIAN;
        }
        // Was a little endian order selected?
        else if (LITTLEENDIAN.equalsIgnoreCase(value))
        {
            // Yes.
            retval = ByteOrder.LITTLE_ENDIAN;
        }
        // Was any order selected?
        else if (value == null || value.isEmpty())
        {
            // No, use the default value.
            retval = defaultValue;
        }
        // Otherwise, the key was set but to what I don't know.
        else
        {
            throw (
                new ConfigException.BadValue(
                    key,
                    key + " value \"" + value + "\" is invalid"));
        }

        return (retval);
    } // end of getByteOrder(String, ByteOrder, Properties)

    /**
     * Returns the buffer byte order stored in the given key and
     * preferences. If not defined in the preferences, then
     * returns {@code defaultValue}.
     * @param key the preference key.
     * @param defaultValue the default byte order.
     * @param p the user preferences.
     * @return the byte order.
     * @throws IllegalStateException
     * if {@code key} contains an invalid byte order.
     */
    private static ByteOrder getByteOrder(
        final String key,
        final ByteOrder defaultValue,
        final Preferences p)
    {
        final String value = p.get(key, null);
        ByteOrder retval;

        // Was a big endian order selected?
        if (BIGENDIAN.equalsIgnoreCase(value))
        {
            // Yes.
            retval = ByteOrder.BIG_ENDIAN;
        }
        // Was a little endian order selected?
        else if (LITTLEENDIAN.equalsIgnoreCase(value))
        {
            // Yes.
            retval = ByteOrder.LITTLE_ENDIAN;
        }
        // Was any order selected?
        else if (value == null || value.isEmpty())
        {
            // No, use the default value.
            retval = defaultValue;
        }
        // Otherwise, the key was set but to what I don't know.
        else
        {
            throw (
                new IllegalStateException(
                    String.format("%s value \"%s\" invalid",
                                  key,
                                  value)));
        }

        return (retval);
    } // end of getByteOrder(String, ByteOrder, Preferences)

    /**
     * Returns the buffer byte order stored in the given key path
     * in {@code config}.
     * preferences. If not defined in the preferences, then
     * returns {@link #DEFAULT_BYTE_ORDER}.
     * @param config JSON configuration containing byte order
     * {@code key}.
     * @param key JSON configuration path key.
     * @return byte order
     * @throws ConfigException
     * if {@code key} contains an invalid byte order.
     */
    private static ByteOrder loadByteOrder(final Config config,
                                           final String key)
        throws ConfigException
    {
        final ByteOrder retval;

        if (!config.hasPathOrNull(BYTE_ORDER_KEY))
        {
            retval = DEFAULT_BYTE_ORDER;
        }
        else
        {
            final String value = config.getString(key);

            switch (value)
            {
                case BIGENDIAN:
                    retval = ByteOrder.BIG_ENDIAN;
                    break;

                case LITTLEENDIAN:
                    retval = ByteOrder.LITTLE_ENDIAN;
                    break;

                default:
                    throw (
                        new ConfigException.BadValue(
                            key,
                            "\"" + value + "\" is not a valid java.nio.ByteOrder"));
            }
        }

        return (retval);
    } // end of loadByteOrder(Config, String)

    /**
     * Returns a buffer byte order stored in the JSON
     * configuration under the given key. If the key is not
     * present, then returns {@link ByteOrder#LITTLE_ENDIAN}.
     * @param key
     * @param defaultValue
     * @param minValue
     * @param p
     * @return
     */
    private static long getDelay(final String key,
                                 final int defaultValue,
                                 final long minValue,
                                 final net.sf.eBus.util.Properties p)
    {
        final long retval = p.getIntProperty(key, defaultValue);

        if (retval < minValue)
        {
            throw (new IllegalStateException("negative " + key));
        }

        return (retval);
    } // end of getDelay(String, long, Properties)

    private static long getDelay(final String key,
                                 final int defaultValue,
                                 final long minValue,
                                 final Preferences p)
    {
        final long retval = p.getInt(key, defaultValue);

        if (retval < minValue)
        {
            throw (new IllegalStateException("negative " + key));
        }

        return (retval);
    } // end of getDelay(String, int, Preferences)

    /**
     * Returns the classes listed in the "classes" Dispatcher
     * property.
     * @param key the fully-qualified ".classes" key.
     * @param p extract the classes from this property.
     * @return classes array.
     * @throws ConfigException
     * if the ".classes" property is either missing, empty, or
     * contains an invalid class name.
     */
    private static Class<?>[]
        getClasses(final String key,
                   final net.sf.eBus.util.Properties p)
    {
        final String[] classNames =
            p.getArrayProperty(key, KEY_IFS);
        final int size = classNames.length;
        int index;
        final Class<?>[] retval;

        // If no classes were provided, then throw an
        // exception.
        if (size == 0)
        {
            throw (
                new ConfigException.BadValue(
                    key, "classes are missing or empty"));
        }

        retval = new Class<?>[size];

        for (index = 0; index < size; ++index)
        {
            try
            {
                retval[index] = Class.forName(classNames[index]);
            }
            catch (ClassNotFoundException jex)
            {
                throw (
                    new ConfigException.BadValue(
                        key,
                        "unknown class " + classNames[index],
                        jex));
            }
        }

        return (retval);
    } // end of getClasses(String, Properties)

    /**
     * Returns the classes listed in the ".classes" Dispatcher
     * property.
     * @param key the fully-qualified ".classes" key.
     * @param p extract the classes from this preference.
     * @return classes array.
     * @throws MissingResourceException
     * if the ".classes" key is either missing, empty, or
     * contains an invalid class name.
     */
    private static Class<?>[] getClasses(final String key,
                                         final Preferences p)
    {
        final String[] classNames = (p.get(key, "")).split(",");
        final int size = classNames.length;
        int index;
        final Class<?>[] retval;

        // If no classes were provided, then throw an
        // exception.
        if (size == 0)
        {
            throw (
                new MissingResourceException(
                    String.format(
                        "%s missing or empty", key),
                    (Class.class).getName(),
                    key));
        }

        retval = new Class<?>[size];
        for (index = 0; index < size; ++index)
        {
            try
            {
                retval[index] = Class.forName(classNames[index]);
            }
            catch (ClassNotFoundException jex)
            {
                throw (
                    new MissingResourceException(
                        String.format(
                            "unknown class %s",
                            classNames[index]),
                        (Class.class).getName(),
                        key));
            }
        }

        return (retval);
    } // end of getClasses(String, Preferences)

    /**
     * Returns the classes listed in the "classes" Dispatcher
     * key.
     * @param config JSON configuration containing the classes
     * list.
     * @return classes array.
     * @throws ConfigException
     * if {@code config} contains an invalid class name.
     */
    private static Class<?>[] loadClasses(final Config config)
        throws ConfigException
    {
        final List<String> classNames =
            config.getStringList(CLASSES_KEY);
        int index = 0;
        final Class<?>[] retval = new Class[classNames.size()];

        // If no classes were provided, then throw an
        // exception.
        if (classNames.isEmpty())
        {
            throw (
                new ConfigException.BadValue(
                    CLASSES_KEY, "classes are missing or empty"));
        }

        for (String className : classNames)
        {
            try
            {
                retval[index] = Class.forName(className);
            }
            catch (ClassNotFoundException classex)
            {
                throw (
                    new ConfigException.BadValue(
                        CLASSES_KEY,
                        "\"" + className + "\" is an unknown class",
                        classex));
            }

            ++index;
        }

        return (retval);
    } // end of loadClasses(Config)

    private void storeService(final java.util.Properties p)
    {
        final Formatter names = new Formatter();
        String sep = "";
        String name;
        String keyPrefix;
        String key;
        AddressFilter filter;

        for (Service service : mServices.values())
        {
            name = service.name();
            keyPrefix = SERVICE_PREFIX + name + ".";
            filter = service.addressFilter();

            names.format("%s%s", sep, name);

            key = keyPrefix + PORT_KEY;
            p.setProperty(key,
                          Integer.toString(service.port()));

            key = keyPrefix + FILTER_KEY;
            p.setProperty(
                key,
                (filter == null ? "" : filter.toString()));

            key = keyPrefix + BYTE_ORDER_KEY;
            p.setProperty(key,
                          (service.byteOrder()).toString());

            key = keyPrefix + INBUFFER_SIZE_KEY;
            p.setProperty(
                key,
                Integer.toString(service.inputBufferSize()));

            key = keyPrefix + OUTBUFFER_SIZE_KEY;
            p.setProperty(
                key,
                Integer.toString(service.outputBufferSize()));

            key = keyPrefix + MSG_QUEUE_SIZE_KEY;
            p.setProperty(
                key,
                Integer.toString(service.messageQueueSize()));
        }

        p.setProperty(
            EBUS_KEY + "." + SERVICES_KEY, names.toString());
    } // end of storeService(Properties)

    private void storeConnections(final java.util.Properties p)
    {
        final Formatter names = new Formatter();
        String sep = "";
        String name;
        String keyPrefix;
        String key;
        InetSocketAddress address;

        for (RemoteConnection connection :
                mRemoteConnections.values())
        {
            name = connection.name();
            keyPrefix = CONNECTION_PREFIX + name + ".";
            address = connection.address();

            names.format("%s%s", sep, name);

            key = keyPrefix + HOST_KEY;
            p.setProperty(key, address.getHostName());

            key = keyPrefix + PORT_KEY;
            p.setProperty(key,
                          Integer.toString(address.getPort()));

            key = keyPrefix + BIND_PORT_KEY;
            p.setProperty(
                key,
                Integer.toString(connection.bindPort()));

            key = keyPrefix + BYTE_ORDER_KEY;
            p.setProperty(key,
                          (connection.byteOrder()).toString());

            key = keyPrefix + INBUFFER_SIZE_KEY;
            p.setProperty(
                key,
                Integer.toString(connection.inputBufferSize()));

            key = keyPrefix + OUTBUFFER_SIZE_KEY;
            p.setProperty(
                key,
                Integer.toString(connection.outputBufferSize()));

            key = keyPrefix + MSG_QUEUE_SIZE_KEY;
            p.setProperty(
                key,
                Integer.toString(connection.messageQueueSize()));

            key = keyPrefix + RECONNECT_KEY;
            p.setProperty(
                key,
                Boolean.toString(connection.reconnectFlag()));

            key = keyPrefix + RECONNECT_DELAY_KEY;
            p.setProperty(
                key,
                Long.toString(connection.reconnectTime()));

            key = keyPrefix + CAN_PAUSE_KEY;
            p.setProperty(
                key,
                Boolean.toString(connection.canPause()));

            if (connection.pauseConfiguration() != null)
            {
                storePause(keyPrefix + PAUSE_KEY + ".",
                           connection.pauseConfiguration(),
                           p);
            }

            sep = KEY_SEP;
        }

        p.setProperty(
            EBUS_KEY + "." + CONNECTIONS_KEY, names.toString());
    } // end of storeConnections(Properties)

    private void storePause(final String keyPrefix,
                            final PauseConfig config,
                            final java.util.Properties p)
    {
        String key;

        key = keyPrefix + PAUSE_DURATION_KEY;
        p.setProperty(key, (config.duration()).toString());

        key = keyPrefix + MAX_BACKLOG_SIZE_KEY;
        p.setProperty(key,
                      Integer.toString(config.maxBacklogSize()));

        key = keyPrefix + DISCARD_POLICY_KEY;
        p.setProperty(key, (config.discardPolicy()).name());

        key = keyPrefix + IDLE_TIME_KEY;
        p.setProperty(key, (config.idleTime()).toString());

        key = keyPrefix + MAX_CONNECT_TIME_KEY;
        p.setProperty(key, (config.maxConnectTime()).toString());

        key = keyPrefix + RESUME_ON_BACKLOG_SIZE_KEY;
        p.setProperty(key,
                      Integer.toString(config.resumeOnQueueLimit()));
    } // end of storePause(Properties)

    private void storeDispatchers(final java.util.Properties p)
    {
        final Formatter names = new Formatter();
        String sep = "";
        String name;
        String keyPrefix;
        String key;

        for (Dispatcher dispatcher : mDispatchers.values())
        {
            name = dispatcher.name();
            keyPrefix =
                EBUS_KEY + "." + DISPATCHER_PREFIX + name + ".";

            names.format("%s%s", sep, name);

            key = keyPrefix + DEFAULT_KEY;
            p.setProperty(
                key,
                Boolean.toString(dispatcher.isDefault()));

            key = keyPrefix + PRIORITY_KEY;
            p.setProperty(
                key,
                String.valueOf(dispatcher.priority()));

            key = keyPrefix + QUANTUM_KEY;
            p.setProperty(
                key,
                Long.toString(dispatcher.quantum()));

            key = keyPrefix + NUM_THREADS_KEY;
            p.setProperty(
                key,
                Integer.toString(dispatcher.numberThreads()));

            sep = KEY_SEP;
        }

        key = EBUS_KEY + "." + DISPATCHERS_KEY;
        p.setProperty(key, names.toString());
    } // end of storeDispatchers(Properties)

    private void storeService(final Preferences p)
    {
        final Formatter names = new Formatter();
        String sep = "";
        String name;
        String keyPrefix;
        String key;
        AddressFilter filter;

        for (Service service : mServices.values())
        {
            name = service.name();
            keyPrefix = SERVICE_PREFIX + name + ".";
            filter = service.addressFilter();

            names.format("%s%s", sep, name);

            key = keyPrefix + PORT_KEY;
            p.putInt(key, service.port());

            key = keyPrefix + FILTER_KEY;
            p.put(
                key, (filter == null ? "" : filter.toString()));

            key = keyPrefix + INBUFFER_SIZE_KEY;
            p.putInt(key, service.inputBufferSize());

            key = keyPrefix + OUTBUFFER_SIZE_KEY;
            p.putInt(key, service.outputBufferSize());

            key = keyPrefix + MSG_QUEUE_SIZE_KEY;
            p.putInt(key, service.messageQueueSize());
        }

        p.put(EBUS_KEY + "." + SERVICES_KEY, names.toString());
    } // end of storeService(Preferences)

    private void storeConnections(final Preferences p)
    {
        final Formatter names = new Formatter();
        String sep = "";
        String name;
        String keyPrefix;
        String key;
        InetSocketAddress address;

        for (RemoteConnection connection:
                mRemoteConnections.values())
        {
            name = connection.name();
            keyPrefix = CONNECTION_PREFIX + name + ".";
            address = connection.address();

            names.format("%s%s", sep, name);

            key = keyPrefix + HOST_KEY;
            p.put(key, address.getHostName());

            key = keyPrefix + PORT_KEY;
            p.putInt(key, address.getPort());

            key = keyPrefix + BIND_PORT_KEY;
            p.putInt(key, connection.bindPort());

            key = keyPrefix + INBUFFER_SIZE_KEY;
            p.putInt(key, connection.inputBufferSize());

            key = keyPrefix + OUTBUFFER_SIZE_KEY;
            p.putInt(key, connection.outputBufferSize());

            key = keyPrefix + MSG_QUEUE_SIZE_KEY;
            p.putInt(key, connection.messageQueueSize());

            key = keyPrefix + RECONNECT_KEY;
            p.putBoolean(key, connection.reconnectFlag());

            key = keyPrefix + RECONNECT_DELAY_KEY;
            p.putLong(key, connection.reconnectTime());

            sep = KEY_SEP;
        }

        p.put(EBUS_KEY + "." + CONNECTIONS_KEY, names.toString());
    } // end of storeConnections(Preferences)

    private void storeDispatchers(final Preferences p)
    {
        final Formatter names = new Formatter();
        String sep = "";
        String name;
        String keyPrefix;
        String key;

        for (Dispatcher dispatcher : mDispatchers.values())
        {
            name = dispatcher.name();
            keyPrefix =
                EBUS_KEY + "." + DISPATCHER_PREFIX + name + ".";

            names.format("%s%s", sep, name);

            key = keyPrefix + DEFAULT_KEY;
            p.putBoolean(key, dispatcher.isDefault());

            key = keyPrefix + PRIORITY_KEY;
            p.putInt(key, dispatcher.priority());

            key = keyPrefix + QUANTUM_KEY;
            p.putLong(key, dispatcher.quantum());

            key = keyPrefix + NUM_THREADS_KEY;
            p.putInt(key, dispatcher.numberThreads());

            sep = KEY_SEP;
        }

        key = EBUS_KEY + "." + DISPATCHERS_KEY;
        p.put(key, names.toString());
    } // end of storeDispatchers(Preferences)

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Base class for service and remote connections. Contains
     * those data members common to both.
     */
    public static abstract class AbstractConfig
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * eBus local name.
         */
        protected final String mName;

        /**
         * The connections are of this type.
         */
        protected final ConnectionType mConnectionType;

        /**
         * Input buffer size.
         */
        protected final int mInputBufferSize;

        /**
         * Output buffer size.
         */
        protected final int mOutputBufferSize;

        /**
         * The input and output buffer use this byte ordering.
         */
        protected final ByteOrder mByteOrder;

        /**
         * Maximum message queue size.
         */
        protected final int mMsgQueueSize;

        /**
         * Send a heartbeat this many milliseconds after the
         * last received input. A zero value means heartbeating
         * is not performed.
         */
        protected final long mHbDelay;

        /**
         * Wait this many milliseconds for a heartbeat reply.
         * This timer is reset when a non-heartbeat reply is
         * received. A zero value means an indefinite reply
         * time.
         */
        protected final long mHbReplyDelay;

        /**
         * If {@link #mConnectionType} is
         * {@link ConnectionType#SECURE_TCP}, then this is the
         * SSL context used to secure the connection.
         */
        protected final SSLContext mSSLContext;

        /**
         * Set to {@code true} if accepted connections may be
         * paused.
         */
        protected final boolean mCanPause;

        /**
         * If {@link #mCanPause} is {@code true}, then the pause
         * configuration is defined. Otherwise set to
         * {@code null}.
         */
        protected final PauseConfig mPauseConfig;

    //-----------------------------------------------------------
    // Member methods.
    //
        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Constructs the common information for service and
         * remote connections.
         * @param builder connection builder.
         */
        protected AbstractConfig(final AbstractBuilder<?> builder)
        {
            this.mName = builder.mName;
            this.mConnectionType = builder.mConnectionType;
            this.mInputBufferSize = builder.mInputBufferSize;
            this.mOutputBufferSize = builder.mOutputBufferSize;
            this.mByteOrder = builder.mByteOrder;
            this.mMsgQueueSize = builder.mMsgQueueSize;
            this.mHbDelay = builder.mHbDelay;
            this.mHbReplyDelay = builder.mHbReplyDelay;
            this.mSSLContext = builder.mSSLContext;
            this.mCanPause = builder.mCanPause;
            this.mPauseConfig = builder.mPauseConfig;
        } // end of AbstractConfig()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the remote connection unique name.
         * @return the remote connection unique name.
         */
        public final String name()
        {
            return (mName);
        } // end of name()

        /**
         * Returns the channel type. This is currently limited to
         * plain text TCP and SSL/TLS secure TCP connections.
         * @return connection type.
         */
        public final ConnectionType connectionType()
        {
            return (mConnectionType);
        } // end of connectionType()

        /**
         * Returns the connection input buffer size.
         * @return the connection input buffer size.
         */
        public final int inputBufferSize()
        {
            return (mInputBufferSize);
        } // end of inputBufferSize()

        /**
         * Returns the connection output buffer size.
         * @return the connection output buffer size.
         */
        public final int outputBufferSize()
        {
            return (mOutputBufferSize);
        } // end of outputBufferSize()

        /**
         * Returns connection buffer byte ordering.
         * @return the connection buffer byte ordering.
         */
        public final ByteOrder byteOrder()
        {
            return (mByteOrder);
        } // end of byteOrder()

        /**
         * Returns the maximum number of messages on the
         * output queue. Returns zero if there is no limit.
         * @return the message queue maximum size.
         */
        public final int messageQueueSize()
        {
            return (mMsgQueueSize);
        } // end of messageQueueSize()

        /**
         * Returns the heartbeat delay in milliseconds. A zero
         * value means that heartbeating is not performed.
         * @return the heartbeat delay in milliseconds.
         */
        public final long heartbeatDelay()
        {
            return (mHbDelay);
        } // end of heartbeatDelay()

        /**
         * Returns the heartbeat reply delay. A zero value means
         * an indefinite delay.
         * @return the heartbeat reply delay.
         */
        public final long heartbeatReplyDelay()
        {
            return (mHbReplyDelay);
        } // end of heartbeatReplyDelay()

        /**
         * Returns the SSL context used for a secure connection
         * type. Returns {@code null} if connection type is
         * <em>not</em> {@link ConnectionType#SECURE_TCP}.
         * @return SSL context. May return {@code null}.
         */
        public final SSLContext sslContext()
        {
            return (mSSLContext);
        } // end of sslContext()

        /**
         * Returns {@code true} if pause requests are accepted
         * and {@code false} if not.
         * @return {@code true} if accepted connections may be
         * paused.
         */
        public final boolean canPause()
        {
            return (mCanPause);
        } // end of canPause()

        /**
         * Returns the connection pause configuration. If
         * {@link #canPause()} returns {@code false}, then
         * returns {@code null}.
         * @return connection pause configuration.
         */
        public final PauseConfig pauseConfiguration()
        {
            return (mPauseConfig);
        } // end of pauseConfiguration()

        //
        // end of Get Methods.
        //-------------------------------------------------------
    } // end of class AbstractConfig

    /**
     * This immutable class stores the configuration for an
     * eBus service. This includes:
     * <ol>
     *   <li>
     *     Service name.
     *   </li>
     *   <li>
     *     {@link ConnectionType}.
     *   </li>
     *   <li>
     *     The TCP service port.
     *   </li>
     *   <li>
     *     The optional {@link AddressFilter address filter}.
     *   </li>
     *   <li>
     *     Each accepted socket input buffer size. Limits how
     *     many bytes are received at one time.
     *   </li>
     *   <li>
     *     Each accepted socket output buffer size. Limits how
     *     many bytes are transmitted at one time.
     *   </li>
     *   <li>
     *     The accepted socket input and output buffer byte
     *     ordering.
     *   </li>
     *   <li>
     *     The accepted connection's message queue size.
     *   </li>
     *   <li>
     *     The selector thread used by the server socket.
     *   </li>
     *   <li>
     *     The selector thread use for accepted client sockets.
     *   </li>
     *   <li>
     *     The accepted connection's heartbeat frequency.
     *   </li>
     *   <li>
     *     The accepted connection's heartbeat reply time limit.
     *   </li>
     * </ol>
     */
    public static final class Service
        extends AbstractConfig
        implements Comparable<Service>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * TCP port.
         */
        private final int mPort;

        /**
         * The optional address filter.
         */
        private final AddressFilter mAddressFilter;

        /**
         * Register this service with this specified selector.
         */
        private final String mServiceSelector;

        /**
         * Register accepted connections with this specified
         * selector.
         */
        private final String mConnSelector;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates an eBus service configuration from the
         * builder's current settings.
         * @param builder construct service configuration from
         * these settings.
         */
        /* package */  Service(final ServerBuilder builder)
        {
            super (builder);

            this.mPort = builder.mPort;
            this.mAddressFilter = builder.mAddressFilter;
            this.mServiceSelector = builder.mServiceSelector;
            this.mConnSelector = builder.mConnSelector;
        } // end of Service(ServerBuilder)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Comparable Interface Implementation.
        //
        /**
         * Compares this service configuration with
         * {@code service} based on the service ports.
         * @param service compare this service instance.
         * @return an integer value.
         */
        @Override
        public int compareTo(
            final Service service)
        {
            return (mPort - service.mPort);
        } // end of compareTo(Service)

        //
        // end of Comparable Interface Implementation.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Object Method Overrides.
        //

        /**
         * Returns {@code true} if {@code o} is a {@code Service}
         * instance with the same port and {@code false}
         * otherwise.
         * @param o Test if this object is equal.
         * @return {@code true} if {@code o} is a {@code Service}
         * instance with the same port and {@code false}
         * otherwise.
         */
        @Override
        public boolean equals(final Object o)
        {
            boolean retcode = (this == o);

            if (!retcode && o instanceof Service)
            {
                final Service svc = (Service) o;

                retcode =
                    (mConnectionType == svc.mConnectionType &&
                     mPort == svc.mPort);
            }

            return (retcode);
        } // end of equals(Object)

        /**
         * Returns the service port.
         * @return the service port.
         */
        @Override
        public int hashCode()
        {
            return (Objects.hash(mConnectionType, mPort));
        } // end of hashCode()

        /**
         * Returns a text representation of this eBus service.
         * @return a text representation of this eBus service.
         */
        @Override
        public String toString()
        {
            final Formatter retval = new Formatter();

            retval.format("[%s %d]%n",
                          mConnectionType,
                          mPort)
                  .format("address filter: %s%n",
                                           mAddressFilter)
                  .format("socket input size: %,d%n",
                                           mInputBufferSize)
                  .format("socket output size: %,d%n",
                                           mOutputBufferSize)
                  .format(" socket byte order: %s%n",
                                           mByteOrder)
                  .format("    max queue size: %,d%n",
                                           mMsgQueueSize)
                  .format("          selector: %s%n",
                                           mConnSelector);

            if (mConnectionType == ConnectionType.SECURE_TCP)
            {
                retval.format("       SSL context: %s%n",
                              mSSLContext);
            }

            retval.format("         can pause: %b%n", mCanPause);

            if (mCanPause)
            {
                retval.format("      pause config: %s%n",
                              mPauseConfig);
            }

            return (retval.toString());
        } // end of toString()

        //
        // end of Object Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the service port.
         * @return the service port.
         */
        public int port()
        {
            return (mPort);
        } // end of port()

        /**
         * Returns the address filter. May return {@code nul}.
         * @return the address filter.
         */
        public AddressFilter addressFilter()
        {
            return (mAddressFilter);
        } // end of addressFilter()

        /**
         * Returns the selector information to which the service
         * socket channel is registered.
         * @return service socket channel selector.
         */
        public String serviceSelector()
        {
            return (mServiceSelector);
        } // end of serviceSelector()

        /**
         * Returns the selector information to which accepted
         * connections are registered.
         * @return accepted connection selector information.
         */
        public String connectionSelector()
        {
            return (mConnSelector);
        } // end of connectionSelector()

        //
        // end of Get Methods.
        //-------------------------------------------------------
    } // end of Service

    /**
     * This immutable class stores the information pertaining
     * to a remote eBus connection. This includes:
     * <ol>
     *   <li>
     *     Service name.
     *   </li>
     *   <li>
     *     {@link ConnectionType}.
     *   </li>
     *   <li>
     *     The TCP/IP address and port.
     *   </li>
     *   <li>
     *     The TCP/IP bind port. This is is the TCP port to which
     *     the connection's local end is bound. If the bind port
     *     is zero, then the local end is bound to any available
     *     TCP port.
     *   </li>
     *   <li>
     *     The socket input buffer size. Limits how many bytes
     *     are received at one time.
     *   </li>
     *   <li>
     *     The socket output buffer size. Limits how many bytes
     *     are transmitted at one time.
     *   </li>
     *   <li>
     *     The socket input and output buffer byte ordering.
     *   </li>
     *   <li>
     *     The connection maximum message queue size . If the
     *     message queue exceeds this size, then the socket is
     *     disconnected.
     *   </li>
     *   <li>
     *     The selector thread used to monitor the socket.
     *   </li>
     *   <li>
     *     A reconnect flag which, if {@code true}, means that
     *     the connection is re-established if lost.
     *   </li>
     *   <li>
     *     When reconnecting wait this many milliseconds before
     *     attempting to re-establish the connection.
     *   </li>
     *   <li>
     *     The heartbeat frequency and heartbeat reply time
     *     limit.
     *   </li>
     * </ol>
     */
    public static final class RemoteConnection
        extends AbstractConfig
        implements Comparable<RemoteConnection>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Remote eBus Internet address.
         */
        private final InetSocketAddress mAddress;

        /**
         * Bind the local address to this TCP port.
         */
        private final int mBindPort;

        /**
         * Register this connection with the specified selector.
         */
        private final String mSelector;

        /**
         * If {@code true} then re-established lost connections.
         */
        private final boolean mReconnectFlag;

        /**
         * Wait this many seconds between reconnect attempts.
         */
        private final long mReconnectTime;

        /**
         * Specifies the protocol family used to open a UDP
         * datagram channel. Ignored for TCP connections.
         * Defaults to {@link StandardProtocolFamily#INET}.
         */
        private final ProtocolFamily mProtocol;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a remote connection configuration based on the
         * builder's current parameters.
         * @param builder contains the remote configuration
         * parameters.
         */
        /* package */ RemoteConnection(final ConnectionBuilder builder)
        {
            super (builder);

            this.mAddress = builder.mAddress;
            this.mBindPort = builder.mBindPort;
            this.mSelector = builder.mSelector;
            this.mReconnectFlag = builder.mReconnectFlag;
            this.mReconnectTime = builder.mReconnectTime;
            this.mProtocol = builder.mProtocol;
        } // end of RemoteConnection(ConnectionBuilder)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Comparable Interface Implementation.
        //

        /**
         * Compares {@code this} remote connection configuration
         * with {@code conn}.
         * @param conn compare with this object.
         * @return &lt; zero if {@code this} instance is less
         * than {@code conn}; zero if {@code this} instance is
         * equal to {@code conn}; &gt; zero if {@code this} is
         * greater than {@code conn}.
         */
        @Override
        public int compareTo(final RemoteConnection conn)
        {
            final InetSocketAddressComparator comparator =
            new InetSocketAddressComparator();

            return (comparator.compare(mAddress, conn.mAddress));
        } // end of compareTo(RemoteConnection)

        //
        // end of Comparable Interface Implementation.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Object Method Overrides.
        //

        /**
         * Returns {@code true} if {@code o} is a
         * non-{@code null} {@code RemoteConnection} instance
         * with the same name; {@code false} otherwise.
         * @param o Test equality with this object.
         * @return {@code true} if {@code o} is a
         * non-{@code null} {@code RemoteConnection} instance
         * with the same address and connection type;
         * {@code false} otherwise.
         */
        @Override
        public boolean equals(final Object o)
        {
            boolean retcode = (this == o);

            if (!retcode && o instanceof RemoteConnection)
            {
                retcode =
                    mAddress.equals(
                        ((RemoteConnection) o).mAddress);
            }

            return (retcode);
        } // end of equals(Object)

        /**
         * Returns the remote connection address hash code.
         * @return the remote connection address hash code.
         */
        @Override
        public int hashCode()
        {
            return (mAddress.hashCode());
        } // end of hashCode()

        /**
         * Returns a text representation for this remote
         * connection.
         * @return a text representation for this remote
         * connection.
         */
        @Override
        public String toString()
        {
            final Formatter retval = new Formatter();

            retval.format("[%s] address: %s%n",
                          mName,
                          mAddress)
                  .format("     connection type: %s%n",
                          mConnectionType)
                  .format("           bind port: %d%n",
                          mBindPort)
                  .format("   input buffer size: %,d%n",
                          mInputBufferSize)
                  .format("  output buffer size: %,d%n",
                          mOutputBufferSize)
                  .format("   buffer byte order: %s%n",
                          mByteOrder)
                  .format("          queue size: %,d%n",
                          mMsgQueueSize)
                  .format("            selector: %s%n",
                          mSelector)
                  .format("           reconnect: %b%n",
                          mReconnectFlag);
            if (mReconnectFlag)
            {
                retval.format(
                    "      reconnect time: %,d msecs%n",
                    mReconnectTime);
            }

            if (mConnectionType == ConnectionType.SECURE_TCP)
            {
                retval.format("       SSL context: %s%n",
                              mSSLContext);
            }

            if (mConnectionType == ConnectionType.UDP)
            {
                retval.format("          protocol: %s%n",
                              mProtocol);
            }

            return (retval.toString());
        } // end of toString()

        //
        // end of Object Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get methods.
        //

        /**
         * Returns the remote eBus socket address.
         * @return the remote eBus socket address.
         */
        public InetSocketAddress address()
        {
            return (mAddress);
        } // end of address()

        /**
         * Returns the connection bind port. This is the TCP
         * port to which the local end is bound.
         * @return the connection bind port.
         */
        public int bindPort()
        {
            return (mBindPort);
        } // end of bindPort()

        /**
         * Returns the selector information to which this
         * connection is registered.
         * @return selector information.
         */
        public String selector()
        {
            return (mSelector);
        } // end of connSelector()

        /**
         * Returns {@code true} if connection is to be
         * re-established when lost.
         * @return {@code true} if connection is to be
         * re-established when lost.
         */
        public boolean reconnectFlag()
        {
            return (mReconnectFlag);
        } // end of reconnectFlag()

        /**
         * Returns the reconnect delay.
         * @return the reconnect delay.
         */
        public long reconnectTime()
        {
            return (mReconnectTime);
        } // end of reconnectTime()

        public ProtocolFamily protocol()
        {
            return (mProtocol);
        } // end of protocol()

        //
        // end of Get methods.
        //-------------------------------------------------------
    } // end of class Remoteconnection

    /**
     * An application may create dispatch threads with a given
     * unique name and the following parameters:
     * <ul>
     *   <li>
     *     {@code Thread priority}: minimum,
     *     maximum, or normal thread priority.
     *   </li>
     *   <li>
     *     {@code Run Queue Type}: specifies whether run queue
     *     threads use a blocking, spinning, spin+park, or
     *     spin+yield technique to poll clients from the run
     *     queue.
     *   </li>
     *   <li>
     *     {@code Spin Limit}: if run queue type is either
     *     spin+park or spin+yield, then this is the number of
     *     times the dispatcher thread will poll the run queue
     *     before parking or yielding.
     *   </li>
     *   <li>
     *     {@code Park Time}: if run queue type is spin+park,
     *     then this is the number of nanoseconds the dispatcher
     *     thread will park after the spin limit is reached. When
     *     park returns, the thread will go back to spinning on
     *     poll.
     *   </li>
     *   <li>
     *     {@code Quantum}: nanosecond run quantum assigned to
     *     each client for this dispatcher group. If a client
     *     exhausts this quantum upon completing a task, the
     *     client is placed back on the end of the FIFO run queue
     *     and its quantum is reset to this value. Note that eBus
     *     does not preempt the client. It checks for quantum
     *     exhaustion when the client task returns.
     *   </li>
     *   <li>
     *     {@code Group thread count}: number of dispatcher
     *     threads in this group.
     *   </li>
     *   <li>
     *     {@code Default group flag}: {@code true} if this is
     *     the default dispatcher group. If a client is not
     *     assigned to any particular group, then it is assigned
     *     to the default group. If this flag is {@code true},
     *     then the {@code Classes} parameter is ignored. There
     *     may be only one default dispatcher group.
     *   </li>
     *   <li>
     *     {@code Classes}: a list of application classes.
     *     Clients which are instances of these classes are
     *     assigned to this dispatcher group. This parameter is
     *     valid only if this is not the default dispatcher
     *     group.
     *   </li>
     * </ul>
     * <h1>Special Dispatchers</h1>
     * There are two special, pre-defined eBus
     * {@code Dispatcher} names:
     * {@link DispatcherType#SWING "swing"} and
     * {@link DispatcherType#JAVAFX "javafx"} (case insensitive).
     * These {@code Dispatcher}s use the Swing/JavaFX GUI thread
     * to deliver eBus messages to a client. This means that the
     * client callback is free to update the GUI because the
     * callback code is running on the GUI thread.
     * <p>
     * Only two properties are supported by special
     * {@code Dispatchers}: {@link #DEFAULT_KEY} and
     * {@link #CLASSES_KEY}. All other properties are ignored by
     * special {@code Dispatchers}. The reason is that the
     * underlying GUI threads are implemented and configured by
     * the GUI package and cannot be altered by eBus.
     * </p>
     * <p>
     * It may be useful to define
     * the GUI thread Dispatcher as the default Dispatcher and
     * then create a separate eBus {@code Dispatcher} for
     * non-GUI classes. This way, an application class updating
     * the display will be assigned to the GUI thread without
     * needing to add that class to the GUI Dispatcher classes
     * property.
     * </p>
     */
    public static final class Dispatcher
        implements Comparable<Dispatcher>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * The unique dispatcher name.
         */
        private final String mName;

        /**
         * The dispatcher type. Defines the dispatch method
         * handle.
         */
        private final DispatcherType mType;

        /**
         * Specifies how the dispatcher acquires the next client
         * from the run queue: blocking, spinning, spin+park, or
         * spin+yield.
         */
        private final ThreadType mRunQueueType;

        /**
         * If {@link #mRunQueueType} is
         * {@link ThreadType#SPINPARK spin+park} or
         * {@link ThreadType#SPINYIELD spin+yield}, then spin
         * this many times on trying to acquire the next client
         * before parking/yielding.
         */
        private final long mSpinLimit;

        /**
         * If {@link #mRunQueueType} is
         * {@link ThreadType#SPINPARK spin+park}, then park for
         * this many nanoseconds before returning to spinning.
         */
        private final long mParkTime;

        /**
         * The dispatcher threads run at this priority. Must be
         * &ge; {@link Thread#MIN_PRIORITY} and
         * &le; {@link Thread#MAX_PRIORITY}.
         */
        private final int mPriority;

        /**
         * The nanosecond run quantum assigned to each client
         * in this run queue.
         */
        private final long mQuantum;

        /**
         * The dispatcher has this many threads.
         */
        private final int mNumThreads;

        /**
         * {@code true} if this dispatcher is the default for all
         * {@link EClient} instances not assigned to another,
         * specific dispatcher.
         */
        private final boolean mIsDefault;

        /**
         * Contains the classes assigned to this dispatcher. Will
         * be an empty array if {@link #mIsDefault} is
         * {@code true}.
         */
        private final Class<?>[] mClasses;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a Dispatcher group configuration based on
         * {@code builder}'s current setting.
         * @param builder contains the Dispatcher configuration
         * parameters.
         */
        /* package */ Dispatcher(final DispatcherBuilder builder)
        {
            this.mName = builder.mName;
            this.mType = builder.mType;
            this.mRunQueueType = builder.mRunQueueType;
            this.mSpinLimit = builder.mSpinLimit;
            this.mParkTime = builder.mParkTime;
            this.mPriority = builder.mPriority;
            this.mQuantum = builder.mQuantum;
            this.mNumThreads = builder.mNumThreads;
            this.mIsDefault = builder.mIsDefault;
            this.mClasses = builder.mClasses;
        } // end of Dispatcher(DispatcherBuilder)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Comparable Interface Implementation.
        //

        @Override
        public int compareTo(
            final Dispatcher o)
        {
            return (mName.compareTo(o.mName));
        } // end of compareTo(Dispatcher)

        //
        // end of Comparable Interface Implementation.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Object Method Overrides.
        //

        @Override
        public boolean equals(final Object o)
        {
            boolean retcode = (this == o);

            if (!retcode)
            {
                retcode = (mName.equals(((Dispatcher) o).mName));
            }

            return (retcode);
        } // end of equals(Object)

        @Override
        public int hashCode()
        {
            return (mName.hashCode());
        } // end of hashCode()

        @Override
        public String toString()
        {
            final Formatter retval = new Formatter();

            retval.format("[%s]%n", mName);
            retval.format("       priority: %s%n", mPriority);
            retval.format("     is default: %b%n", mIsDefault);

            return (retval.toString());
        } // end of toString()

        //
        // end of Object Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get methods.
        //

        /**
         * Returns the unique dispatcher name.
         * @return dispatcher thread name.
         */
        public String name()
        {
            return (mName);
        } // end of name()

        /**
         * Returns the dispatch method type.
         * @return dispatch method type.
         */
        public DispatcherType dispatchType()
        {
            return (mType);
        } // end of dispatchType()

        /**
         * Returns the thread type which defines how the next
         * client is acquired from the run queue.
         * @return run queue-acquisition type.
         */
        public ThreadType runQueueType()
        {
            return (mRunQueueType);
        } // end of runQueueType()

        /**
         * Returns the spin limit used by
         * {@link ThreadType#SPINPARK spin+park} and
         * {@link ThreadType#SPINYIELD spin+yield} thread types.
         * @return spin limit.
         */
        public long spinLimit()
        {
            return (mSpinLimit);
        } // end of spinLimit()

        /**
         * Returns the park time used by
         * {@link ThreadType#SPINPARK spin+park} thread type.
         * @return nanosecond park time.
         */
        public long parkTime()
        {
            return (mParkTime);
        } // end of parkTime()

        /**
         * Returns {@code true} if this user-defined dispatcher
         * is the default dispatcher.
         * @return {@code true} if the default dispatcher.
         */
        public boolean isDefault()
        {
            return (mIsDefault);
        } // end of isDefault()

        /**
         * Returns the classes assigned to this dispatcher.
         * Returned array may be empty but will not be
         * {@code null}.
         * @return dispatcher class array.
         */
        public Class<?>[] classes()
        {
            return (Arrays.copyOf(mClasses, mClasses.length));
        } // end of classes()

        /**
         * Returns the dispatcher thread priority.
         * @return thread priority.
         */
        public int priority()
        {
            return (mPriority);
        } // end of priority()

        /**
         * Returns the client run-time quantum assigned by this
         * dispatcher.
         * @return client run-time quantum.
         */
        public long quantum()
        {
            return (mQuantum);
        } // end of quantum()

        /**
         * Returns the number of threads in this dispatcher.
         * @return dispatcher thread count.
         */
        public int numberThreads()
        {
            return (mNumThreads);
        } // end of numberThreads()

        //
        // end of Get methods.
        //-------------------------------------------------------
    } // end of class Dispatcher

    /**
     * If a remote connection may be paused, then this is the
     * allowed pause configuration. When defined for a connection
     * acceptor, only the pause duration and maximum backlog size
     * properties are used. A connection initiator has all
     * properties used.
     * <p>
     * When a connection is paused, the actual connection is
     * dropped but all the subscriptions and advertisements are
     * kept in place. That means that system and application
     * messages destined for the other side are queued, ready
     * for transmit when the connection is resumed. While all
     * system messages are queued, the number of queued
     * application messages may be limited. When that limit
     * is reached, further application messages are discard as
     * per the configured {@link EConfigure.DiscardPolicy}.
     * </p>
     * <p>
     * This feature is targeted for eBus-based applications
     * running on mobile devices. Such devices cannot maintain
     * active connections for long without overheating and
     * draining the battery.
     * </p>
     */
    public static final class PauseConfig
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * If {@link #mCanPause} is {@code true}, then this is
         * the maximum allowed pause duration.
         */
        private final Duration mDuration;

        /**
         * If {@link #mCanPause} is {@code true}, then this is
         * the maximum allowed message backlog size.
         */
        private final int mMaxBacklogSize;

        /**
         * If {@link #mCanPause} is {@code true}, then this is
         * the message discard policy used when
         * {@link #mMaxBacklogSize} is breached.
         */
        private final DiscardPolicy mDiscardPolicy;

        /**
         * If a remote connection does not send or receive any
         * messages for this duration, the connection is paused.
         */
        private final Duration mIdleTime;

        /**
         * A remote connection remains connected for this
         * duration at most. Once this time limit is reached, the
         * connection is automatically paused - independent of
         * when the last message was sent or received.
         */
        private final Duration mMaxConnectTime;

        /**
         * When the pending transmit queue reaches this limit
         * (application messages only), then the client
         * connection is resumed. This feature is turned off
         * when set to zero.
         */
        private final int mResumeOnQueueLimit;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private PauseConfig(final PauseBuilder builder)
        {
            mDuration = builder.mDuration;
            mMaxBacklogSize = builder.mMaxBacklogSize;
            mDiscardPolicy = builder.mDiscardPolicy;
            mIdleTime = builder.mIdleTime;
            mMaxConnectTime = builder.mMaxConnectTime;
            mResumeOnQueueLimit = builder.mResumeOnBacklogSize;
        } // end of PauseConfig(PauseBuilder)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Object Method Overrides.
        //

        /**
         * Returns the pause configuration as text.
         * @return textual representation of the pause
         * configuration.
         */
        @Override
        public String toString()
        {
            final StringBuilder retval = new StringBuilder();

            retval.append("[duration=").append(mDuration)
                  .append(", backlog size=").append(mMaxBacklogSize);

            if (mDiscardPolicy != null)
            {
                  retval.append(", discard policy=")
                        .append(mDiscardPolicy);
            }

            retval.append(", idle time=").append(mIdleTime)
                  .append(", max connect time=").append(mMaxConnectTime)
                  .append(", resume-on-send=").append(mResumeOnQueueLimit)
                  .append("]");

            return (retval.toString());
        } // end of toString()

        //
        // end of Object Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the maximum allowed pause duration.
         * @return pause duration.
         */
        public Duration duration()
        {
            return (mDuration);
        } // end of duration()

        /**
         * Returns the maximum allowed message backlog size.
         * @return message backlog size.
         */
        public int maxBacklogSize()
        {
            return (mMaxBacklogSize);
        } // end of maxBacklogSize()

        /**
         * Returns the message discard policy used when the
         * maximum message backlog size is breached.
         * @return message discard policy.
         */
        public DiscardPolicy discardPolicy()
        {
            return (mDiscardPolicy);
        } // end of discardPolicy()

        /**
         * Returns the amount of time wherein no messages are
         * sent or received on the connection before the
         * connection is paused.
         * @return connection idle time.
         */
        public Duration idleTime()
        {
            return (mIdleTime);
        } // end of idleTime()

        /**
         * Returns the maximum time a connection may remain up
         * before being paused. Note that this pausing is
         * irrespective of the message arrival or departure.
         * @return maximum connection time between pauses.
         */
        public Duration maxConnectTime()
        {
            return (mMaxConnectTime);
        } // end of maxConnectTime()

        /**
         * Returns the transmit queue limit which triggers an
         * automatic client connection resumption. If zero is
         * returned, then this trigger is disabled.
         * @return resume on reaching transmit queue limit.
         */
        public int resumeOnQueueLimit()
        {
            return (mResumeOnQueueLimit);
        } // end of resumeOnBacklogSize()

        //
        // end of Get Methods.
        //-------------------------------------------------------
    } // end of class PauseConfig

    /**
     * Base class for {@link ServerBuilder} and
     * {@link ConnectionBuilder}, containing the properties
     * common to both.
     *
     * @param <T> builder subclass.
     *
     * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
     */
    @SuppressWarnings ("unchecked")
    public static abstract class AbstractBuilder<T extends AbstractBuilder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        /**
         * Unique service or connection name.
         */
        protected String mName;

        /**
         * Server-accepted or remote connection type is either
         * plain text TCP or SSL/TLS secure TCP.
         */
        private ConnectionType mConnectionType;

        /**
         * Maximum input buffer size. May be overridden by
         * {@link javax.net.ssl.SSLSession#getApplicationBufferSize()}
         * when using a secure TCP connection.
         */
        protected int mInputBufferSize;

        /**
         * Maximum output buffer size. May be override by
         * {@link javax.net.ssl.SSLSession#getPacketBufferSize()}
         * when using a secure TCP connection.
         */
        protected int mOutputBufferSize;

        /**
         * Code and decode messages in this byte order.
         */
        protected ByteOrder mByteOrder;

        /**
         * Maximum eBus message outbound queue size for remote
         * connections. When the queue outbound message count
         * exceeds this size, the remote connection is
         * automatically closed and the outbound messages
         * discarded.
         */
        protected int mMsgQueueSize;

        /**
         * Send a heartbeat message after this many milliseconds
         * of inbound message inactivity.
         */
        protected long mHbDelay;

        /**
         * The number of milliseconds the far-end has to respond
         * to a heartbeat. If the response is not received after
         * this many milliseconds, the remote connection is
         * automatically closed.
         */
        protected long mHbReplyDelay;

        /**
         * The SSL/TLS context used for a secure TCP connection.
         */
        protected SSLContext mSSLContext;

        /**
         * Set to {@code true} if remote connections may be
         * paused.
         */
        protected boolean mCanPause;

        /**
         * Contains the pause parameters. Will be {@code null} if
         * {@link #mCanPause} is {@code false}.
         */
        protected PauseConfig mPauseConfig;

        /**
         * Set to {@code true} if this builder is used when
         * loading a configuration. This flag is used to detect
         * when a secure TCP session is defined in a
         * configuration as opposed to built by an application.
         * If so, then the session definition is not required
         * to define the SSL context.
         */
        protected boolean mLoaderFlag;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new abstract configuration builder.
         */
        protected AbstractBuilder()
        {
            mName = null;
            mConnectionType = ConnectionType.TCP;
            mInputBufferSize = 0;
            mOutputBufferSize = 0;
            mByteOrder = DEFAULT_BYTE_ORDER;
            mMsgQueueSize = 0;
            mHbDelay = 0;
            mHbReplyDelay = 0;
            mSSLContext = null;
            mCanPause = false;
            mPauseConfig = null;
            mLoaderFlag = false;
        } // end of AbstractBuilder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the service/connection name.
         * @param name service/connection name.
         * @return {@code this} service/connection builder.
         * @throws ConfigException
         * if {@code name} is {@code null} or empty.
         */
        public final T name(final String name)
        {
            if (Strings.isNullOrEmpty(name))
            {
                throw (
                    new ConfigException.BadValue(
                        NAME_KEY, "name is null or empty"));
            }

            mName = name;

            return ((T) this);
        } // end of name(String)

        /**
         * Sets the underlying channel type.
         * @param connType channel type.
         * @return {@code this} service/connection builder.
         * @throws ConfigException
         * if {@code connType} is {@code null}.
         */
        public final T connectionType(final ConnectionType connType)
        {
            if (connType == null)
            {
                throw (
                    new ConfigException.BadValue(
                        CONN_TYPE_KEY, "connType is null"));
            }

            mConnectionType = connType;

            return ((T) this);
        } // end of connectionType(ConnectionType)

        /**
         * Sets the input buffer size for the connection. If
         * {@code size} is zero, then the default input buffer
         * size is used.
         * @param size connection input buffer size.
         * @return {@code this} service/connection builder.
         * @throws ConfigException
         * if {@code size} &lt; zero.
         */
        public final T inputBufferSize(final int size)
        {
            if (size < 0)
            {
                throw (
                    new ConfigException.BadValue(
                        INBUFFER_SIZE_KEY,
                        "input buffer size < 0"));
            }

            mInputBufferSize = size;

            return ((T) this);
        } // end of inputBufferSize(int)

        /**
         * Sets the output buffer size for the connection. If
         * {@code size} is zero, then the default output buffer
         * size is used.
         * @param size accepted connection output buffer size.
         * @return {@code this} service/connection builder.
         * @throws ConfigException
         * if {@code size} &lt; zero.
         */
        public final T outputBufferSize(final int size)
        {
            if (size < 0)
            {
                throw (
                    new ConfigException.BadValue(
                        OUTBUFFER_SIZE_KEY,
                        "output buffer size < 0"));
            }

            mOutputBufferSize = size;

            return ((T) this);
        } // end of outputBufferSize(int)

        /**
         * Sets the byte order used by the connection.
         * @param byteOrder connection serialize and
         * de-serialize messages using this byte order.
         * @return {@code this} service/connection builder.
         * @throws ConfigException
         * if {@code byteOrder} is {@code nuill}.
         */
        public final T byteOrder(final ByteOrder byteOrder)
        {
            if (byteOrder == null)
            {
                throw (
                    new ConfigException.BadValue(
                        BYTE_ORDER_KEY, "byteOrder is null"));
            }

            mByteOrder = byteOrder;

            return ((T) this);
        } // end of byteOrder(ByteOrder)

        /**
         * Sets the maximum queue size for the connection. When
         * the message queue size reaches this maximum, the
         * connection is automatically closed. If zero, then the
         * default maximum queue size is used.
         * @param size message queue maximum size.
         * @return {@code this} service/connection builder.
         * @throws ConfigException
         * if {@code size} &lt; zero.
         */
        public final T messageQueueSize(final int size)
        {
            if (size < 0)
            {
                throw (
                    new ConfigException.BadValue(
                        MSG_QUEUE_SIZE_KEY,
                        "message queue size < 0"));
            }

            mMsgQueueSize = size;

            return ((T) this);
        } // end of messageQueueSize(int)

        /**
         * Sets the heartbeat delay used for the connection. If
         * no message is received after {@code delay}
         * milliseconds, then a heartbeat is sent. If
         * {@code delay} is zero, then heartbeating is turned
         * off.
         * @param delay millisecond heartbeat rate.
         * @return {@code this} service/connection builder.
         * @throws ConfigException
         * if {@code delay} &lt; zero.
         */
        public final T heartbeatDelay(final long delay)
        {
            if (delay < 0)
            {
                throw (
                    new ConfigException.BadValue(
                        HB_DELAY_KEY, "heartbeat delay < 0"));
            }

            mHbDelay = delay;

            return ((T) this);
        } // end of heartbeatDelay(long)

        /**
         * Sets the heartbeat reply delay used for the
         * connection. The remote eBus application has this
         * many milliseconds to reply to a heartbeat. If no reply
         * is received after {@code delay} milliseconds, the
         * connection is closed. This value is ignored if the
         * heartbeat delay is zero.
         * @param delay millisecond heartbeat reply delay.
         * @return {@code this} service/connection builder.
         * @throws ConfigException
         * if {@code delay} &lt; zero.
         */
        public final T heartbeatReplyDelay(final long delay)
        {
            if (delay < 0)
            {
                throw (
                    new ConfigException.BadValue(
                        HB_REPLY_DELAY_KEY,
                        "heartbeat reply delay < 0"));
            }

            mHbReplyDelay = delay;

            return ((T) this);
        } // end of heartbeatReplyDelay(long)

        /**
         * Sets the SSL/TLS context used by a secure location.
         * This should be called only if the connection type
         * is set to {@link ConnectionType#SECURE_TCP}.
         * @param context secure TCP context.
         * @return {@code this} server/connection builder.
         * @throws ConfigException
         * if {@code context} is {@code null}.
         */
        public final T sslContext(final SSLContext context)
        {
            if (context == null)
            {
                throw (
                    new ConfigException.BadValue(
                        SSL_CONTEXT_KEY, "context is null"));
            }

            mSSLContext = context;

            return ((T) this);
        } // end of sslContext(SSLContext)

        /**
         * Sets a flag specifying whether accepted connections
         * may be paused ({@code true}) or not.
         * @param flag pause-able connection flag.
         * @return {@code this} connection builder.
         */
        public final T canPause(final boolean flag)
        {
            mCanPause = flag;

            return ((T) this);
        } // end of canPause(boolean)

        /**
         * Set the pause configuration for the connection.
         * <p>
         * <strong>{@link #canPause(boolean) canPause(boolean)}</strong>
         * must be set to {@code true} <em>before</em> setting
         * the pause configuration.
         * </p>
         * @param pc pause configuration
         * @return {@code this} server/connection builder.
         * @throws ConfigException
         * if {@link #canPause(boolean)} returns {@code false}
         * (meaning the connection cannot be pause) or if
         * {@code pc} is {@code null}.
         */
        public final T pauseConfig(final PauseConfig pc)
        {
            if (!mCanPause)
            {
                throw (
                    new ConfigException.Generic(
                        "connection cannot be paused"));
            }

            if (pc == null)
            {
                throw (
                    new ConfigException.BadValue(
                        PAUSE_KEY, "pause config is null"));
            }

            mPauseConfig = pc;

            return ((T) this);
        } // end of pauseDuration(Duration)

        /**
         * Copies the values found in {@code config} to this
         * configuration. This method is used to modify an
         * existing immutable configuration.
         * @param config copy this configuration's values.
         * @return {@code this} server/connection builder.
         */
        protected T configuration(final AbstractConfig config)
        {
            mName = config.name();
            mConnectionType = config.connectionType();
            mInputBufferSize = config.inputBufferSize();
            mOutputBufferSize = config.outputBufferSize();
            mByteOrder = config.byteOrder();
            mMsgQueueSize = config.messageQueueSize();
            mHbDelay = config.heartbeatDelay();
            mHbReplyDelay = config.heartbeatReplyDelay();
            mSSLContext = config.sslContext();
            mCanPause = config.canPause();
            mPauseConfig = config.pauseConfiguration();

            return ((T) this);
        } // end of configuration(AbstractConfig)

        /**
         * Sets the loader flag to the given value.
         * @param flag loader flag value.
         * @return {@code this} server/connection builder.
         */
        protected final T loaderFlag(final boolean flag)
        {
            mLoaderFlag = flag;

            return ((T) this);
        } // end of loaderFlag(flag)

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Validates builder arguments are correctly set,
         * throwing the appropriate exception if validation
         * fails.
         * @throws ConfigException
         * if service name or connection type is not set, if
         * SSL context not provided for a secure connection, or
         * if SSL context provided for a non-secure connection.
         */
        protected void validate()
        {
            if (mName == null)
            {
                throw (
                    new ConfigException.BadValue(
                        NAME_KEY, "service name is not set"));
            }

            // Check if an SSL context was provided for a secure
            // TCP connection.
            if (mConnectionType == ConnectionType.SECURE_TCP &&
                !mLoaderFlag &&
                mSSLContext == null)
            {
                throw (
                    new ConfigException.BadValue(
                        SSL_CONTEXT_KEY,
                        "SSL context not provided for secure connection"));
            }

            // Check if an SSL context was provided for a
            // non-secure connection.
            if (mConnectionType != ConnectionType.SECURE_TCP &&
                mSSLContext != null)
            {
                throw (
                    new ConfigException.BadValue(
                        SSL_CONTEXT_KEY,
                        "SSL context provided for non-secure connection"));
            }

            if (mCanPause && mPauseConfig == null)
            {
                throw (
                    new ConfigException.Missing(
                        "pause configuration not set"));
            }
        } // end of validate()
    } // end of class AbstractBuilder

    /**
     * Constructs an {@link EConfigure.Service} instance based on
     * the parameters set via this builder's API. The minimally
     * allowed configuration is the service name, connection type
     * and port. The remaining parameters are set to the
     * following defaults:
     * <ul>
     *   <li>
     *     connection type: {@link ConnectionType#TCP} - plain
     *     text TCP connection.
     *   </li>
     *   <li>
     *     address filter: {@code null} - no address filter
     *     applied.
     *   </li>
     *   <li>
     *     input buffer size: zero - use system default buffer
     *     size.
     *   </li>
     *   <li>
     *     output buffer size: zero - use system default buffer
     *     size.
     *   </li>
     *   <li>
     *     byte order: little-endian.
     *   </li>
     *   <li>
     *     maximum message queue size: zero - unlimited queue
     *     size.
     *   </li>
     *   <li>
     *     service selector thread:
     *     {@code AsyncChannel.defaultSelector}.
     *   </li>
     *   <li>
     *     accepted connection select thread:
     *     {@code AsyncChannel.defaultSelector}.
     *   </li>
     *   <li>
     *     heartbeat millisecond delay: zero - heartbeating off.
     *   </li>
     *   <li>
     *     heartbeat millisecond reply delay: zero - wait
     *     indefinitely for a reply.
     *   </li>
     * </ul>
     * <p>
     * If either the service name or port are not set, then
     * {@link #build()} will throw an exception.
     * </p>
     * <h1>Example building an {@code EServer}</h1>
     * <pre><code>final AddressFilter filter = ...;
final SSLContext secureContext = ...;
final EConfigure.ServerBuilder builder = EConfigure.serverBuilder();

EServer.openServer(builder.name("AppServer")
                          .port(6789)
                          .connectionType(EConfigure.ConnectionType.SECURE_TCP)
                          .sslContext(secureContext)
                          .addressFilter(filter)
                          .inputBufferSize(1_024)
                          .outputBufferSize(1_024)
                          .byteOrder(ByteOrder.BIG_ENDIAN)
                          .messageQueueSize(10_000)
                          .serviceSelector("svcSelector")
                          .connectionSelector("connSelector")
                          .heartbeatDelay(60_000L)
                          .heartbeatReplyDelay(30_000L)
                          .build());</code></pre>
     *
     * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
     *
     * @see ConnectionBuilder
     */
    public static final class ServerBuilder
        extends AbstractBuilder<ServerBuilder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private int mPort;
        private AddressFilter mAddressFilter;
        private String mServiceSelector;
        private String mConnSelector;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new server configuration builder.
         */
        private ServerBuilder()
        {
            mPort = 0;
            mAddressFilter = null;
            mServiceSelector =
                (ENetConfigure.defaultSelector()).name();
            mConnSelector = mServiceSelector;
        } // end of ServerBuilder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Copies in the settings from {@code config} to this
         * builder. This method is provided for making changes to
         * an existing configuration since
         * {@link EConfigure.Service} is immutable.
         * <p>
         * if {@code config} is {@code null}, then nothing is
         * done and the builder remains unchanged.
         * </p>
         * @param config copy settings from this configuration.
         * @return {@code this} service builder.
         */
        public ServerBuilder configuration(final EConfigure.Service config)
        {
            if (config != null)
            {
                super.configuration(config);

                mPort = config.port();
                mAddressFilter = config.addressFilter();
                mServiceSelector = config.serviceSelector();
                mConnSelector = config.connectionSelector();
            }

            return (this);
        } // end of configuration(EConfigure.Service)

        /**
         * Sets the service TCP port.
         * @param port service TCP port.
         * @return {@code this} service builder.
         * @throws ConfigException
         * if {@code port} is not a valid TCP port.
         */
        public ServerBuilder port(final int port)
        {
            if (port < MIN_PORT || port > MAX_PORT)
            {
                throw (
                    new ConfigException.BadValue(
                        PORT_KEY,
                        "invalid port (" + port + ")"));
            }

            mPort = port;

            return (this);
        } // end of port(int)

        /**
         * Sets the optional service address filter.
         * @param filter the optional address filter. May be
         * {@code null}.
         * @return {@code this} service builder.
         */
        public ServerBuilder addressFilter(final AddressFilter filter)
        {
            mAddressFilter = filter;

            return (this);
        } // end of addressFilter(AddressFilter)

        /**
         * Sets the selector used for the service connection.
         * @param selector eBus selector name.
         * @return {@code this} service builder.
         * @throws ConfigException
         * if {@code name} is {@code null} or empty or is not a
         * known selector.
         */
        public ServerBuilder serviceSelector(final String selector)
        {
            if (Strings.isNullOrEmpty(selector))
            {
                throw (
                    new ConfigException.BadValue(
                        SVC_SELECTOR_KEY,
                        "service selector is null or empty"));
            }

            if (!ENetConfigure.isKnownSelector(selector))
            {
                throw (
                    new ConfigException.BadValue(
                        SVC_SELECTOR_KEY,
                        "\"" +
                        selector +
                        "\" unknown service selector"));
            }

            mServiceSelector = selector;

            return (this);
        } // end of serviceSelector(String)

        /**
         * Sets the selector used for accepted TCP connections.
         * @param selector eBus selector name.
         * @return {@code this} service builder.
         * @throws ConfigException
         * if {@code name} is {@code null} or empty or not a
         * known selector.
         */
        public ServerBuilder connectionSelector(final String selector)
        {
            if (Strings.isNullOrEmpty(selector))
            {
                throw (
                    new ConfigException.BadValue(
                        CONN_SELECTOR_KEY,
                        "connection selector is null or empty"));
            }

            if (!ENetConfigure.isKnownSelector(selector))
            {
                throw (
                    new ConfigException.BadValue(
                        CONN_SELECTOR_KEY,
                        "\"" +
                        selector +
                        "\" unknown connection selector"));
            }

            mConnSelector = selector;

            return (this);
        } // end of connectionSelector(String)

        /**
         * Returns the eBus service configuration built from the
         * previously set parameters.
         * @return an eBus service configuration.
         * @throws ConfigException
         * if any service name or service port is not set.
         */
        public EConfigure.Service build()
        {
            validate();

            return (new EConfigure.Service(this));
        } // end of build()

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Validates the builder parameters. Called for effect
         * only.
         * @throws ConfigException
         * if a required parameter is not set or is not set to a
         * valid value with respect to another parameter.
         */
        @Override
        protected void validate()
        {
            super.validate();

            if (mPort == 0)
            {
                throw (
                    new ConfigException.BadValue(
                        PORT_KEY, "service port not set"));
            }
        } // end of validate()
    } // end of class ServerBuilder

    /**
     * Constructs an {@link EConfigure.RemoteConnection} instance
     * based on the parameters set via the builder's API. The
     * minimally allowed configuration is the connection name and
     * address. The remaining parameters are set to the following
     * defaults:
     * <ul>
     *   <li>
     *     connection type: {@link ConnectionType#TCP} - plain
     *     text TCP connection.
     *   </li>
     *   <li>
     *     bind port: {@code ERemoteApp.ANY_PORT}.
     *   </li>
     *   <li>
     *     input buffer size: zero - use system default buffer
     *     size.
     *   </li>
     *   <li>
     *     output buffer size: zero - use system default buffer
     *     size.
     *   </li>
     *   <li>
     *     byte order: little-endian.
     *   </li>
     *   <li>
     *     maximum message queue size: zero - unlimited queue
     *     size.
     *   </li>
     *   <li>
     *     connection selector thread:
     *     {@code AsyncChannel.defaultSelector}.
     *   </li>
     *   <li>
     *     reconnect flag and delay: reconnection is turned off and
     *     delay set to zero.
     *   </li>
     *   <li>
     *     heartbeat millisecond delay: zero - heartbeating off.
     *   </li>
     *   <li>
     *     heartbeat millisecond reply delay: zero - wait
     *     indefinitely for a reply.
     *   </li>
     * </ul>
     * <h1>Example building an {@code ERemoteApp}</h1>
     * <pre><code>final InetSocketAddress address = new InetSocketAddress(InetAddress.getLocalHost(), 12345);
final SSLContext secureContext = ...;
final EConfigure.ConnectionBuilder builder = EConfigure.connectionBuilder();

ERemoteApp.openConnection(builder.name("Conn0")
                                 .address(address)
                                 .bindPort(0)
                                 .connectionType(EConfigure.ConnectionType.SECURE_TCP)
                                 .sslContext(secureContext)
                                 .inputBufferSize(4_996)
                                 .outputBufferSize(8_192)
                                 .byteOrder(ByteOrder.BIG_ENDIAN)
                                 .messageQueueSize(0)
                                 .selector("connSelector")
                                 .reconnect(true)
                                 .reconnectDelay(500L)
                                 .heartbeatDelay(0L)
                                 .heartbeatReplyDelay(0L)
                                 .build());</code></pre>
     *
     * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
     *
     * @see ServerBuilder
     */
    public static final class ConnectionBuilder
        extends AbstractBuilder<ConnectionBuilder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private InetSocketAddress mAddress;
        private int mBindPort;
        private String mSelector;
        private boolean mReconnectFlag;
        private long mReconnectTime;
        private ProtocolFamily mProtocol;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new eBus connection configuration builder.
         */
        private ConnectionBuilder()
        {
            mAddress = null;
            mBindPort = ANY_PORT;
            mSelector = (ENetConfigure.defaultSelector()).name();
            mReconnectFlag = false;
            mReconnectTime = 0L;
            mProtocol = StandardProtocolFamily.INET;
        } // end of ConnectionBuilder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Copies in the settings from {@code config} to this
         * builder. This method is provided for making changes to
         * an existing configuration since
         * {@link EConfigure.RemoteConnection} is immutable.
         * <p>
         * if {@code config} is {@code null}, then nothing is
         * done and the builder remains unchanged.
         * </p>
         * @param config copy settings from this configuration.
         * @return {@code this} connection builder.
         */
        public ConnectionBuilder configuration(final EConfigure.RemoteConnection config)
        {
            if (config != null)
            {
                super.configuration(config);

                mAddress = config.address();
                mBindPort = config.bindPort();
                mSelector = config.selector();
                mReconnectFlag = config.reconnectFlag();
                mReconnectTime = config.reconnectTime();
            }

            return (this);
        } // end of configuration(EConfigure.RemoteConnection)

        /**
         * Set the connection address.
         * @param address connection address.
         * @return {@code this} connection builder.
         * @throws ConfigException
         * if {@code address} is {@code null}.
         */
        public ConnectionBuilder address(final InetSocketAddress address)
        {
            if (address == null)
            {
                throw (
                    new ConfigException.BadValue(
                        HOST_KEY, "address is null"));
            }

            mAddress = address;

            return (this);
        } // end of address(InetSocketAddress)

        /**
         * Bind the connection local port to this value. If
         * {@code port} is set to {@link ENetConfigure#ANY_PORT},
         * then the connection local port is set to any available
         * value.
         * @param port connection local bind port.
         * @return {@code this} connection builder.
         * @throws ConfigException
         * if {@code port} is either &lt;
         * {@link ENetConfigure#ANY_PORT} or &gt;
         * {@link ENetConfigure#MAX_PORT}.
         */
        public ConnectionBuilder bindPort(final int port)
        {
            if (port < ANY_PORT || port > MAX_PORT)
            {
                throw (
                    new ConfigException.BadValue(
                        BIND_PORT_KEY,
                        "invalid bind port (" + port + ")"));
            }

            mBindPort = port;

            return (this);
        } // end of bindPort(int)

        /**
         * Sets the selector used for the connection.
         * @param selector eBus selector name.
         * @return {@code this} connection builder.
         * @throws ConfigException
         * if {@code name} is {@code null}, an empty string or
         * not a known selector.
         */
        public ConnectionBuilder selector(final String selector)
        {
            if (Strings.isNullOrEmpty(selector))
            {
                throw (
                    new ConfigException.BadValue(
                        SELECTOR_KEY,
                        "selector is null or empty"));
            }

            if (!ENetConfigure.isKnownSelector(selector))
            {
                throw (
                    new ConfigException.BadValue(
                        SELECTOR_KEY,
                        "\"" + selector + "\" unknown selector"));
            }

            mSelector = selector;

            return (this);
        } // end of selector(String)

        /**
         * Sets the reconnect flag to the given value.
         * {@code true} means the connection will be
         * re-established if lost; {@code false} means a lost
         * connection is left down.
         * @param flag establish lost connection flag.
         * @return {@code this} connection builder.
         *
         * @see #reconnectDelay(long)
         */
        public ConnectionBuilder reconnect(final boolean flag)
        {
            mReconnectFlag = flag;

            return (this);
        } // end of reconnect(boolean)

        /**
         * Sets the reconnect delay to the given value. This value
         * is used on if the reconnect flag is {@code true}.
         * @param time reconnect a lost connect after this
         * millisecond delay.
         * @return {@code this} connection builder.
         * @throws ConfigException
         * if {@code time} &lt; zero.
         *
         * @see #reconnect(boolean)
         */
        public ConnectionBuilder reconnectDelay(final long time)
        {
            if (time < 0L)
            {
                throw (
                    new ConfigException.BadValue(
                        RECONNECT_DELAY_KEY,
                        "reconnect time < 0"));
            }

            mReconnectTime = time;

            return (this);
        } // end of reconnectDelay(long)

        public ConnectionBuilder protocol(final ProtocolFamily protocol)
        {
            mProtocol =
                Objects.requireNonNull(
                    protocol, "protocol is null");

            return (this);
        } // end of protocol(final ProtocolFamily protocol)

        /**
         * Returns the eBus connection configuration built from
         * the previously set parameters.
         * @return an eBus connection configuration.
         * @throws ConfigException
         * if either connection name or address is not set or if
         * reconnect flag is set to {@code true} but the
         * reconnect delay is not set.
         */
        public EConfigure.RemoteConnection build()
        {
            validate();

            return (new EConfigure.RemoteConnection(this));
        } // end of build()

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Validates the builder parameters. Called for effect
         * only.
         * @throws ConfigException
         * if a required parameter is not set or if a parameter
         * is not set to a valid value with respect to another
         * parameter.
         */
        @Override
        protected void validate()
        {
            super.validate();

            if (mName == null)
            {
                throw (
                    new ConfigException.BadValue(
                        NAME_KEY, "connection name not set"));
            }

            if (mAddress == null)
            {
                throw (
                    new ConfigException.BadValue(
                        HOST_KEY, "address not set"));
            }

            // If the reconnect flag is set, the the reconnect delay
            // must also be set.
            if (mReconnectFlag && mReconnectTime <= 0L)
            {
                throw (
                    new ConfigException.BadValue(
                        RECONNECT_DELAY_KEY,
                        "reconnect time == zero"));
            }
        } // end of validate()
    } // end of class ConnectionBuilder

    /**
     * Constructs an {@link EConfigure.Dispatcher} configuration
     * instance based on the parameters set via the builder's
     * API. The minimally allowed configuration is the Dispatcher
     * name and Dispatcher type if that type is either
     * {@link DispatcherType#SWING} or
     * {@link DispatcherType#JAVAFX}. For a
     * {@link DispatcherType#EBUS} Dispatcher type, then the
     * thread type must be provided. If this Dispatcher is
     * <em>not</em> marked as the default, then the classes
     * assigned to this Dispatcher must be provided.
     * <p>
     * The remaining properties are set to the following
     * defaults:
     * </p>
     * <ul>
     *   <li>
     *     spin limit: {@link EConfigure#DEFAULT_SPIN_LIMIT}.
     *   </li>
     *   <li>
     *     park time: {@link EConfigure#DEFAULT_PARK_TIME}.
     *   </li>
     *   <li>
     *     priority: {@link Thread#NORM_PRIORITY}.
     *   </li>
     *   <li>
     *     quantum: {@link EConfigure#DEFAULT_QUANTUM}.
     *   </li>
     *   <li>
     *     thread count: {@link EConfigure#DEFAULT_NUMBER_THREADS}.
     *   </li>
     *   <li>
     *     default Dispatcher: {@code false}.
     *   </li>
     * </ul>
     * <p>
     * <strong>Note:</strong> there is no example of how to
     * build a {@code Dispatcher} in application code because
     * eBus dispatchers must be created on system start. Once the
     * application {@code main} method is reached, it is too late
     * to create a dispatcher.
     * </p>
     *
     * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
     */
    public static final class DispatcherBuilder
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private String mName;
        private DispatcherType mType;
        private ThreadType mRunQueueType;
        private long mSpinLimit;
        private long mParkTime;
        private int mPriority;
        private long mQuantum;
        private int mNumThreads;
        private boolean mIsDefault;
        private Class<?>[] mClasses;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new dispatcher builder instance used to
         * construct an {@link EConfigure.Dispatcher} instance.
         */
        private DispatcherBuilder()
        {
            mName = null;
            mType = null;
            mRunQueueType = null;
            mSpinLimit = 0L;
            mParkTime = 0L;
            mPriority = Thread.NORM_PRIORITY;
            mQuantum = EConfigure.DEFAULT_QUANTUM;
            mNumThreads = EConfigure.DEFAULT_NUMBER_THREADS;
            mIsDefault = false;
            mClasses = new Class<?>[0];
        } // end of DispatcherBuilder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Copies in the settings from {@code config} to this
         * builder. This method is provided for making changes to
         * an existing configuration since
         * {@link EConfigure.Dispatcher} is immutable.
         * <p>
         * if {@code config} is {@code null}, then nothing is
         * done and the builder remains unchanged.
         * </p>
         * @param config copy settings from this configuration.
         * @return {@code this} dispatcher builder.
         */
        public DispatcherBuilder configuration(final EConfigure.Dispatcher config)
        {
            if (config != null)
            {
                mName = config.name();
                mType = config.dispatchType();
                mRunQueueType = config.runQueueType();
                mSpinLimit = config.spinLimit();
                mParkTime = config.parkTime();
                mPriority = config.priority();
                mQuantum = config.quantum();
                mNumThreads = config.numberThreads();
                mIsDefault = config.isDefault();
                mClasses = config.classes();
            }

            return (this);
        } // end of configuration(final EConfigure.Dispatcher config)

        /**
         * Sets the dispatcher name.
         * @param name dispatcher name.
         * @return {@code this} dispatcher builder.
         * @throws ConfigException
         * if {@code name} is null or empty.
         */
        public DispatcherBuilder name(final String name)
        {
            if (Strings.isNullOrEmpty(name))
            {
                throw (
                    new ConfigException.BadValue(
                        NAME_KEY, "name is null or empty"));
            }

            mName = name;

            return (this);
        } // end of name(String)

        /**
         * Sets the dispatcher type.
         * @param type dispatcher type.
         * @return {@code this} dispatcher builder.
         * @throws ConfigException
         * if {@code type} is {@code null} or an unknown
         * dispatcher type.
         */
        public DispatcherBuilder dispatcherType(final DispatcherType type)
        {
            if (type == null)
            {
                throw (
                    new ConfigException.BadValue(
                        DISPATCHER_TYPE_KEY,
                        "type is null or unknown"));
            }

            mType = type;

            return (this);
        } // end of dispatcherType(DispatcherType)

        /**
         * Sets the run queue thread type. This setting is
         * ignored if the Dispatcher type is
         * {@link DispatcherType#SWING} or
         * {@link DispatcherType#JAVAFX}.
         * @param type thread type.
         * @return {@code this} dispatcher builder.
         * @throws ConfigException
         * if {@code type} is {@code null} or an unknown thread
         * type.
         */
        public DispatcherBuilder threadType(final ThreadType type)
        {
            if (type == null)
            {
                throw (
                    new ConfigException.BadValue(
                        THREAD_TYPE_KEY, "type is null or unknown"));
            }

            mRunQueueType = type;

            return (this);
        } // end of threadType(ThreadType)

        /**
         * Sets the {@link ThreadType#SPINPARK} or
         * {@link ThreadType#SPINYIELD} spin limit. This setting
         * is ignored for any other run queue thread type.
         * @param limit spin limit.
         * @return {@code this} dispatcher builder.
         * @throws ConfigException
         * if {@code limit} &lt; zero.
         */
        public DispatcherBuilder spinLimit(final long limit)
        {
            if (limit < 0L)
            {
                throw (
                    new ConfigException.BadValue(
                        SPIN_LIMIT_KEY, "limit < zero"));
            }

            mSpinLimit = limit;

            return (this);
        } // end of spinLimit(long)

        /**
         * Sets the {@link ThreadType#SPINPARK} nanosecond park
         * time. This setting is ignored for any other run queue
         * thread type.
         * @param time nanosecond park time.
         * @return {@code this} dispatcher builder.
         * @throws ConfigException
         * if {@code time} &lt; zero.
         */
        public DispatcherBuilder parkTime(final long time)
        {
            if (time < 0L)
            {
                throw (
                    new ConfigException.BadValue(
                        PARK_TIME_KEY, "time < zero"));
            }

            mParkTime = time;

            return (this);
        } // end of parkTime(long)

        /**
         * Sets the run queue thread priority. Must be &ge;
         * {@link Thread#MIN_PRIORITY} and &le;
         * {@link Thread#MAX_PRIORITY}. This setting is ignored
         * if the Dispatcher type is {@link DispatcherType#SWING}
         * or {@link DispatcherType#JAVAFX}.
         * @param priority assigned thread priority for run queue
         * threads.
         * @return {@code this} dispatcher builder.
         * @throws ConfigException
         * if {@code priority} &lt; zero or &gt;
         * {@code Thread.MAX_PRIORITY}.
         */
        public DispatcherBuilder priority(final int priority)
        {
            if (priority < 0 || priority > Thread.MAX_PRIORITY)
            {
                throw (
                    new ConfigException.BadValue(
                        PRIORITY_KEY, "priority out of bounds"));
            }

            mPriority = priority;

            return (this);
        } // end of priority(int)

        /**
         * Sets the run quantum assigned to each eBus client.
         * @param quantum run quantum.
         * @return {@code this} dispatcher builder.
         * @throws ConfigException
         * if {@code quantum} &lt; zero.
         */
        public DispatcherBuilder quantum(final long quantum)
        {
            if (quantum < 0L)
            {
                throw (
                    new ConfigException.BadValue(
                        QUANTUM_KEY, "quantum < zero"));
            }

            mQuantum = quantum;

            return (this);
        } // end of quantum(long)

        /**
         * Sets the number of threads used by the run queue. This
         * setting is ignored if the Dispatcher type is
         * {@link DispatcherType#SWING} or
         * {@link DispatcherType#JAVAFX}.
         * @param numThreads number of run queue threads.
         * @return {@code this} dispatcher builder.
         * @throws ConfigException
         * if {@code numThreads} &lt; zero.
         */
        public DispatcherBuilder numberThreads(final int numThreads)
        {
            if (numThreads < 0)
            {
                throw (
                    new ConfigException.BadValue(
                        NUM_THREADS_KEY, "numThreads < zero"));
            }

            mNumThreads = numThreads;

            return (this);
        } // end of numberThreads(int)

        /**
         * If {@code flag} is {@code true}, then marks the
         * dispatcher as the default dispatcher.
         * @param flag {@code true} marks the dispatcher as the
         * default.
         * @return {@code this} dispatcher builder.
         */
        public DispatcherBuilder isDefault(final boolean flag)
        {
            mIsDefault = flag;

            return (this);
        } // end of isDefault(boolean)

        /**
         * Lists the classes assigned to the dispatcher. If the
         * default dispatcher, this list is ignored and
         * {@code classes} may be {@code null} or empty. If the
         * dispatcher is <em>not</em> default, then
         * {@code classes} must be defined as a non-{@code null},
         * non-empty array.
         * <p>
         * This parameter is checked when the dispatcher is
         * {@link #build() built} since the validation depends on
         * whether the dispatcher is default or not.
         * </p>
         * @param classes eBus client classes assigned to the
         * Dispatcher.
         * @return {@code this} dispatcher builder.
         */
        public DispatcherBuilder classes(final Class<?>[] classes)
        {
            if (classes == null)
            {
                mClasses = new Class<?>[0];
            }
            else
            {
                mClasses =
                    Arrays.copyOf(classes, classes.length);
            }

            return (this);
        } // end of classes(Class<?>)

        /**
         * Returns the eBus Dispatcher configuration built from
         * the previously set parameters.
         * @return an eBus Dispatcher configuration.
         * @throws ConfigException
         * if either the Dispatcher name, Dispatcher type, or run
         * queue thread type are not set, or if the Dispatcher is
         * not marked as default and no classes were provided.
         */
        public EConfigure.Dispatcher build()
        {
            validate();

            return (new EConfigure.Dispatcher(this));
        } // end of build()

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Validates the builder parameters. Called for effect
         * only.
         * @throws ConfigException
         * if a required parameter is not set or if a parameter
         * is not set to a valid value with respect to another
         * parameter.
         */
        private void validate()
        {
            if (mName == null)
            {
                throw (
                    new ConfigException.BadValue(
                        NAME_KEY, "name not set"));
            }

            if (mType == null)
            {
                throw (
                    new ConfigException.BadValue(
                        DISPATCHER_TYPE_KEY,
                        "dispatcher type not set"));
            }

            if (mRunQueueType == null)
            {
                throw (
                    new ConfigException.BadValue(
                        THREAD_TYPE_KEY,
                        "thread type not set"));
            }

            // Is this a spin+park Dispatcher?
            if (mRunQueueType == ThreadType.SPINPARK)
            {
                // Yes. Are spin limit and park time set?
                if (mSpinLimit <= 0L)
                {
                    throw (
                        new ConfigException.BadValue(
                            SPIN_LIMIT_KEY,
                            "spin limit not set for spin+park thread type"));
                }

                if (mParkTime <= 0L)
                {
                    throw (
                        new ConfigException.BadValue(
                            PARK_TIME_KEY,
                            "park limit not set for spin+park thread type"));
                }
            }
            // Is this a spin+yield Dispatcher?
            else if (mRunQueueType == ThreadType.SPINYIELD)
            {
                // Yes. Is spin limit set?
                if (mSpinLimit <= 0L)
                {
                    throw (
                        new ConfigException.BadValue(
                            SPIN_LIMIT_KEY,
                            "spin limit not set for spin+yield thread type"));
                }
            }

            // Is this a default dispatcher?
            // Are classes provided?
            if (!mIsDefault &&
                (mClasses == null || mClasses.length == 0))
            {
                // Non-default dispatcher but no classes given.
                throw (
                    new ConfigException.BadValue(
                        CLASSES_KEY,
                        "classes not set for non-default dispatcher"));
            }
        } // end of validate()
    } // end of class DispatcherBuilder

    /**
     * Constructs a {@link PauseConfig} instance based on the
     * parameters set via the builder API. Configuration settings
     * are based on the connection role. Pause duration and
     * maximum backlog size may be set for both connection
     * acceptor and initiator. The remaining settings (discard
     * policy, idle time, maximum connection time, and
     * resume-on-backlog-size) may only be set for initiators.
     * <p>
     * Properties depend on the connection role. For acceptor,
     * only the pause duration (required) and maximum backlog
     * size (optional) may be set. For initiator, all properties
     * may be used where pause duration and maximum connect time
     * are required and the remaining properties are optional.
     * </p>
     * <h1>Example building a {@link PauseConfig} - acceptor</h1>
     * <pre><code> final EConfigure.PauseBuilder builder = new EConfigure.pauseBuilder(EConfigure.ConnectionRole.Acceptor);
     * final EConfigure.PauseConfig pauseConfig = builder.duration(Duration.ofMinutes(5L)
     *                                                   .maxBacklogSize(10)
     *                                                   .build();</code></pre>
     * <h1>Example building a {@link PauseConfig} - initiator</h1>
     * <pre><code> final EConfigure.PauseBuilder builder = new EConfigure.pauseBuilder(EConfigure.ConnectionRole.Acceptor);
     * final EConfigure.PauseConfig pauseConfig = builder.duration(Duration.ofMinutes(5L)
     *                                                   .maxBacklogSize(20)
     *                                                   .discardPolicy(EConfigure.DiscardPolicy.OLDEST_FIRST)
     *                                                   .maxConnectionTime(Duration.ofSeconds(30L))
     *                                                   .resumeOnBacklogSize(5)
     *                                                   .build();</code></pre>
     */
    public static final class PauseBuilder
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * The role defines which pause properties are required,
         * optional, and not allowed.
         */
        private final ConnectionRole mRole;

        /**
         * If {@link #mCanPause} is {@code true}, then this is
         * the maximum allowed pause duration. Once this duration
         * is reached the connection initiator attempts to
         * resume the connection.
         * <p>
         * This value must be set for both connection acceptor
         * and initiator.
         * </p>
         */
        private Duration mDuration;

        /**
         * If {@link #mCanPause} is {@code true}, then this is
         * the maximum allowed message backlog size. If set to
         * zero then the message backlog size is unlimited.
         */
        private int mMaxBacklogSize;

        /**
         * If {@link #mCanPause} is {@code true}, then this is
         * the message discard policy used when
         * {@link #mMaxBacklogSize} is breached.
         */
        private DiscardPolicy mDiscardPolicy;

        /**
         * If a remote connection does not send or receive any
         * messages for this duration, the connection is paused.
         */
        private Duration mIdleTime;

        /**
         * A remote connection remains connected for this
         * duration at most. Once this time limit is reached, the
         * connection is automatically paused - independent of
         * when the last message was sent or received.
         */
        private Duration mMaxConnectTime;

        /**
         * Resume when transmit queue reaches this limit. This
         * can only be true for the client side.
         */
        private int mResumeOnBacklogSize;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a pause configuration builder for the
         * specified connection role.
         * @param role either a connection acceptor or a
         * connection initiator.
         */
        private PauseBuilder(final ConnectionRole role)
        {
            mRole = role;
            mDuration = Duration.ZERO;
            mMaxBacklogSize = 0;
            mIdleTime = Duration.ZERO;
            mMaxConnectTime = Duration.ZERO;
            mResumeOnBacklogSize = 0;
        } // end of PauseBuilder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the pause duration and returns {@code this}
         * pause configuration builder. This value must be set
         * for both acceptor and initiator connections.
         * @param duration pause duration.
         * @return {@code this PauseBuilder}.
         * @throws ConfigException
         * if {@code duration} is {@code null} or is &le; zero.
         */
        public PauseBuilder duration(final Duration duration)
        {
            if (duration == null)
            {
                throw (
                    new ConfigException.BadValue(
                        PAUSE_DURATION_KEY, "duration is null"));
            }

            if (duration.compareTo(Duration.ZERO) <= 0)
            {
                throw (
                    new ConfigException.BadValue(
                        PAUSE_DURATION_KEY, "duration <= 0"));
            }

            mDuration = duration;

            return (this);
        } // end of duration(final Duration duration)

        /**
         * Sets the maximum backlog size and returns {@code this}
         * pause configuration builder. This value may be set
         * for both acceptor and initiator connections.
         * @param size maximum backlog size.
         * @return {@code this PauseBuilder}.
         * @throws ConfigException
         * if {@code size} is &lt; zero.
         */
        public PauseBuilder maxBacklogSize(final int size)
        {
            if (size < 0)
            {
                throw (
                    new ConfigException.BadValue(
                        MAX_BACKLOG_SIZE_KEY, "size < 0"));
            }
            mMaxBacklogSize = size;

            return (this);
        } // end of maxBacklogSize(final int size)

        /**
         * Sets the backlog discard policy and returns
         * {@code this} pause configuration builder. This value
         * may only be set for initiator connections.
         * @param policy backlog overflow discard policy.
         * @return {@code this PauseBuilder}.
         * @throws ConfigException
         * if this is a acceptor connection or {@code policy} is
         * {@code null}.
         */
        public PauseBuilder discardPolicy(final DiscardPolicy policy)
        {
            if (mRole == ConnectionRole.ACCEPTOR)
            {
                throw (
                    new ConfigException.Generic(
                        "cannot set discard policy for acceptor connection"));
            }

            if (policy == null)
            {
                throw (
                    new ConfigException.BadValue(
                        DISCARD_POLICY_KEY, "policy is null"));
            }

            mDiscardPolicy = policy;

            return (this);
        } // end of discardPolicy(final DiscardPolicy policy)

        /**
         * Sets the idle time duration and returns {@code this}
         * pause configuration builder. This value may only be
         * set for initiator connections.
         * @param duration maximum idle time duration.
         * @return {@code this PauseBuilder}.
         * @throws ConfigException
         * if this is a acceptor connection or {@code duration}
         * is {@code null} or is &le; zero.
         */
        public PauseBuilder idleTime(final Duration duration)
        {
            if (mRole == ConnectionRole.ACCEPTOR)
            {
                throw (
                    new ConfigException.Generic(
                        "cannot set idle time for acceptor connection"));
            }

            if (duration == null)
            {
                throw (
                    new ConfigException.BadValue(
                        IDLE_TIME_KEY, "duration is null"));
            }

            if (duration.compareTo(Duration.ZERO) <= 0)
            {
                throw (
                    new ConfigException.BadValue(
                        IDLE_TIME_KEY, "duration <= 0"));
            }

            mIdleTime = duration;

            return (this);
        } // end of idleTime(final Duration duration)

        /**
         * Sets the maximum connection time duration and
         * returns {@code this} pause configuration builder.
         * This value may only be set for initiator connections.
         * @param duration maximum connection time duration.
         * @return {@code this PauseBuilder}.
         * @throws ConfigException
         * if this is a acceptor connection or {@code duration}
         * is {@code null} or is &le; zero.
         */
        public PauseBuilder maxConnectionTime(final Duration duration)
        {
            if (mRole == ConnectionRole.ACCEPTOR)
            {
                throw (
                    new ConfigException.Generic(
                        "cannot set max connection time for acceptor connection"));
            }

            if (duration == null)
            {
                throw (
                    new ConfigException.BadValue(
                        MAX_CONNECT_TIME_KEY,
                        "duration is null"));
            }

            if (duration.compareTo(Duration.ZERO) <= 0)
            {
                throw (
                    new ConfigException.BadValue(
                        MAX_CONNECT_TIME_KEY, "duration <= 0"));
            }

            mMaxConnectTime = duration;

            // If idle time is not set, then set that time to
            // the max connection time.
            if (mIdleTime.equals(Duration.ZERO))
            {
                mIdleTime = duration;
            }

            return (this);
        } // end of maxConnectionTime(final Duration duration)

        /**
         * Sets the transmit queue limit which trigger automatic
         * client connection resumption. If {@code limit} is zero
         * then this feature is disabled. Note: the transmit
         * queue size is based on the number of queued
         * application messages only.
         * <p>
         * This value may only be set for initiator connections.
         * </p>
         * @param size transmit queue size trigger connection
         * resumption.
         * @return {@code this PauseBuilder}.
         * @throws ConfigException
         * if this is a acceptor connection or {@code size}
         * &lt; zero.
         */
        public PauseBuilder resumeOnBacklogSize(final int size)
        {
            if (mRole == ConnectionRole.ACCEPTOR)
            {
                throw (
                    new ConfigException.Generic(
                        "cannot set max connection time for acceptor connection"));
            }

            if (size < 0)
            {
                throw (
                    new ConfigException.BadValue(
                        RESUME_ON_BACKLOG_SIZE_KEY, "size < 0"));
            }

            mResumeOnBacklogSize = size;

            return (this);
        } // end of resumeOnBacklogSize(int)

        /**
         * Returns the pause configuration created from the
         * configured parameters.
         * @return a pause configuration instance.
         * @throws ConfigException
         * if the pause configuration contains errors.
         */
        public PauseConfig build()
        {
            validate();

            return (new PauseConfig(this));
        } // end of build()

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Validates the pause configuration setting before
         * building said configuration. This method is called for
         * effect only.
         * @throws ConfigException
         * if the pause configuration contains errors.
         */
        private void validate()
        {
            // Both acceptor and initiator must have the pause
            // duration set.
            if (mDuration.equals(Duration.ZERO))
            {
                throw (
                    new ConfigException.Generic(
                        "pause duration not set"));
            }

            // Maximum connect time is required for connection
            // initiator
            if (mRole == ConnectionRole.INITIATOR &&
                mMaxConnectTime.equals(Duration.ZERO))
            {
                throw (
                    new ConfigException.Generic(
                        "maximum connect time not set"));
            }
        } // end of validate()
    } // end of class PauseBuilder
} // end of class EConfigure
