//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2009. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.config;

import java.io.Serializable;
import java.net.InetSocketAddress;
import java.util.Comparator;

/**
 * Compares two {@code java.net.InetSocketAddress} instances
 * based on the address bytes and the TCP port. Uses
 * {@code net.sf.eBus.net.InetAddressComparator} to compare
 * the socket addresses.
 *
 * @author crapp
 */

public class InetSocketAddressComparator
    implements Comparator<InetSocketAddress>,
               Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Java serialization identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Use this comparator for the InetAddress.
     */
    private final InetAddressComparator mAddressComparator;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new instance of InetSocketAddressComparator.
     */
    public InetSocketAddressComparator()
    {
        mAddressComparator = new InetAddressComparator();
    } // end of InetSocketAddressComparator()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Comparable Interface Implementation.
    //

    /**
     * Returns an integer &lt;, equal to or &gt; zero if
     * {@code a1} is &lt;, equal to or &gt; {@code a2}.
     * The {@code InetAddress}es are compared first. If equal,
     * the ports are compared.
     * @param a1 First compared address.
     * @param a2 Second compared address.
     * @return an integer &lt;, equal to or &gt; zero if
     * {@code a1} is &lt;, equal to or &gt; {@code a2}.
     * @see InetAddressComparator
     */
    @Override
    public int compare(final InetSocketAddress a1,
                       final InetSocketAddress a2)
    {
        int retval =
            mAddressComparator.compare(
                a1.getAddress(), a2.getAddress());

        if (retval == 0)
        {
            retval = (a1.getPort() - a2.getPort());
        }

        return (retval);
    } // end of compare(InetSocketAddress, InetSocketAddress)

    //
    // end of Comparable Interface Implementation.
    //-----------------------------------------------------------
} // end of class InetSocketAddressComparator
