//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2009, 2012, 2013, 2019. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.config;

import com.typesafe.config.Config;
import com.typesafe.config.ConfigException;
import java.io.Serializable;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.text.ParseException;
import java.util.Collection;
import java.util.Formatter;
import java.util.Set;
import java.util.TreeSet;
import static net.sf.eBus.config.ENetConfigure.MAX_PORT;
import static net.sf.eBus.config.ENetConfigure.MIN_PORT;

/**
 * Stores a set of {@code InetSocketAddress}es and determines if
 * a given address passes the filter. An address passes if:
 * <ol>
 *   <li>
 *     The filter contains both the host and port or
 *   </li>
 *   <li>
 *     The filter contains the host and a zero port. A zero port
 *     means that the client may bind to any port on the host.
 *   </li>
 * </ol>
 * <p>
 * This filter acts as a positive filter. Only clients whose
 * address is listed may connect. eBus does not use a negative
 * filter where all clients are accepted <i>except</i> those
 * listed.
 * </p>
 * <p>
 * An address filter may be defined in text for storage purposes.
 * The format is {@code <address>[',' <address>]*}
 * where &lt;address&gt; format is either
 * {@code (host | IP address) ':' port} or just
 * {@code (host | IP address)} and {@code port} is assumed to be
 * zero.
 * </p>
 * <p>
 * {@code net.sf.eBus.client.EServer} can be configured to use an
 * address filter, restricting accepted connections to those
 * specified by the filter.
 * </p>
 * <p>
 * As of eBus release 7.1.0, an address filter may be loaded from
 * a {@code com.typesafe} HOCON configuration.
 * </p>
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class AddressFilter
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Addresses are separated by a comma.
     */
    private static final String IFS0 = ",";

    /**
     * The host address and port are separated by a colon.
     */
    private static final String IFS1= ":";

    /**
     * The host name/dotted-notation address is the first token.
     */
    private static final int HOST_INDEX = 0;

    /**
     * The TCP port is the second token.
     */
    private static final int PORT_INDEX = 1;

    /**
     *  This is eBus version 2.1.1.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The address filter set.
     */
    private final Set<InetSocketAddress> mFilter;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates an empty address filter.
     */
    public AddressFilter()
    {
        mFilter =
            new TreeSet<>(new InetSocketAddressComparator());
    } // end of AddressFilter()

    /**
     * Creates a new address filter containing those elements in
     * the specified collection.
     * @param c collection whose elements will comprise the new
     * address filter.
     */
    public AddressFilter(final Collection<InetSocketAddress> c)
    {
        mFilter =
            new TreeSet<>(new InetSocketAddressComparator());
        mFilter.addAll(c);
    } // end of AddressFilter(Collection<InetSocketAddress>)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Compares the given object with this filter for equality.
     * Returns {@code true} if {@code o} is a
     * non-{@code null AddressFilter} with the same size and
     * addresses as {@code this} filter; otherwise, returns
     * {@code false}.
     * @param o comparison object.
     * @return {@code true} if {@code o} is equal to this address
     * filter.
     */
    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (this == o);

        if (!retcode && o instanceof AddressFilter)
        {
            retcode =
                mFilter.equals(((AddressFilter) o).mFilter);
        }

        return (retcode);
    } // end of equals(Object)

    /**
     * Returns the hash code value for this set. The hash code of
     * a set is defined to be the sum of the hash codes of the
     * elements in the set, where the hash code of a {@code null}
     * element is defined to be zero. This ensures that
     * {@code s1.equals(s2)} implies that
     * {@code s1.hashCode()==s2.hashCode()} for any two sets
     * {@code s1} and {@code s2}, as required by the general
     * contract of {@link Object#hashCode()}.
     * @return the hash code value for this address filter.
     */
    @Override
    public int hashCode()
    {
        return (mFilter.hashCode());
    } // end of hashCode()

    /**
     * Returns the address filter in text, formatted for storage
     * in {@link EConfigure}.
     * @return address filter in text format.
     */
    @Override
    public String toString()
    {
        int port;
        String sep = "";
        final Formatter retval = new Formatter();

        for (InetSocketAddress address : mFilter)
        {
            port = address.getPort();

            retval.format("%s%s", sep, address.getHostName());

            if (port != 0)
            {
                retval.format("%s%d", IFS1, port);
            }

            sep = IFS0;
        }

        return (retval.toString());
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns {@code true} if this address filter is empty.
     * @return {@code true} if this address filter is empty.
     */
    public boolean isEmpty()
    {
        return (mFilter.isEmpty());
    } // end of isEmpty()

    /**
     * Returns the address filter size.
     * @return the address filter size.
     */
    public int size()
    {
        return (mFilter.size());
    } // end of size()

    /**
     * Returns {@code true} if this address filter contains the
     * specified address; otherwise returns {@code false}.
     * @param address look for this object in the filter set.
     * @return {@code true} if {@code address} is in the filter.
     */
    public boolean contains(final InetSocketAddress address)
    {
        return (mFilter.contains(address));
    } // end of contains(InetSocketAddress)

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    /**
     * Adds the address to the filter set.
     * @param address the address added to this filter.
     * @return {@code true} if this filter did not already
     * contain {@code address}.
     * @exception NullPointerException
     * if {@code address} is {@code null}.
     */
    public boolean add(final InetSocketAddress address)
    {
        return (mFilter.add(address));
    } // end of add(InetSocketAddress)

    /**
     * Adds all the addresses in the collection to this set.
     * @param c the addresses to be added to this collection.
     * @return {@code true} if the filter changed as a result of
     * this call.
     * @exception NullPointerException
     * if either {@code c} is {@code null} or contains a
     * {@code null} element.
     */
    public boolean addAll(final Collection<InetSocketAddress> c)
    {
        return (mFilter.addAll(c));
    } // end of addAll(Collection<InetSocketAddress>)

    /**
     * Removes the specified address from this filter if it is
     * present. Returns {@code true} if the filter contained this
     * address.
     * @param address remove this address if present.
     * @return {@code true} if the address was removed.
     * @exception NullPointerException
     * if {@code address} is {@code null}.
     */
    public boolean remove(final InetSocketAddress address)
    {
        return (mFilter.remove(address));
    } // end of remove(InetSocketAddress)

    /**
     * Removes all the specified addresses from this filter.
     * Returns {@code true} if this filter was changed by this
     * operation.
     * @param c addresses to be removed.
     * @return {@code true} if this filter was changed by this
     * operation.
     * @exception NullPointerException
     * if {@code c} is {@code null} or contains a {@code null}
     * address.
     */
    public boolean removeAll(final Collection<InetSocketAddress> c)
    {
        return (mFilter.removeAll(c));
    } // end of removeAll(Collection<InetSocketAddress>)

    /**
     * Removes all addresses from this filter.
     */
    public void clear()
    {
        mFilter.clear();
        return;
    } // end of clear()

    /**
     * Returns {@code true} if {@code iAddress} is acceptable
     * to this filter and {@code false} otherwise.
     * @param iAddress Check if this address is in the filter.
     * @return {@code true} if {@code iAddress} is acceptable
     * to this filter and {@code false} otherwise.
     */
    public boolean passes(final InetSocketAddress iAddress)
    {
        final InetSocketAddress anyAddress =
            new InetSocketAddress(iAddress.getAddress(), 0);

        return (mFilter.contains(iAddress) ||
                mFilter.contains(anyAddress));
    } // end of passes(InetSocketAddress)

    /**
     * Returns the address filter parsed from the given text.
     * The filter addresses are comma-separated and the filter
     * address must be formatted either "host:port" or just
     * "host" (and port is assumed to be zero).
     * <p>
     * Returns a non-{@code null} empty address filter if not
     * defined.
     * </p>
     * @param s the address filter text.
     * @return the parsed address filter.
     * @exception ParseException
     * if {@code s} is either improperly formatted or contains
     * an invalid filter address.
     */
    public static AddressFilter parse(final String s)
        throws ParseException
    {
        AddressFilter retval = new AddressFilter();

        if (s != null && !s.isEmpty())
        {
            for (String filter : s.split(IFS0))
            {
                retval.add(parseAddress(filter));
            }
        }

        return (retval);
    } // end of parse(String)

    /**
     * Returns an address filter extracted from the given JSON
     * configuration. The filter addresses are defined as a JSON
     * string list and the filter address must be formatted
     * either "host:port" or just "host" and port is assumed to
     * be zero.
     * <p>
     * Returns a non-{@code null} empty address filter if not
     * defined.
     * </p>
     * @param config JSON configuration containing address
     * filter.
     * @param key address filter list is stored under this key.
     * @return parsed address filter.
     * @throws ConfigException
     * if {@code config} contains an improperly formatted filter.
     */
    public static AddressFilter load(final Config config,
                                     final String key)
    {
        AddressFilter retval = null;

        if (config.hasPath(key))
        {
            final AddressFilter filterList = new AddressFilter();

            config.getStringList(key)
                  .forEach(
                      filter ->
                      {
                          try
                          {
                              filterList.add(parseAddress(filter));
                          }
                          catch (ParseException parsex)
                          {
                              throw (
                                  new ConfigException.BadValue(
                                      key,
                                      "invalid address filter",
                                      parsex));
                          }
                      });

            retval = filterList;
        }

        return (retval);
    } // end of load(Config)

    /**
     * Returns an IP address and port from the given string. The
     * string format must be either "host:port" or "host". If a
     * port is no provided, it is assumed to be zero.
     * @param s parse this host and port text.
     * @return the IP address and port.
     * @exception ParseException
     * if {@code s} has an invalid format, references an unknown
     * host address, or contains an invalid port.
     */
    private static InetSocketAddress parseAddress(final String s)
        throws ParseException
    {
        final String[] tokens = s.split(IFS1);
        InetAddress host;
        int port = 0;

        if (tokens.length > 2)
        {
            throw (
                new ParseException("invalid address filter", 0));
        }

        try
        {
            host = InetAddress.getByName(tokens[HOST_INDEX]);
        }
        catch (UnknownHostException hostex)
        {
            final ParseException parsex =
                new ParseException(
                    String.format(
                        "unknown host %s", tokens[HOST_INDEX]),
                    0);

            parsex.initCause(hostex);

            throw (parsex);
        }

        if (tokens.length == 2)
        {
            try
            {
                port = Integer.parseInt(tokens[PORT_INDEX]);

                if (port < MIN_PORT || port > MAX_PORT)
                {
                    throw (
                        new ParseException(
                            String.format(
                                "invalid port %s",
                                tokens[PORT_INDEX]),
                            0));
                }
            }
            catch (NumberFormatException formex)
            {
                final ParseException parsex =
                    new ParseException(
                        String.format(
                            "invalid port %s",
                            tokens[PORT_INDEX]),
                        0);

                parsex.initCause(formex);

                throw (parsex);
            }
        }

        return (new InetSocketAddress(host, port));
    } // end of parseAddress(String)
} // end of class AddressFilter
