//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2015, 2016. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.config;

import java.io.IOException;
import java.util.Collections;
import java.util.Formatter;
import java.util.HashMap;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.logging.Level;
import java.util.logging.Logger;
import net.sf.eBus.util.Properties;

/**
 * This immutable class contains selector thread configuration
 * extracted from properties. Selector threads configuration is
 * taken from the properties file specified by the command line
 * parameter
 * {@code -Dnet.sf.eBus.config.file=}&lt;<em>property file path</em>&gt;.
 * Note that selector threads are <em>not</em> started at this
 * time but only when its name referenced by an
 * {@code AsyncChannel}. So if no channels use the
 * selector during an application run, then the thread is never
 * started.
 * <p>
 * Selector thread configuration uses the following property
 * keys:
 * </p>
 * <ul>
 *   <li>
 *     {@code eBus.net.selectors}: a comma-separated list of
 *     selector thread names. The names should be unique (no
 *     duplicates). All duplicates are logged and ignored. The
 *     selector thread name is used to retrieve the remaining
 *     keys.
 *   </li>
 *   <li>
 *    {@code eBus.net.selector}.<em>name</em>.{@code type}:
 *    The selector thread type must be one of {@link ThreadType}.
 *    <p>
 *    This property is required.
 *    </p>
 *   </li>
 *   <li>
 *    {@code eBus.net.selector}.<em>name</em>.{@code isDefault}:
 *    Set to "true" if this is the default selector. The default
 *    selector is used by all {@code AsyncChannels} not
 *    explicitly assigned to a selector. If multiple selectors
 *    are designated as the default, then it cannot be determined
 *    which selector will be used as the default.
 *    <p>
 *    The default value for this setting is {@code false}.
 *    </p>
 *   </li>
 *   <li>
 *    {@code eBus.net.selector}.<em>name</em>.{@code priority}:
 *    The selector thread runs at this thread priority. The
 *    specified value must be &ge; {@link Thread#MIN_PRIORITY}
 *    and &le; {@link Thread#MAX_PRIORITY}.
 *    <p>
 *    The default value for this setting is
 *    {@link Thread#NORM_PRIORITY}.
 *    </p>
 *   </li>
 *   <li>
 *    {@code eBus.net.selector}.<em>name</em>.{@code spinLimit}:
 *    If {@code type} is {@link ThreadType#SPINPARK}, then this
 *    property specifies the number of times the thread will spin
 *    on {@link java.nio.channels.Selector#selectNow()} before
 *    parking.
 *    <p>
 *    The default value is {@link #DEFAULT_SPIN_LIMIT}.
 *    </p>
 *   </li>
 *   <li>
 *    {@code eBus.net.selector}.<em>name</em>.{@code parkTime}:
 *    If {@code type} is {@link ThreadType#SPINPARK}, then this
 *    property specifies the number of nanoseconds the thread
 *    will park before returning to spinning.
 *    <p>
 *    The default value is {@link #DEFAULT_PARK_TIME}.
 *    </p>
 *   </li>
 * </ul>
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public class ENetConfigure
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * The minimum allowed port number is zero.
     */
    public static final int MIN_PORT = 0;

    /**
     * The maximum allowed port number is 65,535.
     */
    public static final int MAX_PORT = 65535;

    /**
     * Use the value {@value} to specify socket is opened and
     * bound to any port.
     */
    public static final int ANY_PORT = -1;

    //
    // System property keys.
    //

    // Property keys.

    /**
     * The key {@code eBus.net.selectors} contains a
     * comma-separated list of selector thread names.
     */
    public static final String SELECTORS_KEY =
        "eBus.net.selectors";

    /**
     * Selector keys are prefixed by {@code eBus.net.selector.}
     * and followed by a selector thread, which must appear in
     * the {@link #SELECTORS_KEY} property.
     */
    public static final String SELECTOR_PREFIX =
        "eBus.net.selector.";

    /**
     * The {@link ThreadType .type} property is used to specify
     * the {@code ESelector} used with the selector thread.
     * <p>
     * This property is required.
     * </p>
     */
    public static final String TYPE_KEY = ".type";

    /**
     * The boolean {@code .isDefault} property is used to
     * specify when this selector thread should be used as the
     * default selector thread. At most one selector should be
     * designated as the default selector. If more than one is
     * so designated, the subsequent {@code .isDefault} keys are
     * ignored.
     * <p>
     * If no user-defined selector is marked as default, then
     * a {@link ThreadType#BLOCKING blocking},
     * {@link Thread#NORM_PRIORITY normal priority} selector
     * named {@code AsyncChannel.DEFAULT_SELECTOR} is used as the
     * default selector.
     * </p>
     * <p>
     * The default value is {@code false}
     * (not the default selector).
     * </p>
     */
    public static final String DEFAULT_KEY = ".isDefault";

    /**
     * The integer {@code .priority} property is used to
     * set the select thread priority.
     * <p>
     * The default value is {@link Thread#NORM_PRIORITY}.
     * </p>
     */
    public static final String PRIORITY_KEY = ".priority";

    /**
     * If the selector type is
     * {@link ThreadType#SPINPARK spin+park}, then the
     * integer {@code .spinLimit} property may be specified. This
     * setting defines the number of times the selector thread
     * may call
     * {@link java.nio.channels.Selector#selectNow()} before
     * parking.
     * <p>
     * This value must be &gt; zero.
     * </p>
     * <p>
     * Default values is {@link #DEFAULT_SPIN_LIMIT}.
     * </p>
     * <p>
     * This property is ignored is the selector type is not
     * spin+park.
     * </p>
     */
    public static final String SPIN_LIMIT_KEY = ".spinLimit";

    /**
     * If the selector type is
     * {@link ThreadType#SPINPARK spin+park}, then the
     * integer {@code .parkTime} property may be specified. This
     * setting specifies the <em>nanosecond</em> park time taken
     * between
     * {@link java.nio.channels.Selector#selectNow()} spin
     * cycles.
     * <p>
     * This value must be &gt; zero.
     * </p>
     * <p>
     * Default values is {@link #DEFAULT_PARK_TIME}.
     * </p>
     * <p>
     * This property is ignored is the selector type is not
     * spin+park.
     * </p>
     */
    public static final String PARK_TIME_KEY = ".parkTime";

    /**
     * Array key separator as a character.
     */
    private static final char KEY_IFS = ',';

    //
    // Default values.
    //

    /**
     * The default selector thread priority is
     * {@link Thread#NORM_PRIORITY normal}.
     */
    public static final int DEFAULT_PRIORITY =
        Thread.NORM_PRIORITY;

    /**
     * The default select spin limit is {@value} calls to
     * {@link java.nio.channels.Selector#selectNow()} before
     * parking the thread.
     */
    public static final int DEFAULT_SPIN_LIMIT = 2_500_000;

    /**
     * The default thread park time is {@value} nanoseconds.
     */
    public static final int DEFAULT_PARK_TIME = 1_000;

    /**
     * The default input and output socket buffer size is
     * {@value} bytes.
     */
    public static final int DEFAULT_BUFFER_SIZE = 2048;

    /**
     * The default selector type is
     * {@link ThreadType#BLOCKING}.
     */
    public static final ThreadType DEFAULT_SELECTOR_TYPE =
        ThreadType.BLOCKING;

    /**
     * The default selector thread name is "__DEFAULT__".
     * The default selector type is
     * {@link ThreadType#BLOCKING blocking}.
     */
    private static final String DEFAULT_SELECTOR =
        "eBus:selectorThread:__DEFAULT__";

    //-----------------------------------------------------------
    // Statics.
    //

    /**
     * The default selector name as defined in the eBus
     * configuration file.
     */
    private static SelectorInfo sDefaultSelector;

    /**
     * The user-defined selectors mapped by selector name.
     * May be empty.
     */
    private static Map<String, SelectorInfo> sSelectors;

    /**
     * Logging subsystem interface.
     */
    private static final Logger sLogger =
        Logger.getLogger(ENetConfigure.class.getName());

    // Class static initialization
    static
    {
        final String configFile =
            System.getProperty(EConfigure.CONFIG_FILE_ENV);

        sDefaultSelector = null;

        // Load the eBus configuration if specified on the
        // command line and fill in the selectors.
        if (configFile != null && configFile.isEmpty() == false)
        {
            try
            {
                final Properties props =
                    Properties.loadProperties(configFile);

                ENetConfigure.load(props);

                // Find the default selector.
                for (SelectorInfo sInfo : sSelectors.values())
                {
                    if (sInfo.isDefault())
                    {
                        sDefaultSelector = sInfo;
                        break;
                    }
                }
            }
            catch (IllegalArgumentException |
                   IOException |
                   MissingResourceException jex)
            {
                String reason = jex.getMessage();

                if (reason == null || reason.isEmpty() == true)
                {
                    reason = "(no reason given)";
                }

                sLogger.log(
                    Level.WARNING,
                    String.format(
                        "Error loading eBus network configuration from %s, %s.",
                        configFile,
                        reason),
                    jex);
            }
        }

        // Was a default selector specified?
        if (sDefaultSelector == null)
        {
            // No. Then set the default selector information to
            // a blocking selector with normal priority.
            sDefaultSelector =
                new SelectorInfo(DEFAULT_SELECTOR,
                                 DEFAULT_SELECTOR_TYPE,
                                 true,
                                 Thread.NORM_PRIORITY,
                                 0L,
                                 0L);
        }

        if (sSelectors == null)
        {
            final Map<String, SelectorInfo> selectors =
               new HashMap<>(1);

            selectors.put(
                sDefaultSelector.name(), sDefaultSelector);
            sSelectors = Collections.unmodifiableMap(selectors);
        }
    } // end of class static initialization.

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Private constructor to prevent instantiation.
     */
    private ENetConfigure()
    {}

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    public static SelectorInfo defaultSelector()
    {
        return (sDefaultSelector);
    } // end of defaultSelector()

    /**
     * Returns {@code true} if {@code name} is a known selector
     * and {@code false} if not known.
     * @param name the name to be checked.
     * @return {@code true} if {@code name} is a known selector.
     */
    public static boolean isKnownSelector(final String name)
    {
        return (sSelectors.containsKey(name));
    } // end of isKnownSelector(String)

    /**
     * Returns the selector for the given name;
     * returns {@code null} if {@code name} does not reference
     * a known selector.
     * @param name selector name.
     * @return configured selector information.
     */
    public static SelectorInfo selector(final String name)
    {
        return (sSelectors.get(name));
    } // end of selector(String)

    /**
     * Returns the selector thread configurations. The returned
     * list is unmodifiable.
     * @return selector thread configurations.
     */
    public static Map<String, SelectorInfo> selectors()
    {
        return (sSelectors);
    } // end of selectors()

    /**
     * Returns the loaded network configuration as text.
     * @return textual representation of network configuration.
     */
    public static String asText()
    {
        String sep = "";
        final Formatter retval = new Formatter();

        for (SelectorInfo info : sSelectors.values())
        {
            retval.format("%s%s", sep, info);
            sep = "\\n";
        }

        return (retval.toString());
    } // end of asText()

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    /**
     * Returns selector thread configuration extracted from the
     * given properties.
     * @param props load the selector configuration from here.
     * @throws MissingResourceException
     * if {@code props} is missing a required property key.
     */
    public static void load(final java.util.Properties props)
        throws MissingResourceException
    {
        load(new net.sf.eBus.util.Properties(props));
        return;
    } // end of load(Properties)

    /**
     * Returns selector thread configuration extracted from the
     * given properties.
     * @param props load the selector configuration from here.
     * @throws MissingResourceException
     * if {@code props} is missing a required property key.
     */
    public static void load(final net.sf.eBus.util.Properties props)
        throws MissingResourceException
    {
        final Map<String, SelectorInfo> selectors =
            new HashMap<>();
        final String[] names =
            props.getArrayProperty(SELECTORS_KEY, KEY_IFS);

        for (String name : names)
        {
            if (selectors.containsKey(name) == true)
            {
                sLogger.warning(
                    String.format(
                        "Selector %s appears twice in %s, ignored.",
                        name,
                        SELECTORS_KEY));
            }
            else
            {
                selectors.put(name, loadInfo(name, props));
            }
        }

        sSelectors = Collections.unmodifiableMap(selectors);

        return;
    } // end of load(Properties)

    /**
     * Returns the selector configuration information for the
     * given selector name, extracting the properties from
     * {@code props}.
     * @param name the unique selector name.
     * @param props extract selector property values from this
     * set.
     * @return selector configuration information.
     * @throws MissingResourceException
     * if {@code props} is missing required selector properties
     * or the property values are invalid.
     */
    private static SelectorInfo
        loadInfo(final String name,
                 final net.sf.eBus.util.Properties props)
            throws MissingResourceException
    {
        String keyPrefix = SELECTOR_PREFIX + name;
        String key;
        String value;
        ThreadType type;
        int priority;
        boolean defaultFlag;
        long spinLimit = 0L;
        long parkTime = 0L;

        key = keyPrefix + TYPE_KEY;
        value = props.getProperty(key);
        type = ThreadType.find(value);
        if (type == null)
        {
            throw (
                new MissingResourceException(
                    String.format(
                        "\"%s\" is not a valid selector type",
                        value),
                    (ThreadType.class).getName(),
                    key));
        }

        key = keyPrefix + DEFAULT_KEY;
        defaultFlag = props.getBooleanProperty(key, false);

        key = keyPrefix + PRIORITY_KEY;
        priority = props.getIntProperty(key, DEFAULT_PRIORITY);
        if (priority < Thread.MIN_PRIORITY ||
            priority > Thread.MAX_PRIORITY)
        {
            throw (
                new MissingResourceException(
                    String.format(
                        "\"%s\" is not a valid thread priority",
                        value),
                    "int",
                    key));
        }

        if (type == ThreadType.SPINPARK ||
            type == ThreadType.SPINYIELD)
        {
            key = keyPrefix + SPIN_LIMIT_KEY;
            spinLimit =
                props.getIntProperty(key, DEFAULT_SPIN_LIMIT);
            if (spinLimit <= 0L)
            {
                throw (
                    new MissingResourceException(
                        String.format(
                            "%,d is not a valid spin limit",
                            spinLimit),
                        (Long.class).getName(),
                        key));
            }
        }

        if (type == ThreadType.SPINPARK)
        {
            key = keyPrefix + PARK_TIME_KEY;
            parkTime =
                props.getIntProperty(key, DEFAULT_PARK_TIME);
            if (parkTime <= 0L)
            {
                throw (
                    new MissingResourceException(
                        String.format(
                            "%,d is not a valid park time",
                            spinLimit),
                        (Long.class).getName(),
                        key));
            }
        }

        return (new SelectorInfo(name,
                                 type,
                                 defaultFlag,
                                 priority,
                                 spinLimit,
                                 parkTime));
    } // end of loadInfo(String)

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * This immutable class contains the selector thread
     * configuration for a specific, named selector.
     */
    public static final class SelectorInfo
        implements Comparable<SelectorInfo>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * The selector thread's unique name.
         */
        private final String mName;

        /**
         * The select thread's encapsulated {@link ESelector}
         * type.
         */
        private final ThreadType mType;

        /**
         * {@code true} if this selector thread is used as the
         * default.
         */
        private final boolean mIsDefault;

        /**
         * The selector thread priority.
         */
        private final int mPriority;

        /**
         * The spin limit for a
         * {@link ThreadType#SPINPARK spin+park} selector
         * type.
         */
        private final long mSpinLimit;

        /**
         * The park time for a
         * {@link ThreadType#SPINPARK spin+park} selector
         * type.
         */
        private final long mParkTime;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new selector thread configuration from the
         * given parameters.
         * @param name selector thread name.
         * @param type selector thread type.
         * @param isDefault {@code true} if this is the default
         * selector thread.
         * @param priority the thread priority.
         * @param spinLimit if {@code type} is spin+park, then
         * this is the spin limit.
         * @param parkTime if {@code type} is spin+park, then
         * this is the park time.
         */
        private SelectorInfo(final String name,
                             final ThreadType type,
                             final boolean isDefault,
                             final int priority,
                             final long spinLimit,
                             final long parkTime)
        {
            mName = name;
            mType = type;
            mIsDefault = isDefault;
            mPriority = priority;
            mSpinLimit = spinLimit;
            mParkTime = parkTime;
        } // end of SelectorInfo(...)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Comparable Interface Implementation.
        //

        @Override
        public int compareTo(final SelectorInfo selector)
        {
            return (mName.compareTo(selector.mName));
        } // end of compareTo(SelectorInfo)

        //
        // end of Comparable Interface Implementation.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Object Method Overrides.
        //

        @Override
        public String toString()
        {
            final Formatter retval = new Formatter();

            retval.format("[%s]%n", mName);
            retval.format("selector type: %s%n", mType);
            retval.format("     priority: %s%n", mPriority);
            retval.format("   is default: %b%n", mIsDefault);

            if (mType == ThreadType.SPINPARK)
            {
                retval.format("   spin limit: %,d%n",
                              mSpinLimit);
                retval.format("    park time: %,d nanoseconds%n",
                              mParkTime);
            }

            return (retval.toString());
        } // end of toString()

        //
        // end of Object Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the unique selector thread name.
         * @return selector thread name.
         */
        public String name()
        {
            return (mName);
        } // end of name()

        /**
         * Returns the {@code ESelector} type.
         * @return selector type.
         */
        public ThreadType type()
        {
            return (mType);
        } // end of type()

        /**
         * Returns {@code true} if this user-defined selector
         * thread is the default selector.
         * @return {@code true} if the default selector thread.
         */
        public boolean isDefault()
        {
            return (mIsDefault);
        } // end of isDefault()

        /**
         * Returns the selector thread priority.
         * @return thread priority.
         */
        public int priority()
        {
            return (mPriority);
        } // end of priority()

        /**
         * Returns a spin+park {@code ESelector} spin limit.
         * @return spin limit.
         */
        public long spinLimit()
        {
            return (mSpinLimit);
        } // end of spinLimit()

        /**
         * Returns the spin+park nanosecond park time.
         * @return park time.
         */
        public long parkTime()
        {
            return (mParkTime);
        } // end of parkTime()

        //
        // end of Get Methods.
        //-------------------------------------------------------
    } // end of class SelectorInfo
} // end of class ENetConfigure
