//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2008 - 2013, 2015 - 2018. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.config;

import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.nio.ByteOrder;
import java.text.ParseException;
import java.util.Arrays;
import java.util.Formatter;
import java.util.HashMap;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.prefs.Preferences;
import javax.net.ssl.SSLContext;
import static net.sf.eBus.config.ENetConfigure.ANY_PORT;
import static net.sf.eBus.config.ENetConfigure.DEFAULT_BUFFER_SIZE;
import static net.sf.eBus.config.ENetConfigure.MAX_PORT;
import static net.sf.eBus.config.ENetConfigure.MIN_PORT;

/**
 * This immutable class contains eBus services, remote
 * connection, and dispatcher configuration. This configuration
 * can be specified:
 * <ul>
 *   <li>
 *     Programmatically by instantiating
 *     {@link EConfigure.Service} and
 *     {@link EConfigure.RemoteConnection}
 *     and passing these objects to
 *     {@code EServer.configure(EConfigure)} and
 *     {@code ERemoteApp.configure(EConfigure)}, respectively.
 *     {@link EConfigure.Dispatcher} has a private constructor
 *     and cannot be instantiated by the application.
 *   </li>
 *   <li>
 *     From properties.
 *   </li>
 *   <li>
 *     From preferences.
 *   </li>
 * </ul>
 * Also provides methods for storing this configuration to
 * {@link #store(java.util.Properties) properties} or
 * {@link #store(Preferences) preferences}.
 * <p>
 * Please note that eBus configuration requires no service or
 * connections to be specified. If this is the case, remote
 * connections are neither accepted nor established.
 * </p>
 * <p>
 * eBus release 4.7.0 supports secure TCP connections based on
 * SSL/TLS using {@link SSLContext}. This connection type is
 * <em>not</em> supported using properties/preferences
 * configuration as this would require placing sensitive
 * information in an un-secure manner. Therefore, creating
 * secure TCP services and connections can only be done
 * programmatically.
 * </p>
 * <p>
 * Dispatcher threads are instantiated on JVM start-up, the
 * dispatcher configuration is taken from the properties file
 * specified by the command line parameter
 * {@code -Dnet.sf.eBus.config.file=}&lt;<em>property file path</em>&gt;.
 * eBus loads the {@code EClientDispatcher} configuration from
 * this properties file and creates the dispatcher threads
 * according to this configuration. If no configuration file is
 * specified or is incorrectly configured, eBus defaults to a
 * single, blocking run queue. The run queue thread count equals
 * {@link Runtime#availableProcessors()}.
 * </p>
 * <p>
 * The eBus configuration uses the following keys for both
 * properties and preferences:
 * </p>
 * <ul>
 *   <li>
 *     {@code eBus.servers}: a comma-separated list of
 *     eBus local server names. This key is optional or may
 *     be an empty string. The names must be unique (no
 *     duplicates). The server name is used to retrieve the
 *     following keys:
 *     <ol>
 *       <li>
 *         <code>eBus.service.<em>name</em>.port</code>:
 *         the service is opened on this TCP port.
 *         <p>
 *         This property is required.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.addressFilter</code>:
 *         the {@link AddressFilter} limiting which clients may
 *         connect to this eBus application. This is a positive
 *         filter meaning that is specifies which clients may
 *         connect to this application and not a negative filter
 *         which allows all connections except those specified.
 *         <p>
 *         This property is a string which must satisfy
 *         {@link AddressFilter#parse(String)}.
 *         </p>
 *         <p>
 *         This property is optional and may be set to an empty
 *         string.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.inputBufferSize</code>:
 *         the socket input buffer size in bytes per connection.
 *         Must be an integer value &gt; zero.
 *         <p>
 *         The default setting is
 *         {@code ERemoteApp.DEFAULT_BUFFER_SIZE}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.outputBufferSize</code>:
 *         the socket output buffer size in bytes per connection.
 *         Must be an integer value &gt; zero.
 *         <p>
 *         The default setting is
 *         {@code ERemoteApp.DEFAULT_BUFFER_SIZE}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.byteOrder</code>:
 *         the socket input and output
 *         {@link java.nio.ByteBuffer}s are set to this byte
 *         order.
 *         <p>
 *         The default setting is
 *         {@link java.nio.ByteOrder#LITTLE_ENDIAN}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.messageQueueSize</code>:
 *         when the output buffer is full, outgoing messages are
 *         queued until the output buffer is no longer full.
 *         This value specifies the maximum number of messages
 *         which may enqueued at any one time. Must be an integer
 *         &ge; zero where zero means the message queue size is
 *         unlimited.
 *         <p>
 *         The default setting is an unlimited message queue.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.server.<em>name</em>.serviceSelector</code>:
 *         the named selector thread used to monitor the server
 *         socket channel.
 *         <p>
 *         The default setting is
 *         {@code net.sf.eBus.net.AsyncChannel.defaultSelector}
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.server.<em>name</em>.connectionSelector</code>:
 *         the named selector thread used to monitor the accepted
 *         client socket channels.
 *         <p>
 *         The default setting is
 *         {@code net.sf.eBus.net.AsyncChannel.defaultSelector}
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.heartbeatDelay</code>:
 *         the accepted connection sends heartbeat messages this
 *         many milliseconds after the last received message.
 *         Every time a message is received, this timer is reset.
 *         Must be an integer &ge; zero where zero means that
 *         no heartbeat message is sent.
 *         <p>
 *         The default is not to send heartbeat message.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.service.<em>name</em>.heartbeatReplyDelay</code>:
 *         wait this many milliseconds for a reply to a heartbeat
 *         message. If a reply is <i>not</i> received in this
 *         time limit, close the connection. Must be an integer
 *         value &ge; zero where zero means no time limit.
 *         <p>
 *         The default is no time limit because heart beating is
 *         off by default.
 *         </p>
 *       </li>
 *     </ol>
 *   </li>
 *   <li>
 *     {@code eBus.connections}: a comma-separated list of
 *     eBus remote connection names. This key is optional or may
 *     be an empty string. The names must be unique (no
 *     duplicates). The connection name is used to retrieve the
 *     following keys:
 *     <ul>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.host</code>:
 *         the connection is made to this host. May be either a
 *         host name or dotted notation IP address.
 *         <p>
 *         This setting is required.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.port</code>:
 *         the connection is made to this TCP port.
 *         <p>
 *         This setting is required.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.bindPort</code>:
 *         the connection's local end is bound to this TCP port.
 *         <p>
 *         The default setting is to bind the local port is
 *         OS-dependent. The OS will bind the local port to what
 *         is considered the next available TCP port.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.inputBufferSize</code>:
 *         the socket input buffer size in bytes. Must be an
 *         integer value &gt; zero.
 *         <p>
 *         The default setting is
 *         {@code ERemoteApp.DEFAULT_BUFFER_SIZE}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.outputBufferSize</code>:
 *         the socket output buffer size in bytes. Must be an
 *         integer value &gt; zero.
 *         <p>
 *         The default setting is
 *         {@code ERemoteApp.DEFAULT_BUFFER_SIZE}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.byteOrder</code>:
 *         the socket input and output
 *         {@link java.nio.ByteBuffer}s are set to this byte
 *         order.
 *         <p>
 *         The default setting is
 *         {@link java.nio.ByteOrder#LITTLE_ENDIAN}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.messageQueueSize</code>:
 *         when the output buffer is full, outgoing messages are
 *         queued until the output buffer is no longer full.
 *         This value specifies the maximum number of messages
 *         which may enqueued at any one time. Must be an integer
 *         &ge; zero where zero means the queue size is
 *         unlimited.
 *         <p>
 *         The default setting is an unlimited message queue
 *         size.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.selector</code>:
 *         the named selector thread used to monitor this socket
 *         channel.
 *         <p>
 *         The default setting is
 *         {@code net.sf.eBus.net.AsyncChannel.defaultSelector}
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.reconnect</code>:
 *         boolean value which specifies whether the connection
 *         is automatically reconnected if lost if true. Should
 *         be {@code true} or {@code false}.
 *         <p>
 *         The default setting is {@code false}. This default
 *         value is used if the key is either missing, has no
 *         value, or the value cannot be decoded.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.reconnectTime</code>:
 *         the number of milliseconds between reconnect attempts.
 *         This is an 8-byte, signed integer value and must be
 *         &gt; zero.
 *         <p>
 *         The default setting is
 *         {@code ERemoteApp.DEFAULT_RECONNECT_TIME}. This value
 *         is ignored if reconnect is {@code false}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.heartbeatDelay</code>:
 *         send a heartbeat message after this many milliseconds
 *         after the last received message. Every time a message
 *         is received, this timer is reset. This integer value
 *         must be &ge; zero where zero means that heartbeat
 *         messages are not sent.
 *         <p>
 *         The default setting is not to send a heartbeat
 *         message.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.connection.<em>name</em>.heartbeatReplyDelay</code>:
 *         wait this many milliseconds for a reply to a heartbeat
 *         message. If a reply is <i>not</i> received in this
 *         time limit, close the connection. This integer value
 *         must be &ge; zero where zero means there is no
 *         heartbeat reply time limit.
 *         <p>
 *         The default setting is no time limit because
 *         heart beating is off by default.
 *         </p>
 *       </li>
 *     </ul>
 *   </li>
 *   <li>
 *     {@code eBus.dispatchers}: a comma-separated list of eBus
 *     dispatcher groups. This key is optional and may be set to
 *     an empty string. The names must be unique with no
 *     duplicates. The dispatcher name is used to retrieve the
 *     following keys:
 *     <br>
 *     (<strong>Note:</strong> Dispatcher configuration must be placed into a
 *     properties file and the {@code java} command line
 *     parameter
 *     {@code -Dnet.sf.eBus.config.file=}&lt;{@code properties file path}&gt;
 *     must be set because dispatchers are started by the
 *     {@code EClient} class initializer block.)
 *     <ul>
 *       <li>
 *         <code>eBus.dispatcher.<em>name</em>.isDefault</code>:
 *         Set to "true" if this is the default dispatcher. The
 *         default dispatcher is used for all client classes not
 *         explicitly assigned to a dispatcher. If multiple
 *         dispatchers are designated as the default, then it
 *         cannot be determined which selector will be used as
 *         default.
 *         <p>
 *         The default value for this setting is {@code false}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.dispatcher.<em>name</em>.runQueueType</code>:
 *         Defines how the dispatcher thread removes clients from
 *         the run queue: {@code blocking}, {@code spinning},
 *         {@code spin+park}, {@code spin+yield}. Blocking is the
 *         most fair since it allows other threads to use the
 *         core while waiting a client to be offered to the run
 *         queue; but blocking is also the slowest. Spinning
 *         means that the dispatcher is repeatedly calling
 *         {@link java.util.Queue#poll()} until a client is
 *         returned. Spinning is the fastest extraction technique
 *         but monopolizes the core, preventing other threads
 *         from using it. The spin+park and spin+yield types are
 *         a compromise between blocking and spinning. The
 *         dispatcher calls {@code Queue.poll} a configured
 *         number of times, trying to extract a client. If no
 *         client is returned after that configured spin limit is
 *         reached, the dispatcher either parks for a specified
 *         nanosecond time or yields the core. When the
 *         dispatcher returns from the park or yield, the spin
 *         limit is reset to the configured limit and the poll
 *         spin begins again.
 *         <p>
 *         The default value for this setting is {@code blocking}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.dispatcher.<em>name</em>.spinLimit</code>:
 *         If {@code runQueueType} is set to {@code spin+park} or
 *         {@code spin+yield}, then this is the
 *         {@code Queue.poll} spin limit.
 *         <p>
 *         The default value for this setting is
 *         {@link #DEFAULT_SPIN_LIMIT}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.dispatcher.<em>name</em>.parkTime</code>:
 *         If {@code runQueueType} is set to {@code spin+park},
 *         then this is the nanosecond park time.
 *         <p>
 *         The default value for this setting is
 *         {@link #DEFAULT_PARK_TIME}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.dispatcher.<em>name</em>.priority</code>:
 *         The dispatcher threads are run at this thread
 *         priority. The specified value must be
 *         &ge; {@link Thread#MIN_PRIORITY} and
 *         &le; {@link Thread#MAX_PRIORITY}.
 *         <p>
 *         The default value for this setting is
 *         {@link Thread#NORM_PRIORITY}.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.dispatcher.<em>name</em>.quantum</code>:
 *         This is the maximum number of nanoseconds a client may
 *         consume on a dispatcher thread before it musts yield
 *         to another consumer. Note that this is <em>not</em> a
 *         preemptive yield. The dispatcher thread tracks how
 *         long a client takes processing each message, adding up
 *         total time spent. When the accumulated time equals or
 *         exceeds the quantum, the client is posted back to the
 *         run queue if it has more messages to process. The
 *         quantum is reset each time the client is acquired by a
 *         dispatcher thread.
 *         <p>
 *         <strong>Note:</strong> it is possible for a client to
 *         monopolize a dispatcher thread, denying the thread to
 *         other clients associated with the dispatcher.
 *         </p>
 *         <p>
 *         The default value for this setting is
 *         {@link #DEFAULT_QUANTUM}
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.dispatcher.<em>name</em>.classes</code>:
 *         specifies the client classes associated with this
 *         dispatcher. A class should appear in at most one
 *         dispatcher {@code classes} list. eBus does not detect
 *         if a class is assigned to multiple dispatchers and it
 *         is unknown to which dispatcher this class will be
 *         assigned.
 *         <p>
 *          This property is ignored if {@code isDefault} is
 *          {@code true}.
 *         </p>
 *         <p>
 *         This property is required if this not a default
 *         dispatcher.
 *         </p>
 *       </li>
 *       <li>
 *         <code>eBus.dispatcher.<em>name</em>.numberThreads</code>:
 *         The number of threads in this dispatcher. Must be
 *         &gt; zero.
 *         <p>
 *         The default value is {@link #DEFAULT_NUMBER_THREADS}.
 *         </p>
 *       </li>
 *     </ul>
 *   </li>
 * </ul>
 * See {@link EConfigure.Dispatcher} for information about two
 * special, pre-defined {@code Dispatcher} types:
 * {@link DispatcherType#SWING} and {@link DispatcherType#JAVAFX}
 * which deliver eBus messages via the GUI thread.
 * <p>
 * See the eBus overview section on connecting eBus applications
 * for a detailed description in configuring eBus connections.
 * </p>
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class EConfigure
{
//---------------------------------------------------------------
// Enums.
//

    /**
     * Enumerates the channel types supported by eBus
     * {@code ERemoteApp}.
     */
    public enum ConnectionType
    {
        /**
         * Remote application connection is a TCP socket.
         */
        TCP,

        /**
         * Remote application connection is a TCP socket with
         * TLS (Transport Layer Security).
         */
        SECURE_TCP

        // TODO: Provide in a future release.
        /*
         * Remote application connection is a UDP socket.
         */
//        UDP
    } // end of ConnectionType

    /**
     * Enumerates the supported eBus dispatcher thread types.
     */
    public enum DispatcherType
    {
        /**
         * Default eBus dispatcher based on a run queue, so the
         * dispatcher method is
         * {@code EClient.doDispatch(Runnable)}.
         */
        EBUS (false, null),

        /**
         * Posts a task to the JavaFX GUI thread using
         * {@code javafx.application.Platform.runLater(Runnable)}.
         */
        JAVAFX (true,
                task ->
                {
                    // If currently running in the JavaFX GUI
                    // thread, then execute the task immediately.
                    if (javafx.application.Platform.isFxApplicationThread())
                    {
                        try
                        {
                            task.run();
                        }
                        catch (Exception jex)
                        {
                            System.err.println(
                                jex.getLocalizedMessage());
                        }

                    }
                    else
                    {
                        javafx.application.Platform.runLater(task);
                    }
                }),

        /**
         * Posts a task to the Swing GUI thread using
         * {@code javax.swing.SwingUtilities.invokeLater(Runnable)}.
         */
        SWING (true,
               task ->
               {
                   // If currently running in the Swing GUI
                   // thread, then execute the task immediately.
                   if (javax.swing.SwingUtilities.isEventDispatchThread())
                   {
                       try
                       {
                           task.run();
                       }
                       catch (Exception jex)
                       {
                            System.err.println(
                                jex.getLocalizedMessage());
                       }
                   }
                   else
                   {
                       javax.swing.SwingUtilities.invokeLater(task);
                   }
               });

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new dispatcher type for the given run queue
         * and handle.
         * @param special marks this as a special dispatcher
         * which may not be configured.
         * @param handle dispatcher method handle. Must be
         * {@code null} when {@code handle} is not {@code null}.
         */
        private DispatcherType(final boolean special,
                               final Consumer<Runnable> handle)
        {
            _special = special;
            _dispatchHandle = handle;
        } // end of DispatcherType(boolean, Consumer<>)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns {@code true} if this dispatcher type is a
         * special, non-configurable dispatcher.
         * @return {@code true} if a special dispatcher.
         */
        public boolean isSpecial()
        {
            return (_special);
        } // end of isSpecial()

        /**
         * Returns the task dispatch method handle. May be
         * {@code null}.
         * @return dispatch method handle.
         */
        public Consumer<Runnable> dispatchHandle()
        {
            return (_dispatchHandle);
        } // end of dispatchHandle()

        /**
         * Returns the {@link DispatcherType} for the given
         * {@code name} using case-insensitive string matching.
         * If {@code name} is either {@code null} or an empty
         * string, then returns {@link DispatcherType#EBUS} by
         * default.
         * @param name find dispatcher type with the given name
         * using case-insensitive matching.
         * @return matching dispatcher type; defaults to
         * {@link #EBUS}.
         */
        public static DispatcherType findType(final String name)
        {
            final DispatcherType[] types =
                DispatcherType.values();
            final int numTypes = types.length;
            int index;
            DispatcherType retval = null;

            for (index = 0;
                 index < numTypes && retval == null;
                 ++index)
            {
                if (name.equalsIgnoreCase(types[index].name()))
                {
                    retval = types[index];
                }
            }

            if (retval == null)
            {
                retval = EBUS;
            }

            return (retval);
        } // end of findType(String)

        //
        // end of Get Methods.
        //-------------------------------------------------------

    //-----------------------------------------------------------
    // Member data.
    //

        /**
         * Special dispatchers may only be marked as the default
         * dispatcher. All other properties are ignored.
         */
        private final boolean _special;

        /**
         * References the method used to dispatch a task. Will be
         * {@code null} if {@code _runQueue} is not {@code null}.
         */
        private final Consumer<Runnable> _dispatchHandle;
    } // end of enum DispatcherType

//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Use {@code -D} to set system property {@value} to point to
     * the eBus configuration property file. eBus will configure
     * itself as per the file's properties.
     */
    public static final String CONFIG_FILE_ENV =
        "net.sf.eBus.config.file";

    //
    // System property keys.
    //

    // Property keys.

    /**
     * The key {@value} contains a comma-separated list of local
     * service names.
     */
    public static final String SERVICES_KEY = "eBus.services";

    /**
     * Service keys are prefixed by {@value} and followed by the
     * service name.
     */
    public static final String SERVICE_PREFIX = "eBus.service.";

    /**
     * The key {@value} contains a comma-separated list of remote
     * connection names.
     */
    public static final String CONNECTIONS_KEY =
        "eBus.connections";

    /**
     * Remote connection keys are prefixed by {@value} and
     * followed by the connection name as found in
     * {@link #CONNECTIONS_KEY}.
     */
    public static final String CONNECTION_PREFIX =
        "eBus.connection.";

    /**
     * The key {@value} contains a comma-separated list of
     * unique, client dispatcher names.
     */
    public static final String DISPATCHERS_KEY =
        "eBus.dispatchers";

    /**
     * Dispatcher keys are prefixed by {@value} and followed by
     * the dispatcher name as found in {@link #DISPATCHERS_KEY}.
     */
    public static final String DISPATCHER_PREFIX =
        "eBus.dispatcher.";

    /**
     * Both the service and connection definitions use the
     * {@value} key suffix. Must be an integer &gt; zero.
     *
     */
    public static final String PORT_KEY = ".port";

    /**
     * Both the service and connection definitions use the
     * {@value} key suffix. Must be an integer &gt; zero.
     */
    public static final String INBUFFER_SIZE_KEY =
        ".inputBufferSize";

    /**
     * Both the service and connection definitions use the
     * {@value} key suffix. Must be an integer &gt; zero.
     */
    public static final String OUTBUFFER_SIZE_KEY =
        ".outputBufferSize";

    /**
     * Both the service and connection definitions use the
     * {@value} key suffix. Must be either
     * {@link ByteOrder#BIG_ENDIAN} or
     * {@link ByteOrder#LITTLE_ENDIAN}.
     * <p>
     * The default is {@link ByteOrder#LITTLE_ENDIAN}.
     * </p>
     */
    public static final String BYTE_ORDER_KEY =
        ".byteOrder";

    /**
     * Both the service and connection definitions use the
     * {@value} key suffix. Must be an integer &gt; zero.
     */
    public static final String MSG_QUEUE_SIZE_KEY =
        ".messageQueueSize";

    /**
     * The service definition uses the key suffix
     * {@value} to define the selector used to monitor the
     * service socket. Must be a known selector. If not
     * specified, then defaults to
     * {@code AsyncChannel.defaultSelector}.
     */
    public static final String SVC_SELECTOR_KEY =
        ".serviceSelector";

    /**
     * The service definition uses the key suffix
     * {@value} to define the selector used to monitor TCP socket
     * channels accepted by the service socket. Must be a known
     * selector. If not specified, then defaults to
     * {@code AsyncChannel.defaultSelector}.
     */
    public static final String CONN_SELECTOR_KEY =
        ".connectionSelector";

    /**
     * Both the service and connection definitions use the
     * {@value} key suffix. Must be a known selector. If not
     * specified, then defaults to
     * {@code AsyncChannel.defaultSelector}.
     */
    public static final String SELECTOR_KEY = ".selector";

    /**
     * The connection definition uses the {@value} key suffix.
     * Must be either a valid host name or IP address in dotted
     * notation.
     */
    public static final String HOST_KEY = ".host";

    /**
     * The connection definition uses the {@value} key suffix.
     * Must be an integer &ge; zero.
     */
    public static final String BIND_PORT_KEY = ".bindPort";

    /**
     * The connection definition uses the {@value} key suffix.
     * Must be a boolean.
     */
    public static final String RECONNECT_KEY = ".reconnect";

    /**
     * The connection definition uses the {@value} key suffix.
     * Must be an integer &ge; zero.
     */
    public static final String RECONNECT_DELAY_KEY =
        ".reconnectTime";

    /**
     * The connection definition uses the {@value} key suffix.
     * Must be an integer &ge; zero.
     */
    public static final String HB_DELAY_KEY = ".heartbeatDelay";

    /**
     * The connection definition uses the {@value} key suffix.
     * Must be an integer &ge; zero.
     */
    public static final String HB_REPLY_DELAY_KEY =
        ".heartbeatReplyDelay";

    /**
     * The service definition uses the {@value} key suffix. Must
     * be a valid address filter list.
     */
    public static final String FILTER_KEY = ".addressFilter";

    /**
     * The boolean {@value} property is used to specify when this
     * dispatcher should be used as the default dispatcher. At
     * most one dispatcher should be designated as the default
     * dispatcher. If more than one is so designated, the
     * subsequent {@code .isDefault} keys are ignored.
     * <p>
     * If no user-defined dispatcher is marked as default, then
     * a {@link ThreadType#BLOCKING blocking},
     * {@link Thread#NORM_PRIORITY normal priority} dispatcher
     * named {@code EClient.DEFAULT_DISPATCHER} is used as
     * the default dispatcher.
     * </p>
     * <p>
     * The default value is {@code false}
     * (not the default selector).
     * </p>
     */
    public static final String DEFAULT_KEY = ".isDefault";

    /**
     * The {@link DispatcherType} {@value} property is used to
     * specify the dispatcher thread type. This property allows
     * eBus tasks to be executed on supported third-party
     * threads.
     * <p>
     * The default value is {@link DispatcherType#EBUS}.
     * </p>
     */
    public static final String DISPATCHER_TYPE_KEY =
        ".dispatcherType";

    /**
     * The {@link ThreadType run queue} {@value} property is used
     * to specify how the dispatcher thread acquires the next
     * available {@code EClient} from the run queue by either
     * blocking, spinning, spin+park, or spin+yield.
     * <p>
     * The default value is {@link #DEFAULT_THREAD_TYPE}.
     * </p>
     */
    public static final String THREAD_TYPE_KEY = ".runQueueType";

    /**
     * If the thread type is
     * {@link ThreadType#SPINPARK spin+park} or
     * {@link ThreadType#SPINYIELD}, then the
     * integer {@code .spinLimit} property may be specified. This
     * setting defines the number of times the Dispatcher thread
     * may call
     * {@link java.util.Queue#poll()} before parking or yielding.
     * <p>
     * This value must be &gt; zero.
     * </p>
     * <p>
     * Default values is {@link #DEFAULT_SPIN_LIMIT}.
     * </p>
     * <p>
     * This property is ignored is the thread type is not
     * spin+park or spin+yield.
     * </p>
     */
    public static final String SPIN_LIMIT_KEY = ".spinLimit";

    /**
     * If the thread type is
     * {@link ThreadType#SPINPARK spin+park}, then the
     * integer {@code .parkTime} property may be specified. This
     * setting specifies the <em>nanosecond</em> park time taken
     * between {@link java.util.Queue#poll()} spin cycles.
     * <p>
     * This value must be &gt; zero.
     * </p>
     * <p>
     * Default values is {@link #DEFAULT_PARK_TIME}.
     * </p>
     * <p>
     * This property is ignored is the thread type is not
     * spin+park.
     * </p>
     */
    public static final String PARK_TIME_KEY = ".parkTime";

    /**
     * The integer {@value} property is used to set the dispatcher
     * thread priority.
     * <p>
     * The default value is {@link Thread#NORM_PRIORITY}.
     * </p>
     */
    public static final String PRIORITY_KEY = ".priority";

    /**
     * The key {@value} defines the run quantum each
     * client is initially assigned. If a client exhausts its
     * quantum, the client is placed back on the run queue.
     */
    public static final String QUANTUM_KEY = ".quantum";

    /**
     * The key {@value} defines the classes which are
     * posted to this dispatcher. This key is ignored if
     * {@link #DEFAULT_KEY .isDefault} is set to {@code true}.
     */
    public static final String CLASSES_KEY = ".classes";

    /**
     * The dispatcher definition uses {@value}
     * key suffix to specify the number of threads in the
     * dispatcher.
     */
    public static final String NUM_THREADS_KEY =
        ".numberThreads";

    /**
     * Array key separator as a character.
     */
    private static final char KEY_IFS = ',';

    /**
     * Array key separator as a string.
     */
    private static final String KEY_SEP = ",";

    //
    // Default values.
    //

    /**
     * Unfortunately, {@link ByteOrder} is <em>not</em> an
     * {@code enum}, so we need to define strings for
     * comparison purposes.
     */
    private static final String BIGENDIAN = "BIG_ENDIAN";

    /**
     * Unfortunately, {@link ByteOrder} is <em>not</em> an
     * {@code enum}, so we need to define strings for
     * comparison purposes.
     */
    private static final String LITTLEENDIAN = "LITTLE_ENDIAN";

    /**
     * The default Dispatcher thread type is
     * {@link ThreadType#BLOCKING}.
     */
    public static final String DEFAULT_THREAD_TYPE =
        (ThreadType.BLOCKING).textName();

    /**
     * The default spin limit is {@value} calls to
     * {@link java.util.Queue#poll()} before parking or yielding.
     */
    public static final int DEFAULT_SPIN_LIMIT = 2_500_000;

    /**
     * The default park time after spinning is {@value}
     * nanoseconds.
     */
    public static final int DEFAULT_PARK_TIME = 1_000;

    /**
     * The default dispatcher thread priority is {@value}.
     */
    public static final int DEFAULT_PRIORITY =
        Thread.NORM_PRIORITY;

    /**
     * The default run quantum is {@value} nanoseconds.
     */
    public static final int DEFAULT_QUANTUM = 500_000;

    /**
     * The default dispatcher thread count is 4.
     */
    public static final int DEFAULT_NUMBER_THREADS = 4;

    /**
     * This value specifies that there is no limit to the
     * transmit queue size.
     */
    public static final int DEFAULT_QUEUE_SIZE = 0;

    /**
     * By default a lost connection is <i>not</i> reconnected.
     */
    public static final boolean DEFAULT_RECONNECT_FLAG = false;

    /**
     * The default reconnect time is 5 seconds. This value is
     * used only if reconnection is set to {@code true} when the
     * connection was
     * {@code ERemoteApp.openConnection(EConfigure.RemoteConnection) opened}.
     */
    public static final long DEFAULT_RECONNECT_TIME = 5000;

    /**
     * By default heart-beating is turned off.
     */
    public static final long DEFAULT_HEARTBEAT_DELAY = 0L;

    /**
     * By default wait indefinitely for a heartbeat reply.
     */
    public static final long DEFAULT_HEARTBEAT_REPLY_DELAY = 0L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Open local services on the given ports. Maps the server
     * name to the local service.
     */
    private final Map<String, Service> mServices;

    /**
     * Open connections to the specified remote eBus instances.
     * Maps the connection name to the remote connection.
     */
    private final Map<String, RemoteConnection> mRemoteConnections;

    /**
     * The task dispatcher thread configurations.
     */
    private final Map<String, Dispatcher> mDispatchers;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates an eBus configuration for the given services and
     * remote connections. Each local service and remote
     * connection must have a unique name in {@code connections}.
     * @param services local eBus services.
     * @param conns the remote eBus connections.
     * @param dispatchers dispatcher thread configurations.
     * @see RemoteConnection
     * @see Service
     */
    private EConfigure(final Map<String, Service> services,
                       final Map<String, RemoteConnection> conns,
                       final Map<String, Dispatcher> dispatchers)
    {
        mServices = services;
        mRemoteConnections = conns;
        mDispatchers = dispatchers;
    } // end of EConfigure(Map<>, Map<>, Map<>)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns {@code true} if {@code o} is a non-{@code null}
     * {@code EConfigure} instance with the same settings;
     * returns {@code false} otherwise.
     * @param o Test equals with this object.
     * @return {@code true} if {@code o} is a non-{@code null}
     * {@code EConfigure} instance with the same settings;
     * returns {@code false} otherwise.
     */
    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (this == o);

        if (!retcode && o instanceof EConfigure)
        {
            final EConfigure config = (EConfigure) o;

            retcode =
                (mServices.equals(config.mServices) &&
                 mRemoteConnections.equals(
                     config.mRemoteConnections));
        }

        return (retcode);
    } // end of equals(Object)

    /**
     * Returns the configuration hash code.
     * @return the configuration hash code.
     */
    @Override
    public int hashCode()
    {
        return (super.hashCode());
    } // end of hashCode()

    /**
     * Returns the configuration text representation.
     * @return the configuration text representation.
     */
    @Override
    public String toString()
    {
        final Formatter retval = new Formatter();

        retval.format("         services:");
        if (mServices.isEmpty())
        {
            retval.format("none%n");
        }
        else
        {
            (mServices.values())
                .stream()
                .forEach((service) ->
                {
                    retval.format("%n    %s%n", service);
                });
        }
        retval.format("       connections:");
        if (mRemoteConnections.isEmpty())
        {
            retval.format(" none%n");
        }
        else
        {
            (mRemoteConnections.values())
                .stream()
                .forEach((conn) ->
                {
                    retval.format("%n    %s", conn);
                });
        }

        return (retval.toString());
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns {@code true} if at least one  service is
     * configured and {@code false} otherwise.
     * @return {@code true} if there is a service.
     */
    public boolean hasServices()
    {
        return (!mServices.isEmpty());
    } // end of hasServices()

    /**
     * Returns the eBus TCP/IP services.
     * @return the eBus TCP/IP services.
     */
    public Map<String, Service> services()
    {
        return (mServices);
    } // end of services()

    /**
     * Returns {@code true} if remote connections are configured
     * and {@code false} if not.
     * @return {@code true} if there are remote connections.
     */
    public boolean hasRemoteConnections()
    {
        return (!mRemoteConnections.isEmpty());
    } // end of hasRemoteConnections()

    /**
     * Returns the set of remote eBus connections.
     * @return the set of remote eBus connections.
     */
    public Map<String, RemoteConnection> remoteConnections()
    {
        return (mRemoteConnections);
    } // end of remoteConnections()

    /**
     * Returns the set of client dispatcher configurations.
     * @return client dispatcher configurations.
     */
    public Map<String, Dispatcher> dispatchers()
    {
        return (mDispatchers);
    } // end of dispatchers()

    //
    // end of Get methods.
    //-----------------------------------------------------------

    /**
     * Creates an eBus configuration for the given service and
     * remote connections. Each remote connection must have a
     * unique name in {@code connections}.
     * @param services the local eBus services.
     * @param connections the remote eBus connections.
     * @param dispatchers dispatcher thread configurations.
     * @return an eBus configuration based on the given
     * parameters.
     * @see RemoteConnection
     * @see Service
     */
    public static EConfigure
        create(final Map<String, Service> services,
               final Map<String, RemoteConnection> connections,
               final Map<String, Dispatcher> dispatchers)
    {
        return (new EConfigure(services,
                               connections,
                               dispatchers));
    } // end of load(Service, Set<>, Set<>)

    /**
     * Returns a new {@code ServerBuilder} instance used to
     * construct an {@code EServer} programmatically.
     * @return new {@code ServerBuilder} instance.
     */
    public static ServerBuilder serverBuilder()
    {
        return (new ServerBuilder());
    } // end of serverBuilder()

    /**
     * Returns a new {@link ConnectionBuilder} instance used to
     * construct a {@code ERemoteApp} programmatically.
     * @return new {@code ConnectionBuilder} instance.
     */
    public static ConnectionBuilder connectionBuilder()
    {
        return (new ConnectionBuilder());
    } // end of connectionBuilder()

    /**
     * Returns a new {@link DispatcherBuilder} instance used to
     * construct a {@link Dispatcher}.
     * @return new {@code DispatcherBuilder} instance.
     */
    public static DispatcherBuilder dispatcherBuilder()
    {
        return (new DispatcherBuilder());
    } // end of dispatcherBuilder()

    /**
     * Returns the eBus service, remote connection, and
     * dispatcher configurations found in the given properties.
     * There are four top-level keys: {@code net},
     * {@code eBus.messageLoaders}, optional {@code eBus.server}
     * and {@code ebus.remoteConnections}.
     * Both keys have a comma-separated list of unique
     * names as the value. Each name is used to extract one
     * server or remote connection configuration.
     * <p>
     * <b>Note:</b> Each name must be unique with the remote
     * remote connection list.
     * </p>
     * <p>
     * The optional server configuration keys are:
     * </p>
     * <ul>
     *   <li>
     *     {@code eBus.service.port}: The TCP
     *     service port integer. Must be &gt; zero and &lt; 65,535.
     *   </li>
     *   <li>
     *     {@code eBus.service.socketBufferSize}:
     *     The accepted connection's output buffer size.
     *   </li>
     * </ul>
     * <p>
     * The remote configuration keys are:
     * </p>
     * <ul>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .host}:
     *     The remote eBus host name or dotted notation IP
     *     address.
     *   </li>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .port}: The
     *     remote eBus TCP service port.
     *   </li>
     *   <li>
     *     {@code eBus.connection}.<i>name</i>{@code .bindPort}:
     *     the connection's local end is bound to this TCP port.
     *     If this key is unspecified or zero, then the local
     *     end is bound to any available TCP port.
     *   </li>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .socketBufferSize}:
     *     The maximum number of outbound bytes that may be
     *     waiting for transmission at a time.
     *   </li>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .reconnect}:
     *     Either {@code true} of {@code false}. If {@code true},
     *     then a lost connection will be re-established.
     *   </li>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .reconnectTime}:
     *     When reconnecting, wait this many milliseconds between
     *     connect attempts.
     *   </li>
     * </ul>
     * <p>
     * The dispatcher configuration keys are:
     * </p>
     * <ul>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .runQueueType}:
     *     the dispatcher uses either a blocking, spinning, or
     *     spin+park run queue.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .isDefault}:
     *     marks this as the default run queue used by all eBus
     *     clients not explicitly assigned a run queue. There
     *     should be only one dispatcher marked as default.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .priority}:
     *     dispatcher threads are all assigned this priority.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .quantum}:
     *     the maximum run quantum assigned to clients.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .classes}:
     *     Instances of these client classes are assigned to this
     *     dispatcher. A class should appear in at most one
     *     dispatcher classes list.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .numberThreads}:
     *     The number of threads in this dispatcher.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .spinLimit}:
     *     If {@code runQueueType} is spin+park, then this is the
     *     spin limit.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .parkTime}:
     *     If {@code runQueueType} is spin+park, then this is the
     *     nanosecond park time.
     *   </li>
     * </ul>
     * @param props extract the eBus configuration information
     * from these properties.
     * @return eBus configuration settings.
     * @throws MissingResourceException
     * if a required property is missing.
     * @throws IllegalStateException
     * if a property is invalid.
     */
    public static EConfigure load(final java.util.Properties props)
    {
        return (new EConfigure(loadServices(props),
                               loadConnections(props),
                               loadDispatchers(props)));
    } // end of load(Properties)

    /**
     * Extracts the services from the properties, returning the
     * service name to configuration map.
     * @param props eBus configuration properties.
     * @return the eBus TCP services map.
     * @throws MissingResourceException
     * if a service property is missing.
     * @throws IllegalStateException
     * if the service is invalid.
     */
    public static Map<String, Service>
        loadServices(final java.util.Properties props)
    {
        final net.sf.eBus.util.Properties p = properties(props);
        final String[] names =
            p.getArrayProperty(SERVICES_KEY, KEY_IFS);
        ServerBuilder builder;
        String keyPrefix;
        String key;
        final Map<String, Service> retval = new HashMap<>();

        for (String name: names)
        {
            keyPrefix = SERVICE_PREFIX + name;
            builder = new ServerBuilder();

            // Only plain text TCP connections are supported.
            builder.name(name)
                   .connectionType(ConnectionType.TCP);

            key = keyPrefix + PORT_KEY;
            builder.port(getPort(key, 0, p));

            key = keyPrefix + FILTER_KEY;
            builder.addressFilter(getFilter(key, p));

            key = keyPrefix +  INBUFFER_SIZE_KEY;
            builder.inputBufferSize(getSize(key,
                                            DEFAULT_BUFFER_SIZE,
                                            0,
                                            p));

            key = keyPrefix +  OUTBUFFER_SIZE_KEY;
            builder.outputBufferSize(getSize(key,
                                             DEFAULT_BUFFER_SIZE,
                                             0,
                                             p));

            key = keyPrefix + BYTE_ORDER_KEY;
            builder.byteOrder(
                getByteOrder(key,
                             ByteOrder.LITTLE_ENDIAN,
                             p));

            key = keyPrefix + MSG_QUEUE_SIZE_KEY;
            builder.messageQueueSize(getSize(key,
                                             DEFAULT_QUEUE_SIZE,
                                             0,
                                             p));

            key = keyPrefix + SVC_SELECTOR_KEY;
            builder.serviceSelector(
                p.getProperty(
                    key,
                    (ENetConfigure.defaultSelector()).name()));

            key = keyPrefix + CONN_SELECTOR_KEY;
            builder.connectionSelector(
                p.getProperty(
                    key,
                    (ENetConfigure.defaultSelector()).name()));

            key = keyPrefix + HB_DELAY_KEY;
            builder.heartbeatDelay(p.getIntProperty(key, 0));

            key = keyPrefix + HB_REPLY_DELAY_KEY;
            builder.heartbeatReplyDelay(p.getIntProperty(key, 0));

            retval.put(name, builder.build());
        }

        return (retval);
    } // end of loadServices(Properties)

    /**
     * Returns the remote eBus connections found in the
     * properties. If there are no remote connections then
     * returns an empty set.
     * @param props the eBus configuration properties.
     * @return the remote eBus connections.
     * @throws MissingResourceException
     * if there are properties missing for a specified remote
     * connection.
     * @throws IllegalStateException
     * if a property is invalid.
     */
    public static Map<String, RemoteConnection>
        loadConnections(final java.util.Properties props)
    {
        final net.sf.eBus.util.Properties p = properties(props);
        final String[] names =
            p.getArrayProperty(CONNECTIONS_KEY, KEY_IFS);
        ConnectionBuilder builder;
        String keyPrefix;
        String key;
        final Map<String, RemoteConnection> retval =
            new HashMap<>();

        for (String name: names)
        {
            keyPrefix = CONNECTION_PREFIX + name;
            builder = new ConnectionBuilder();

            // Only plain text TCP connections are supported.
            builder.name(name)
                   .connectionType(ConnectionType.TCP)
                   .address(getAddress(name, keyPrefix, p))
                   .bindPort(
                getPort(keyPrefix + BIND_PORT_KEY,
                        ANY_PORT, p))
                   .inputBufferSize(
                getSize(keyPrefix + INBUFFER_SIZE_KEY,
                        DEFAULT_BUFFER_SIZE,
                        0,
                        p))
                   .outputBufferSize(
                getSize(keyPrefix + OUTBUFFER_SIZE_KEY,
                        DEFAULT_BUFFER_SIZE,
                        0,
                        p))
                   .byteOrder(
                getByteOrder(keyPrefix + BYTE_ORDER_KEY,
                             ByteOrder.LITTLE_ENDIAN,
                             p))
                   .messageQueueSize(
                getSize(keyPrefix + MSG_QUEUE_SIZE_KEY,
                        DEFAULT_QUEUE_SIZE,
                        0,
                        p))
                   .selector(
                p.getProperty(
                    keyPrefix + SELECTOR_KEY,
                    (ENetConfigure.defaultSelector()).name()));

            key = keyPrefix + RECONNECT_KEY;
            builder.reconnect(p.getBooleanProperty(key, false));
            if (!builder.reconnect())
            {
                builder.reconnectDelay(0L);
            }
            else
            {
                key = keyPrefix + RECONNECT_DELAY_KEY;
                builder.reconnectDelay(
                    getDelay(
                        key,
                        (int) DEFAULT_RECONNECT_TIME,
                        0L,
                        p));
            }

            builder.heartbeatDelay(
                p.getIntProperty(
                    keyPrefix + HB_DELAY_KEY,
                    (int) DEFAULT_HEARTBEAT_DELAY))
                   .heartbeatReplyDelay(
                p.getIntProperty(
                    keyPrefix + HB_REPLY_DELAY_KEY,
                    (int) DEFAULT_HEARTBEAT_REPLY_DELAY));

            retval.put(name, builder.build());
        }

        return (retval);
    } // end of loadConnections(Properties)

    /**
     * Returns the eBus dispatcher configuration found in the
     * given properties. If this is no dispatcher configuration
     * found in the properties, then returns an empty set.
     * @param props eBus configuration properties.
     * @return eBus dispatcher configurations.
     * @throws MissingResourceException
     * if there are properties missing for a specified remote
     * connection.
     * @throws IllegalStateException
     * if a property is invalid.
     */
    public static Map<String, Dispatcher>
        loadDispatchers(final java.util.Properties props)
    {
        final net.sf.eBus.util.Properties p = properties(props);
        final String[] names =
            p.getArrayProperty(DISPATCHERS_KEY, KEY_IFS);
        DispatcherBuilder builder;
        final Map<String, Dispatcher> retval = new HashMap<>();

        for (String name : names)
        {
            builder = new DispatcherBuilder();
            builder.name(name);
            builder.dispatcherType(
                DispatcherType.findType(name));

            // If this dispatcher type is special, then only
            // certain properties apply.
            if ((builder.dispatcherType()).isSpecial())
            {
                loadSpecialDispatcher(builder, p);
            }
            else
            {
                loadDispatcher(builder, p);
            }

            retval.put(name, builder.build());
        }

        return (retval);
    } // end of loadDispatchers(Properties)

    /**
     * Returns an eBus configuration extracted from the
     * preferences. There are four top-level keys: {@code net},
     * {@code eBus.messageLoaders}, optional {@code eBus.server}
     * and {@code ebus.remoteConnections}.
     * Both keys have a comma-separated list of unique
     * names as the value. Each name is used to extract one
     * server or remote connection configuration.
     * <p>
     * <b>Note:</b> Each name must be unique with the remote
     * remote connection list.
     * </p>
     * <p>
     * The optional server configuration keys are:
     * </p>
     * <ul>
     *   <li>
     *     {@code eBus.service.port}: The TCP
     *     service port integer. Must be &gt; zero and &lt; 65,535.
     *   </li>
     *   <li>
     *     {@code eBus.service.socketBufferSize}:
     *     The accepted connection's output buffer size.
     *   </li>
     * </ul>
     * <p>
     * The remote configuration keys are:
     * </p>
     * <ul>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .host}:
     *     The remote eBus host name or dotted notation IP
     *     address.
     *   </li>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .port}: The
     *     remote eBus TCP service port.
     *   </li>
     *   <li>
     *     {@code eBus.connection}.<i>name</i>{@code .bindPort}:
     *     the connection's local end is bound to this TCP port.
     *     If this key is unspecified or zero, then the local
     *     end is bound to any available TCP port.
     *   </li>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .socketBufferSize}:
     *     The maximum number of outbound bytes that may be
     *     waiting for transmission at a time.
     *   </li>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .reconnect}:
     *     Either {@code true} of {@code false}. If {@code true},
     *     then a lost connection will be re-established.
     *   </li>
     *   <li>
     *     {@code eBus.connection.}<i>name</i>{@code .reconnectTime}:
     *     When reconnecting, wait this many milliseconds between
     *     connect attempts.
     *   </li>
     * </ul>
     * <p>
     * The dispatcher configuration keys are:
     * </p>
     * <ul>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .runQueueType}:
     *     the dispatcher uses either a blocking, spinning, or
     *     spin+park run queue.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .isDefault}:
     *     marks this as the default run queue used by all eBus
     *     clients not explicitly assigned a run queue. There
     *     should be only one dispatcher marked as default.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .priority}:
     *     dispatcher threads are all assigned this priority.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .quantum}:
     *     the maximum run quantum assigned to clients.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .classes}:
     *     Instances of these client classes are assigned to this
     *     dispatcher. A class should appear in at most one
     *     dispatcher classes list.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .numberThreads}:
     *     The number of threads in this dispatcher.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .spinLimit}:
     *     If {@code runQueueType} is spin+park, then this is the
     *     spin limit.
     *   </li>
     *   <li>
     *     {@code eBus.dispatcher.}<i>name</i>{@code .parkTime}:
     *     If {@code runQueueType} is spin+park, then this is the
     *     nanosecond park time.
     *   </li>
     * </ul>
     * @param prefs Read in the eBus configuration from these
     * preferences.
     * @return an eBus configuration loaded from the preferences.
     * @exception MissingResourceException
     * if a required property is missing.
     * @throws IllegalStateException
     * if a property contains an invalid value.
     *
     */
    public static EConfigure load(final Preferences prefs)
    {
        return (new EConfigure(loadServices(prefs),
                               loadConnections(prefs),
                               loadDispatchers(prefs)));
    } // end of load(Preferences)

    /**
     * Returns the eBus TCP service. Returns {@code null} if
     * there is no eBus service.
     * @param p eBus configuration preferences.
     * @return the eBus TCP service.
     * @throws MissingResourceException
     * if a service property is missing.
     * @throws IllegalStateException
     * if a service property is invalid.
     */
    public static Map<String, Service>
        loadServices(final Preferences p)
    {
        final String[] names =
            (p.get(SERVICES_KEY, "")).split(",");
        ServerBuilder builder;
        String keyPrefix;
        String key;
        final Map<String, Service> retval = new HashMap<>();

        for (String name: names)
        {
            keyPrefix = SERVICE_PREFIX + name;
            builder = new ServerBuilder();

            // Only plain text TCP connections are supported.
            builder.name(name)
                   .connectionType(ConnectionType.TCP);

            key = keyPrefix + PORT_KEY;
            builder.port(getPort(key, 0, p));

            key = keyPrefix + FILTER_KEY;
            builder.addressFilter(getFilter(key, p));

            key = keyPrefix +  INBUFFER_SIZE_KEY;
            builder.inputBufferSize(
                getSize(key,
                        DEFAULT_BUFFER_SIZE,
                        0,
                        p));

            key = keyPrefix + OUTBUFFER_SIZE_KEY;
            builder.outputBufferSize(
                getSize(key,
                        DEFAULT_BUFFER_SIZE,
                        0,
                        p));

            key = keyPrefix + BYTE_ORDER_KEY;
            builder.byteOrder(
                getByteOrder(
                    key, ByteOrder.LITTLE_ENDIAN, p));

            key = keyPrefix + MSG_QUEUE_SIZE_KEY;
            builder.messageQueueSize(
                getSize(key,
                        DEFAULT_QUEUE_SIZE,
                        0,
                        p));

            key = keyPrefix + SVC_SELECTOR_KEY;
            builder.serviceSelector(
                p.get(key,
                      (ENetConfigure.defaultSelector()).name()));

            key = keyPrefix + SELECTOR_KEY;
            builder.connectionSelector(
                p.get(key,
                      (ENetConfigure.defaultSelector()).name()));

            key = keyPrefix + HB_DELAY_KEY;
            builder.heartbeatDelay(p.getLong(key, 0L));

            key = keyPrefix + HB_REPLY_DELAY_KEY;
            builder.heartbeatReplyDelay(p.getLong(key, 0L));

            retval.put(name, builder.build());
        }

        return (retval);
    } // end of loadServices(Preferences)

    /**
     * Returns the remote eBus connections found in the
     * preferences. If there are no remote connections then
     * returns an empty set.
     * @param p the eBus configuration preferences.
     * @return the remote eBus connections.
     * @throws MissingResourceException
     * if there are properties missing for a specified remote
     * connection.
     * @throws IllegalStateException
     * if a property is invalid.
     */
    public static Map<String, RemoteConnection>
        loadConnections(final Preferences p)
    {
        final String[] names =
            (p.get(CONNECTIONS_KEY, "")).split(",");
        String keyPrefix;
        ConnectionBuilder builder;
        String key;
        final Map<String, RemoteConnection> retval =
            new HashMap<>();

        for (String name: names)
        {
            keyPrefix = CONNECTION_PREFIX + name;
            builder = new ConnectionBuilder();

            // Only plain text TCP connections are supported.
            builder.name(name)
                   .connectionType(ConnectionType.TCP)
                   .address(getAddress(name, keyPrefix, p));

            key = keyPrefix + BIND_PORT_KEY;
            builder.bindPort(
                getPort(key, ANY_PORT, p));

            // The output buffer size property is not required.
            builder.inputBufferSize(
                getSize(keyPrefix + INBUFFER_SIZE_KEY,
                        DEFAULT_BUFFER_SIZE,
                        0,
                        p));
            builder.outputBufferSize(
                getSize(keyPrefix + OUTBUFFER_SIZE_KEY,
                        DEFAULT_BUFFER_SIZE,
                        0,
                        p));
            builder.byteOrder(
                getByteOrder(keyPrefix + BYTE_ORDER_KEY,
                             ByteOrder.LITTLE_ENDIAN,
                             p));
            builder.messageQueueSize(
                getSize(keyPrefix + MSG_QUEUE_SIZE_KEY,
                        DEFAULT_QUEUE_SIZE,
                        0,
                        p));
            builder.selector(
                p.get(keyPrefix + SELECTOR_KEY,
                      (ENetConfigure.defaultSelector()).name()));

            key = keyPrefix + RECONNECT_KEY;
            builder.reconnect(p.getBoolean(key, false));
            if (!builder.reconnect())
            {
                builder.reconnectDelay(0L);
            }
            else
            {
                key = keyPrefix + RECONNECT_DELAY_KEY;
                builder.reconnectDelay(
                    getDelay(
                        key,
                        (int) DEFAULT_RECONNECT_TIME,
                        0L,
                        p));
            }

            key = keyPrefix + HB_DELAY_KEY;
            builder.heartbeatDelay(
                p.getLong(
                    key,
                    DEFAULT_HEARTBEAT_DELAY));

            key = keyPrefix + HB_REPLY_DELAY_KEY;
            builder.heartbeatReplyDelay(
                p.getLong(
                    key,
                    DEFAULT_HEARTBEAT_REPLY_DELAY));

            retval.put(name, builder.build());
        }

        return (retval);
    } // end of loadConnections()

    /**
     *
     * Returns the eBus dispatcher configuration found in the
     * given preferences. If this is no dispatcher configuration
     * found in the preferences, then returns an empty set.
     * @param p the eBus configuration preferences.
     * @return eBus dispatcher configurations.
     * @throws MissingResourceException
     * if there are properties missing for a specified remote
     * connection.
     */
    public static Map<String, Dispatcher>
        loadDispatchers(final Preferences p)
    {
        final String[] names =
            (p.get(DISPATCHERS_KEY, "")).split(",");
        DispatcherBuilder builder;
        final Map<String, Dispatcher> retval = new HashMap<>();

        for (String name : names)
        {
            builder = new DispatcherBuilder();
            builder.name(name);
            builder.dispatcherType(
                DispatcherType.findType(name));

            // If this dispatcher type is special, then only
            // certain properties apply.
            if ((builder.dispatcherType()).isSpecial())
            {
                loadSpecialDispatcher(builder, p);
            }
            else
            {
                loadDispatcher(builder, p);
            }

            retval.put(name, builder.build());
        }

        return (retval);
    } // end of loadDispatchers(Preferences)

    /**
     * Stores the eBus configuration to the given properties.
     * @param props store the configuration to this properties
     * map.
     * @see EConfigure#load(java.util.Properties)
     * @see EConfigure#load(java.util.prefs.Preferences)
     * @see EConfigure#store(java.util.prefs.Preferences)
     */
    public void store(final java.util.Properties props)
    {
        storeService(props);
        storeConnections(props);
        storeDispatchers(props);

        return;
    } // end of store(Properties)

    /**
     * Stores the eBus configuration to the given preferences.
     * @param prefs store the configuration to these preferences.
     * @see EConfigure#load(Preferences)
     * @see EConfigure#load(java.util.Properties)
     * @see EConfigure#store(java.util.Properties)
     */
    public void store(final Preferences prefs)
    {
        storeService(prefs);
        storeConnections(prefs);
        storeDispatchers(prefs);

        return;
    } // end of store(Preferences)

    /**
     * Returns the given {@link java.util.Properties} as
     * {@link net.sf.eBus.util.Properties}. If {@code props} is
     * an eBus properties, then performs a typecast. Otherwise,
     * creates a new eBus properties instance based on the the
     * Java properties.
     * @param props convert to eBus properties.
     * @return eBus properties.
     */
    private static net.sf.eBus.util.Properties properties(final java.util.Properties props)
    {
        net.sf.eBus.util.Properties retval;

        if (props instanceof net.sf.eBus.util.Properties)
        {
            retval = (net.sf.eBus.util.Properties) props;
        }
        else
        {
            retval = new net.sf.eBus.util.Properties(props);
        }

        return (retval);
    } // end of properties(Properties)

    /**
     * Returns the named {@link DispatcherType#EBUS eBus}
     * dispatcher loaded in from the given properties.
     * @param name eBus dispatcher name.
     * @param p properties containing the dispatcher
     * configuration.
     * @return eBus dispatcher.
     * @throws MissingResourceException
     * if {@code p} is missing a property required for the
     * eBus dispatcher definition.
     * @throws IllegalStateException
     * if there are unsupported keys for this special dispatcher.
     */
    private static void loadDispatcher(final DispatcherBuilder builder,
                                       final net.sf.eBus.util.Properties p)
    {
        String keyName = DISPATCHER_PREFIX + builder.name();
        String key;

        key = keyName + THREAD_TYPE_KEY;
        builder.threadType(
            ThreadType.find(
                p.getProperty(key, DEFAULT_THREAD_TYPE)));

        if (builder.threadType() == ThreadType.SPINPARK ||
            builder.threadType() == ThreadType.SPINYIELD)
        {
            key = keyName + SPIN_LIMIT_KEY;
            builder.spinLimit(
                p.getIntProperty(key, DEFAULT_SPIN_LIMIT));
        }

        if (builder.threadType() == ThreadType.SPINPARK)
        {
            key = keyName + PARK_TIME_KEY;
            builder.parkTime(
                p.getIntProperty(key, DEFAULT_PARK_TIME));
        }

        key = keyName + PRIORITY_KEY;
        builder.priority(p.getIntProperty(key, DEFAULT_PRIORITY));

        key = keyName + QUANTUM_KEY;
        builder.quantum(p.getIntProperty(key, DEFAULT_QUANTUM));

        key = keyName + NUM_THREADS_KEY;
        builder.numberThreads(
            p.getIntProperty(key, DEFAULT_NUMBER_THREADS));

        key = keyName + DEFAULT_KEY;
        builder.isDefault(p.getBooleanProperty(key, false));

        // Is this the default dispatcher?
        if (builder.isDefault())
        {
            // Yes. Then use an empty class array.
            builder.classes(new Class<?>[0]);
        }
        // No. Read in the class names and create the class
        // array from that.
        else
        {
            key = keyName + CLASSES_KEY;
            builder.classes(getClasses(key, p));
        }

        return;
    } // end of loadDispatcher(String, Properties)

    /**
     * Returns the named special dispatcher loaded from the given
     * properties. Special dispatchers only use the ".isDefault"
     * and ".classes" attributes.
     * @param name special dispatcher name.
     * @param dType special dispatcher type.
     * @param p read in dispatcher configuration from here.
     * @return special dispatcher.
     * @throws MissingResourceException
     * if {@code p} is missing a property required for the
     * special dispatcher definition.
     */
    private static void loadSpecialDispatcher(final DispatcherBuilder builder,
                                              final net.sf.eBus.util.Properties p)
    {
        String keyName = DISPATCHER_PREFIX + builder.name();
        String key;

        key = keyName + DEFAULT_KEY;
        builder.isDefault(p.getBooleanProperty(key, false));

        // Is this the default dispatcher?
        if (builder.isDefault())
        {
            // Yes. Then use an empty class array.
            builder.classes(new Class<?>[0]);
        }
        // No. Read in the class names and create the class
        // array from that.
        else
        {
            key = keyName + CLASSES_KEY;
            builder.classes(getClasses(key, p));
        }

        // Set the priority, quantum, and numThreads to zero
        // since special Dispatcher threads are 3rd party and
        // not configured by eBus. Use a blocking thread type.
        builder.threadType(ThreadType.BLOCKING)
               .spinLimit(0L)
               .parkTime(0L)
               .priority(0)
               .quantum(0L)
               .numberThreads(0);

        return;
    } // end of loadSpecialDispatcher(...)

    /**
     * Returns the named {@link DispatcherType#EBUS eBus}
     * dispatcher loaded in from the given preferences
     * @param name eBus dispatcher name.
     * @param p preferences containing the dispatcher
     * configuration.
     * @return eBus dispatcher.
     * @throws MissingResourceException
     * if {@code p} is missing a key required for the
     * eBus dispatcher definition.
     */
    private static void loadDispatcher(final DispatcherBuilder builder,
                                       final Preferences p)
    {
        String keyName = DISPATCHER_PREFIX + builder.name();
        String key;

        key = keyName + THREAD_TYPE_KEY;
        builder.threadType(
            ThreadType.find(p.get(key, DEFAULT_THREAD_TYPE)));

        if (builder.threadType() == ThreadType.SPINPARK ||
            builder.threadType() == ThreadType.SPINYIELD)
        {
            key = keyName + SPIN_LIMIT_KEY;
            builder.spinLimit(p.getInt(key, DEFAULT_SPIN_LIMIT));
        }

        if (builder.threadType() == ThreadType.SPINPARK)
        {
            key = keyName + PARK_TIME_KEY;
            builder.parkTime(p.getInt(key, DEFAULT_PARK_TIME));
        }

        key = keyName + PRIORITY_KEY;
        builder.priority(p.getInt(key, DEFAULT_PRIORITY));

        key = keyName + QUANTUM_KEY;
        builder.quantum(p.getInt(key, DEFAULT_QUANTUM));

        key = keyName + NUM_THREADS_KEY;
        builder.numberThreads(
            p.getInt(key, DEFAULT_NUMBER_THREADS));

        key = keyName + DEFAULT_KEY;
        builder.isDefault(p.getBoolean(key, false));

        // Is this the default dispatcher?
        if (builder.isDefault())
        {
            // Yes. Then use an empty class array.
            builder.classes(new Class<?>[0]);
        }
        // No. Read in the class names and create the class
        // array from that.
        else
        {
            builder.classes(getClasses(key, p));
        }

        return;
    } // end of loadDispatcher(DispatcherBuilder, Preferences)

    /**
     * Returns the named special dispatcher loaded from the given
     * preferences. Special dispatchers only use the ".isDefault"
     * and ".classes" attributes.
     * @param name special dispatcher name.
     * @param dType special dispatcher type.
     * @param p read in dispatcher configuration from here.
     * @return special dispatcher.
     * @throws MissingResourceException
     * if {@code p} is missing a property required for the
     * special dispatcher definition.
     */
    private static void loadSpecialDispatcher(final DispatcherBuilder builder,
                                              final Preferences p)
    {
        String keyName = DISPATCHER_PREFIX + builder.name();
        String key;

        key = keyName + DEFAULT_KEY;
        builder.isDefault(p.getBoolean(key, false));

        // Is this the default dispatcher?
        if (builder.isDefault())
        {
            // Yes. Then use an empty class array.
            builder.classes(new Class<?>[0]);
        }
        // No. Read in the class names and create the class
        // array from that.
        else
        {
            builder.classes(getClasses(key, p));
        }

        // Set the priority, quantum, and numThreads to zero
        // since special Dispatcher threads are 3rd party and
        // not configured by eBus. Use a blocking thread type.
        builder.threadType(ThreadType.BLOCKING)
               .spinLimit(0L)
               .parkTime(0L)
               .priority(0)
               .quantum(0)
               .numberThreads(0);

        return;
    } // end of loadSpecialDispatcher(...)

    private static int getPort(
        final String key,
        final int defaultPort,
        final net.sf.eBus.util.Properties p)
    {
        final int retval = p.getIntProperty(key, defaultPort);

        if ((retval < 0 && retval != ANY_PORT) ||
            retval > MAX_PORT)
        {
            throw (
                new IllegalStateException(
                    key +
                    " invalid port (" +
                    Integer.toString(retval) +
                    ")"));
        }

        return (retval);
    } // end of getPort(String, Properties)

    private static int getPort(final String key,
                               final int defaultPort,
                               final Preferences p)
    {
        final int retval = p.getInt(key, defaultPort);

        if ((retval < 0 && retval != ANY_PORT) ||
            retval > MAX_PORT)
        {
            throw (
                new IllegalStateException(
                    key +
                    " invalid port (" +
                    Integer.toString(retval) +
                    ")"));
        }


        return (retval);
    } // end of getPort(String, int, Preferences)

    private static AddressFilter getFilter(
        final String key,
        final net.sf.eBus.util.Properties p)
    {
        AddressFilter retval;

        try
        {
            retval = AddressFilter.parse(p.getProperty(key));
        }
        catch (ParseException parsex)
        {
            throw (new IllegalStateException(parsex));
        }

        return (retval);
    } // end of getFilter(String, Properties)

    private static AddressFilter getFilter(final String key,
                                           final Preferences p)
    {
        AddressFilter retval;

        try
        {
            retval = AddressFilter.parse(p.get(key, null));
        }
        catch (ParseException parsex)
        {
            throw (new IllegalStateException(parsex));
        }

        return (retval);
    } // end of getFilter(String, Preferences)

    private static InetSocketAddress getAddress(final String name,
                                                final String keyPrefix,
                                                final net.sf.eBus.util.Properties p)
    {
        String key;
        String host;
        int port;
        InetSocketAddress retval;

        key = keyPrefix + HOST_KEY;
        host = p.getProperty(key);
        if (host == null || host.isEmpty())
        {
            throw (
                new MissingResourceException(
                    name + " missing host",
                    (InetSocketAddress.class).getName(),
                    key));
        }

        key = keyPrefix + PORT_KEY;
        port = getPort(key, -1, p);

        try
        {
            retval =
                new InetSocketAddress(
                    InetAddress.getByName(host), port);
        }
        catch (UnknownHostException jex)
        {
            throw (
                new IllegalStateException(
                    name + " invalid address \"" + host + "\"",
                jex));
        }

        return (retval);
    } // end of getAddress(String, String, Properties)

    private static InetSocketAddress getAddress(final String name,
                                                final String keyPrefix,
                                                final Preferences p)
    {
        String key;
        String host;
        int port;
        InetSocketAddress retval;

        key = keyPrefix + HOST_KEY;
        host = p.get(key, null);
        if (host == null || host.length() == 0)
        {
            throw (
                new MissingResourceException(
                    name + " missing host",
                    (InetSocketAddress.class).getName(),
                    key));
        }

        key = keyPrefix + PORT_KEY;
        port = getPort(key, ANY_PORT, p);

        try
        {
            retval =
                new InetSocketAddress(
                    InetAddress.getByName(host), port);
        }
        catch (UnknownHostException jex)
        {
            throw (
                new IllegalStateException(
                    name + " invalid address \"" + host + "\"",
                jex));
        }

        return (retval);
    } // end of getAddress(String, String, Preferences)

    private static int getSize(
        final String key,
        final int defaultValue,
        final int minValue,
        final net.sf.eBus.util.Properties p)
    {
        final int retval = p.getIntProperty(key, defaultValue);

        if (retval < minValue)
        {
            throw (
                new IllegalStateException(
                    String.format("%s < %,d", key, minValue)));
        }

        return (retval);
    } // end of getSize(String, int, Properties)

    /**
     * Returns the size value stored in the given key and
     * preferences.
     * @param key the preference key.
     * @param defaultValue this value is returned if {@code key}
     * is not in the preferences.
     * @param minValue the minimum accepted values.
     * @param p the user preferences.
     * @return the size value retrieved from the preferences.
     * @throws IllegalStateException
     * if {@code key} does not contain a valid size.
     */
    private static int getSize(final String key,
                               final int defaultValue,
                               final int minValue,
                               final Preferences p)
    {
        final int retval = p.getInt(key, defaultValue);

        if (retval < minValue)
        {
            throw (
                new IllegalStateException(
                    key + " < " + Integer.toString(minValue)));
        }

        return (retval);
    } // end of getSize(String, int, Preferences)

    /**
     * Returns the buffer byte order stored in the given key and
     * properties.
     * @param key the property key.
     * @param defaultValue the default byte order.
     * @param p the properties.
     * @return the buffer byte order.
     * @throws IllegalStateException
     * if {@code key} contains an invalid byte order.
     */
    private static ByteOrder getByteOrder(
        final String key,
        final ByteOrder defaultValue,
        final net.sf.eBus.util.Properties p)
    {
        final String value = p.getProperty(key);
        ByteOrder retval;

        // Was a big endian order selected?
        if (BIGENDIAN.equalsIgnoreCase(value))
        {
            // Yes.
            retval = ByteOrder.BIG_ENDIAN;
        }
        // Was a little endian order selected?
        else if (LITTLEENDIAN.equalsIgnoreCase(value))
        {
            // Yes.
            retval = ByteOrder.LITTLE_ENDIAN;
        }
        // Was any order selected?
        else if (value == null || value.isEmpty())
        {
            // No, use the default value.
            retval = defaultValue;
        }
        // Otherwise, the key was set but to what I don't know.
        else
        {
            throw (
                new IllegalStateException(
                    String.format("%s value \"%s\" is invalid",
                                  key,
                                  value)));
        }

        return (retval);
    } // end of getByteOrder(String, ByteOrder, Properties)

    /**
     * Returns the buffer byte order stored in the given key and
     * preferences.
     * @param key the preference key.
     * @param defaultValue the default byte order.
     * @param p the user preferences.
     * @return the byte order.
     * @throws IllegalStateException
     * if {@code key} contains an invalid byte order.
     */
    private static ByteOrder getByteOrder(
        final String key,
        final ByteOrder defaultValue,
        final Preferences p)
    {
        final String value = p.get(key, null);
        ByteOrder retval;

        // Was a big endian order selected?
        if (BIGENDIAN.equalsIgnoreCase(value))
        {
            // Yes.
            retval = ByteOrder.BIG_ENDIAN;
        }
        // Was a little endian order selected?
        else if (LITTLEENDIAN.equalsIgnoreCase(value))
        {
            // Yes.
            retval = ByteOrder.LITTLE_ENDIAN;
        }
        // Was any order selected?
        else if (value == null || value.isEmpty())
        {
            // No, use the default value.
            retval = defaultValue;
        }
        // Otherwise, the key was set but to what I don't know.
        else
        {
            throw (
                new IllegalStateException(
                    String.format("%s value \"%s\" invalid",
                                  key,
                                  value)));
        }

        return (retval);
    } // end of getByteOrder(String, ByteOrder, Preferences)

    private static long getDelay(
        final String key,
        final int defaultValue,
        final long minValue,
        final net.sf.eBus.util.Properties p)
    {
        final long retval = p.getIntProperty(key, defaultValue);

        if (retval < minValue)
        {
            throw (new IllegalStateException("negative " + key));
        }

        return (retval);
    } // end of getDelay(String, long, Properties)

    private static long getDelay(final String key,
                                 final int defaultValue,
                                 final long minValue,
                                 final Preferences p)
    {
        final long retval = p.getInt(key, defaultValue);

        if (retval < minValue)
        {
            throw (new IllegalStateException("negative " + key));
        }

        return (retval);
    } // end of getDelay(String, int, Preferences)

    /**
     * Returns the classes listed in the ".classes" Dispatcher
     * property.
     * @param key the fully-qualified ".classes" key.
     * @param p extract the classes from this property.
     * @return classes array.
     * @throws MissingResourceException
     * if the ".classes" property is either missing, empty, or
     * contains an invalid class name.
     */
    private static Class<?>[]
        getClasses(final String key,
                   final net.sf.eBus.util.Properties p)
    {
        final String[] classNames =
            p.getArrayProperty(key, KEY_IFS);
        final int size = classNames.length;
        int index;
        final Class<?>[] retval;

        // If no classes were provided, then throw an
        // exception.
        if (size == 0)
        {
            throw (
                new MissingResourceException(
                    String.format(
                        "%s missing or empty", key),
                    (Class.class).getName(),
                    key));
        }

        retval = new Class<?>[size];

        for (index = 0; index < size; ++index)
        {
            try
            {
                retval[index] = Class.forName(classNames[index]);
            }
            catch (ClassNotFoundException jex)
            {
                throw (
                    new MissingResourceException(
                        String.format(
                            "unknown class %s",
                            classNames[index]),
                        (Class.class).getName(),
                        key));
            }
        }

        return (retval);
    } // end of getClasses(String, Properties)

    /**
     * Returns the classes listed in the ".classes" Dispatcher
     * property.
     * @param key the fully-qualified ".classes" key.
     * @param p extract the classes from this preference.
     * @return classes array.
     * @throws MissingResourceException
     * if the ".classes" key is either missing, empty, or
     * contains an invalid class name.
     */
    private static Class<?>[] getClasses(final String key,
                                         final Preferences p)
    {
        final String[] classNames = (p.get(key, "")).split(",");
        final int size = classNames.length;
        int index;
        final Class<?>[] retval;

        // If no classes were provided, then throw an
        // exception.
        if (size == 0)
        {
            throw (
                new MissingResourceException(
                    String.format(
                        "%s missing or empty", key),
                    (Class.class).getName(),
                    key));
        }

        retval = new Class<?>[size];
        for (index = 0; index < size; ++index)
        {
            try
            {
                retval[index] = Class.forName(classNames[index]);
            }
            catch (ClassNotFoundException jex)
            {
                throw (
                    new MissingResourceException(
                        String.format(
                            "unknown class %s",
                            classNames[index]),
                        (Class.class).getName(),
                        key));
            }
        }

        return (retval);
    } // end of getClasses(String, Preferences)

    private void storeService(final java.util.Properties p)
    {
        final Formatter names = new Formatter();
        String sep = "";
        String name;
        String keyPrefix;
        String key;
        AddressFilter filter;

        for (Service service : mServices.values())
        {
            name = service.name();
            keyPrefix = SERVICE_PREFIX + name;
            filter = service.addressFilter();

            names.format("%s%s", sep, name);

            key = keyPrefix + PORT_KEY;
            p.setProperty(key,
                          Integer.toString(service.port()));

            key = keyPrefix + FILTER_KEY;
            p.setProperty(
                key,
                (filter == null ? "" : filter.toString()));

            key = keyPrefix + BYTE_ORDER_KEY;
            p.setProperty(key,
                          (service.byteOrder()).toString());

            key = keyPrefix + INBUFFER_SIZE_KEY;
            p.setProperty(
                key,
                Integer.toString(service.inputBufferSize()));

            key = keyPrefix + OUTBUFFER_SIZE_KEY;
            p.setProperty(
                key,
                Integer.toString(service.outputBufferSize()));

            key = keyPrefix + MSG_QUEUE_SIZE_KEY;
            p.setProperty(
                key,
                Integer.toString(service.messageQueueSize()));
        }

        p.setProperty(SERVICES_KEY, names.toString());

        return;
    } // end of storeService(Properties)

    private void storeConnections(final java.util.Properties p)
    {
        final Formatter names = new Formatter();
        String sep = "";
        String name;
        String keyPrefix;
        String key;
        InetSocketAddress address;

        for (RemoteConnection connection :
                mRemoteConnections.values())
        {
            name = connection.name();
            keyPrefix = CONNECTION_PREFIX + name;
            address = connection.address();

            names.format("%s%s", sep, name);

            key = keyPrefix + HOST_KEY;
            p.setProperty(key, address.getHostName());

            key = keyPrefix + PORT_KEY;
            p.setProperty(key,
                          Integer.toString(address.getPort()));

            key = keyPrefix + BIND_PORT_KEY;
            p.setProperty(
                key,
                Integer.toString(connection.bindPort()));

            key = keyPrefix + BYTE_ORDER_KEY;
            p.setProperty(key,
                          (connection.byteOrder()).toString());

            key = keyPrefix + INBUFFER_SIZE_KEY;
            p.setProperty(
                key,
                Integer.toString(connection.inputBufferSize()));

            key = keyPrefix + OUTBUFFER_SIZE_KEY;
            p.setProperty(
                key,
                Integer.toString(connection.outputBufferSize()));

            key = keyPrefix + MSG_QUEUE_SIZE_KEY;
            p.setProperty(
                key,
                Integer.toString(connection.messageQueueSize()));

            key = keyPrefix + RECONNECT_KEY;
            p.setProperty(
                key,
                Boolean.toString(connection.reconnectFlag()));

            key = keyPrefix + RECONNECT_DELAY_KEY;
            p.setProperty(
                key,
                Long.toString(connection.reconnectTime()));

            sep = KEY_SEP;
        }

        p.setProperty(CONNECTIONS_KEY, names.toString());

        return;
    } // end of storeConnections(Properties)

    private void storeDispatchers(final java.util.Properties p)
    {
        final Formatter names = new Formatter();
        String sep = "";
        String name;
        String keyPrefix;
        String key;

        for (Dispatcher dispatcher : mDispatchers.values())
        {
            name = dispatcher.name();
            keyPrefix = DISPATCHER_PREFIX + name;

            names.format("%s%s", sep, name);

            key = keyPrefix + DEFAULT_KEY;
            p.setProperty(
                key,
                Boolean.toString(dispatcher.isDefault()));

            key = keyPrefix + PRIORITY_KEY;
            p.setProperty(
                key,
                String.valueOf(dispatcher.priority()));

            key = keyPrefix + QUANTUM_KEY;
            p.setProperty(
                key,
                Long.toString(dispatcher.quantum()));

            key = keyPrefix + NUM_THREADS_KEY;
            p.setProperty(
                key,
                Integer.toString(dispatcher.numberThreads()));

            sep = KEY_SEP;
        }

        p.setProperty(DISPATCHERS_KEY, names.toString());

        return;
    } // end of storeDispatchers(Properties)

    private void storeService(final Preferences p)
    {
        final Formatter names = new Formatter();
        String sep = "";
        String name;
        String keyPrefix;
        String key;
        AddressFilter filter;

        for (Service service : mServices.values())
        {
            name = service.name();
            keyPrefix = SERVICE_PREFIX + name;
            filter = service.addressFilter();

            names.format("%s%s", sep, name);

            key = keyPrefix + PORT_KEY;
            p.putInt(key, service.port());

            key = keyPrefix + FILTER_KEY;
            p.put(
                key, (filter == null ? "" : filter.toString()));

            key = keyPrefix + INBUFFER_SIZE_KEY;
            p.putInt(key, service.inputBufferSize());

            key = keyPrefix + OUTBUFFER_SIZE_KEY;
            p.putInt(key, service.outputBufferSize());

            key = keyPrefix + MSG_QUEUE_SIZE_KEY;
            p.putInt(key, service.messageQueueSize());
        }

        p.put(SERVICES_KEY, names.toString());

        return;
    } // end of storeService(Preferences)

    private void storeConnections(final Preferences p)
    {
        final Formatter names = new Formatter();
        String sep = "";
        String name;
        String keyPrefix;
        String key;
        InetSocketAddress address;

        for (RemoteConnection connection:
                mRemoteConnections.values())
        {
            name = connection.name();
            keyPrefix = CONNECTION_PREFIX + name;
            address = connection.address();

            names.format("%s%s", sep, name);

            key = keyPrefix + HOST_KEY;
            p.put(key, address.getHostName());

            key = keyPrefix + PORT_KEY;
            p.putInt(key, address.getPort());

            key = keyPrefix + BIND_PORT_KEY;
            p.putInt(key, connection.bindPort());

            key = keyPrefix + INBUFFER_SIZE_KEY;
            p.putInt(key, connection.inputBufferSize());

            key = keyPrefix + OUTBUFFER_SIZE_KEY;
            p.putInt(key, connection.outputBufferSize());

            key = keyPrefix + MSG_QUEUE_SIZE_KEY;
            p.putInt(key, connection.messageQueueSize());

            key = keyPrefix + RECONNECT_KEY;
            p.putBoolean(key, connection.reconnectFlag());

            key = keyPrefix + RECONNECT_DELAY_KEY;
            p.putLong(key, connection.reconnectTime());

            sep = KEY_SEP;
        }

        p.put(CONNECTIONS_KEY, names.toString());

        return;
    } // end of storeConnections(Preferences)

    private void storeDispatchers(final Preferences p)
    {
        final Formatter names = new Formatter();
        String sep = "";
        String name;
        String keyPrefix;
        String key;

        for (Dispatcher dispatcher : mDispatchers.values())
        {
            name = dispatcher.name();
            keyPrefix = DISPATCHER_PREFIX + name;

            names.format("%s%s", sep, name);

            key = keyPrefix + DEFAULT_KEY;
            p.putBoolean(key, dispatcher.isDefault());

            key = keyPrefix + PRIORITY_KEY;
            p.putInt(key, dispatcher.priority());

            key = keyPrefix + QUANTUM_KEY;
            p.putLong(key, dispatcher.quantum());

            key = keyPrefix + NUM_THREADS_KEY;
            p.putInt(key, dispatcher.numberThreads());

            sep = KEY_SEP;
        }

        p.put(DISPATCHERS_KEY, names.toString());

        return;
    } // end of storeDispatchers(Preferences)

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * This immutable class stores the configuration for an
     * eBus service. This includes:
     * <ol>
     *   <li>
     *     Service name.
     *   </li>
     *   <li>
     *     {@link ConnectionType}.
     *   </li>
     *   <li>
     *     The TCP service port.
     *   </li>
     *   <li>
     *     The optional {@link AddressFilter address filter}.
     *   </li>
     *   <li>
     *     Each accepted socket input buffer size. Limits how
     *     many bytes are received at one time.
     *   </li>
     *   <li>
     *     Each accepted socket output buffer size. Limits how
     *     many bytes are transmitted at one time.
     *   </li>
     *   <li>
     *     The accepted socket input and output buffer byte
     *     ordering.
     *   </li>
     *   <li>
     *     The accepted connection's message queue size.
     *   </li>
     *   <li>
     *     The selector thread used by the server socket.
     *   </li>
     *   <li>
     *     The selector thread use for accepted client sockets.
     *   </li>
     *   <li>
     *     The accepted connection's heartbeat frequency.
     *   </li>
     *   <li>
     *     The accepted connection's heartbeat reply time limit.
     *   </li>
     * </ol>
     */
    public static final class Service
        implements Comparable<Service>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * eBus local service name.
         */
        private final String mName;

        /**
         * The server socket accepts connections of this type.
         */
        private final ConnectionType mConnectionType;

        /**
         * TCP port.
         */
        private final int mPort;

        /**
         * The optional address filter.
         */
        private final AddressFilter mAddressFilter;

        /**
         * Input buffer size.
         */
        private final int mInputBufferSize;

        /**
         * Output buffer size.
         */
        private final int mOutputBufferSize;

        /**
         * The input and output buffer use this byte ordering.
         */
        private final ByteOrder mByteOrder;

        /**
         * Maximum message queue size.
         */
        private final int mMsgQueueSize;

        /**
         * Register this service with this specified selector.
         */
        private final String mServiceSelector;

        /**
         * Register accepted connections with this specified
         * selector.
         */
        private final String mConnSelector;

        /**
         * Send a heartbeat this many milliseconds after the
         * last received input. A zero value means heartbeating
         * is not performed.
         */
        private final long mHbDelay;

        /**
         * Wait this many milliseconds for a heartbeat reply.
         * This timer is reset when a non-heartbeat reply is
         * received. A zero value means an indefinite reply
         * time.
         */
        private final long mHbReplyDelay;

        /**
         * If {@link #mConnectionType} is
         * {@link ConnectionType#SECURE_TCP}, then this is the
         * SSL context used to secure the connection.
         */
        private final SSLContext mSSLContext;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates an eBus service configuration from the
         * builder's current settings.
         * @param builder construct service configuration from
         * these settings.
         */
        /* package */  Service(final ServerBuilder builder)
        {
            this.mName = builder.name();
            this.mConnectionType = builder.connectionType();
            this.mPort = builder.port();
            this.mAddressFilter = builder.addressFilter();
            this.mInputBufferSize = builder.inputBufferSize();
            this.mOutputBufferSize = builder.outputBufferSize();
            this.mByteOrder = builder.byteOrder();
            this.mMsgQueueSize = builder.messageQueueSize();
            this.mServiceSelector = builder.serviceSelector();
            this.mConnSelector = builder.connectionSelector();
            this.mHbDelay = builder.heartbeatDelay();
            this.mHbReplyDelay = builder.heartbeatReplyDelay();
            this.mSSLContext = builder.sslContext();
        } // end of Service(ServerBuilder)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Comparable Interface Implementation.
        //
        /**
         * Compares this service configuration with
         * {@code service} based on the service ports.
         * @param service compare this service instance.
         * @return an integer value.
         */
        @Override
        public int compareTo(
            final Service service)
        {
            return (mPort - service.mPort);
        } // end of compareTo(Service)

        //
        // end of Comparable Interface Implementation.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Object Method Overrides.
        //

        /**
         * Returns {@code true} if {@code o} is a {@code Service}
         * instance with the same port and {@code false}
         * otherwise.
         * @param o Test if this object is equal.
         * @return {@code true} if {@code o} is a {@code Service}
         * instance with the same port and {@code false}
         * otherwise.
         */
        @Override
        public boolean equals(final Object o)
        {
            boolean retcode = (this == o);

            if (!retcode && o instanceof Service)
            {
                final Service svc = (Service) o;

                retcode =
                    (mConnectionType == svc.mConnectionType &&
                     mPort == svc.mPort);
            }

            return (retcode);
        } // end of equals(Object)

        /**
         * Returns the service port.
         * @return the service port.
         */
        @Override
        public int hashCode()
        {
            return (Objects.hash(mConnectionType, mPort));
        } // end of hashCode()

        /**
         * Returns a text representation of this eBus service.
         * @return a text representation of this eBus service.
         */
        @Override
        public String toString()
        {
            final Formatter retval = new Formatter();

            retval.format("[%s %d]%n",
                          mConnectionType,
                          mPort)
                  .format("address filter: %s%n",
                                           mAddressFilter)
                  .format("socket input size: %,d%n",
                                           mInputBufferSize)
                  .format("socket output size: %,d%n",
                                           mOutputBufferSize)
                  .format(" socket byte order: %s%n",
                                           mByteOrder)
                  .format("    max queue size: %,d",
                                           mMsgQueueSize)
                  .format("          selector: %s%n",
                                           mConnSelector);

            if (mConnectionType == ConnectionType.SECURE_TCP)
            {
                retval.format("       SSL context: %s%n",
                              mSSLContext);
            }

            return (retval.toString());
        } // end of toString()

        //
        // end of Object Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the remote connection unique name.
         * @return the remote connection unique name.
         */
        public String name()
        {
            return (mName);
        } // end of name()

        /**
         * Returns the accepted channel type. This is currently
         * limited to plain text TCP and SSL/TLS secure TCP
         * connections.
         * @return connection type.
         */
        public ConnectionType connectionType()
        {
            return (mConnectionType);
        } // end of connectionType()

        /**
         * Returns the service port.
         * @return the service port.
         */
        public int port()
        {
            return (mPort);
        } // end of port()

        /**
         * Returns the address filter. May return {@code nul}.
         * @return the address filter.
         */
        public AddressFilter addressFilter()
        {
            return (mAddressFilter);
        } // end of addressFilter()

        /**
         * Returns the accepted connection input buffer size.
         * @return the accepted connection input buffer size.
         */
        public int inputBufferSize()
        {
            return (mInputBufferSize);
        } // end of inputBufferSize()

        /**
         * Returns the accepted connection output buffer size.
         * @return the accepted connection output buffer size.
         */
        public int outputBufferSize()
        {
            return (mOutputBufferSize);
        } // end of outputBufferSize()

        /**
         * Returns the accepted connection buffer byte ordering.
         * @return the accepted connection buffer byte ordering.
         */
        public ByteOrder byteOrder()
        {
            return (mByteOrder);
        } // end of byteOrder()

        /**
         * Returns the maximum number of messages on the
         * output queue. Returns zero if there is no limit.
         * @return the message queue maximum size.
         */
        public int messageQueueSize()
        {
            return (mMsgQueueSize);
        } // end of messageQueueSize()

        /**
         * Returns the selector information to which the service
         * socket channel is registered.
         * @return service socket channel selector.
         */
        public String serviceSelector()
        {
            return (mServiceSelector);
        } // end of serviceSelector()

        /**
         * Returns the selector information to which accepted
         * connections are registered.
         * @return accepted connection selector information.
         */
        public String connectionSelector()
        {
            return (mConnSelector);
        } // end of connectionSelector()

        /**
         * Returns the heartbeat delay in milliseconds. A zero
         * value means that heartbeating is not performed.
         * @return the heartbeat delay in milliseconds.
         */
        public long heartbeatDelay()
        {
            return (mHbDelay);
        } // end of heartbeatDelay()

        /**
         * Returns the heartbeat reply delay. A zero value means
         * an indefinite delay.
         * @return the heartbeat reply delay.
         */
        public long heartbeatReplyDelay()
        {
            return (mHbReplyDelay);
        } // end of heartbeatReplyDelay()

        /**
         * Returns the SSL context used for a secure connection
         * type. Returns {@code null} if connection type is
         * <em>not</em> {@link ConnectionType#SECURE_TCP}.
         * @return SSL context. May return {@code null}.
         */
        public SSLContext sslContext()
        {
            return (mSSLContext);
        } // end of sslContext()

        //
        // end of Get Methods.
        //-------------------------------------------------------
    } // end of Service

    /**
     * This immutable class stores the information pertaining
     * to a remote eBus connection. This includes:
     * <ol>
     *   <li>
     *     Service name.
     *   </li>
     *   <li>
     *     {@link ConnectionType}.
     *   </li>
     *   <li>
     *     The TCP/IP address and port.
     *   </li>
     *   <li>
     *     The TCP/IP bind port. This is is the TCP port to which
     *     the connection's local end is bound. If the bind port
     *     is zero, then the local end is bound to any available
     *     TCP port.
     *   </li>
     *   <li>
     *     The socket input buffer size. Limits how many bytes
     *     are received at one time.
     *   </li>
     *   <li>
     *     The socket output buffer size. Limits how many bytes
     *     are transmitted at one time.
     *   </li>
     *   <li>
     *     The socket input and output buffer byte ordering.
     *   </li>
     *   <li>
     *     The connection maximum message queue size . If the
     *     message queue exceeds this size, then the socket is
     *     disconnected.
     *   </li>
     *   <li>
     *     The selector thread used to monitor the socket.
     *   </li>
     *   <li>
     *     A reconnect flag which, if {@code true}, means that
     *     the connection is re-established if lost.
     *   </li>
     *   <li>
     *     When reconnecting wait this many milliseconds before
     *     attempting to re-establish the connection.
     *   </li>
     *   <li>
     *     The heartbeat frequence and heartbeat reply time
     *     limit.
     *   </li>
     * </ol>
     */
    public static final class RemoteConnection
        implements Comparable<RemoteConnection>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * eBus remote connection name.
         */
        private final String mName;

        /**
         * The server socket accepts connections of this type.
         */
        private final ConnectionType mConnectionType;

        /**
         * Remote eBus Internet address.
         */
        private final InetSocketAddress mAddress;

        /**
         * Bind the local address to this TCP port.
         */
        private final int mBindPort;

        /**
         * Socket input buffer size.
         */
        private final int mInputBufferSize;

        /**
         * Socket output buffer size.
         */
        private final int mOutputBufferSize;

        /**
         * The input and output buffer use this byte ordering.
         */
        private final ByteOrder mByteOrder;

        /**
         * Maximum message queue size.
         */
        private final int mMsgQueueSize;

        /**
         * Register this connection with the specified selector.
         */
        private final String mSelector;

        /**
         * If {@code true} then re-established lost connections.
         */
        private final boolean mReconnectFlag;

        /**
         * Wait this many seconds between reconnect attempts.
         */
        private final long mReconnectTime;

        /**
         * Send a heartbeat this many milliseconds after the
         * last received input. A zero value means heartbeating
         * is not performed.
         */
        private final long mHbDelay;

        /**
         * Wait this many milliseconds for a heartbeat reply.
         * This timer is reset when a non-heartbeat reply is
         * received. A zero value means an indefinite reply
         * time.
         */
        private final long mHbReplyDelay;

        /**
         * If {@link #mConnectionType} is
         * {@link ConnectionType#SECURE_TCP}, then this is the
         * SSL context used to secure the connection.
         */
        private final SSLContext mSSLContext;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a remote connection configuration based on the
         * builder's current parameters.
         * @param builder contains the remote configuration
         * parameters.
         */
        /* package */ RemoteConnection(final ConnectionBuilder builder)
        {
            this.mName = builder.name();
            this.mConnectionType = builder.connectionType();
            this.mAddress = builder.address();
            this.mBindPort = builder.bindPort();
            this.mInputBufferSize = builder.inputBufferSize();
            this.mOutputBufferSize = builder.outputBufferSize();
            this.mByteOrder = builder.byteOrder();
            this.mMsgQueueSize = builder.messageQueueSize();
            this.mSelector = builder.selector();
            this.mReconnectFlag = builder.reconnect();
            this.mReconnectTime = builder.reconnectDelay();
            this.mHbDelay = builder.heartbeatDelay();
            this.mHbReplyDelay = builder.heartbeatReplyDelay();
            this.mSSLContext = builder.sslContext();
        } // end of RemoteConnection(ConnectionBuilder)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Comparable Interface Implementation.
        //

        /**
         * Compares {@code this} remote connection configuration
         * with {@code conn}.
         * @param conn compare with this object.
         * @return &lt; zero if {@code this} instance is less
         * than {@code conn}; zero if {@code this} instance is
         * equal to {@code conn}; &gt; zero if {@code this} is
         * greater than {@code conn}.
         */
        @Override
        public int compareTo(
            final RemoteConnection conn)
        {
            final InetSocketAddressComparator comparator =
            new InetSocketAddressComparator();

            return (comparator.compare(mAddress, conn.mAddress));
        } // end of compareTo(RemoteConnection)

        //
        // end of Comparable Interface Implementation.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Object Method Overrides.
        //

        /**
         * Returns {@code true} if {@code o} is a
         * non-{@code null} {@code RemoteConnection} instance
         * with the same name; {@code false} otherwise.
         * @param o Test equality with this object.
         * @return {@code true} if {@code o} is a
         * non-{@code null} {@code RemoteConnection} instance
         * with the same address and connection type;
         * {@code false} otherwise.
         */
        @Override
        public boolean equals(final Object o)
        {
            boolean retcode = (this == o);

            if (!retcode && o instanceof RemoteConnection)
            {
                retcode =
                    mAddress.equals(
                        ((RemoteConnection) o).mAddress);
            }

            return (retcode);
        } // end of equals(Object)

        /**
         * Returns the remote connection address hash code.
         * @return the remote connection address hash code.
         */
        @Override
        public int hashCode()
        {
            return (mAddress.hashCode());
        } // end of hashCode()

        /**
         * Returns a text representation for this remote
         * connection.
         * @return a text representation for this remote
         * connection.
         */
        @Override
        public String toString()
        {
            final Formatter retval = new Formatter();

            retval.format("[%s] address: %s%n",
                          mName,
                          mAddress)
                  .format("     connection type: %s%n",
                          mConnectionType)
                  .format("           bind port: %d%n",
                          mBindPort)
                  .format("   input buffer size: %,d%n",
                          mInputBufferSize)
                  .format("  output buffer size: %,d%n",
                          mOutputBufferSize)
                  .format("   buffer byte order: %s%n",
                          mByteOrder)
                  .format("          queue size: %,d%n",
                          mMsgQueueSize)
                  .format("            selector: %s%n",
                          mSelector)
                  .format("           reconnect: %b%n",
                          mReconnectFlag);
            if (mReconnectFlag)
            {
                retval.format(
                    "      reconnect time: %,d msecs%n",
                    mReconnectTime);
            }

            if (mConnectionType == ConnectionType.SECURE_TCP)
            {
                retval.format("       SSL context: %s%n",
                              mSSLContext);
            }

            return (retval.toString());
        } // end of toString()

        //
        // end of Object Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get methods.
        //

        /**
         * Returns the remote connection unique name.
         * @return the remote connection unique name.
         */
        public String name()
        {
            return (mName);
        } // end of name()

        /**
         * Returns the target channel type.
         * @return channel type.
         */
        public ConnectionType connectionType()
        {
            return (mConnectionType);
        } // end of connectionType()

        /**
         * Returns the remote eBus socket address.
         * @return the remote eBus socket address.
         */
        public InetSocketAddress address()
        {
            return (mAddress);
        } // end of address()

        /**
         * Returns the connection bind port. This is the TCP
         * port to which the local end is bound.
         * @return the connection bind port.
         */
        public int bindPort()
        {
            return (mBindPort);
        } // end of bindPort()

        /**
         * Returns the socket input buffer size.
         * @return the socket input buffer size.
         */
        public int inputBufferSize()
        {
            return (mInputBufferSize);
        } // end of inputBufferSize()

        /**
         * Returns the socket output buffer size.
         * @return the socket output buffer size.
         */
        public int outputBufferSize()
        {
            return (mOutputBufferSize);
        } // end of outputBufferSize()

        /**
         * Returns the socket buffer byte ordering.
         * @return the socket buffer byte ordering.
         */
        public ByteOrder byteOrder()
        {
            return (mByteOrder);
        } // end of byteOrder()

        /**
         * Returns the maximum number of messages on the
         * output queue. Returns zero if there is no limit.
         * @return the message queue maximum size.
         */
        public int messageQueueSize()
        {
            return (mMsgQueueSize);
        } // end of messageQueueSize()

        /**
         * Returns the selector information to which this
         * connection is registered.
         * @return selector information.
         */
        public String selector()
        {
            return (mSelector);
        } // end of connSelector()

        /**
         * Returns {@code true} if connection is to be
         * re-established when lost.
         * @return {@code true} if connection is to be
         * re-established when lost.
         */
        public boolean reconnectFlag()
        {
            return (mReconnectFlag);
        } // end of reconnectFlag()

        /**
         * Returns the reconnect delay.
         * @return the reconnect delay.
         */
        public long reconnectTime()
        {
            return (mReconnectTime);
        } // end of reconnectTime()

        /**
         * Returns the heartbeat delay in milliseconds. A zero
         * value means that heartbeating is not performed.
         * @return the heartbeat delay in milliseconds.
         */
        public long heartbeatDelay()
        {
            return (mHbDelay);
        } // end of heartbeatDelay()

        /**
         * Returns the heartbeat reply delay. A zero value means
         * an indefinite delay.
         * @return the heartbeat reply delay.
         */
        public long heartbeatReplyDelay()
        {
            return (mHbReplyDelay);
        } // end of heartbeatReplyDelay()

        /**
         * Returns the SSL context used for a secure connection
         * type. Returns {@code null} if connection type is
         * <em>not</em> {@link ConnectionType#SECURE_TCP}.
         * @return SSL context. May return {@code null}.
         */
        public SSLContext sslContext()
        {
            return (mSSLContext);
        } // end of sslContext()

        //
        // end of Get methods.
        //-------------------------------------------------------
    } // end of class Remoteconnection

    /**
     * An application may create dispatch threads with a given
     * unique name and the following parameters:
     * <ul>
     *   <li>
     *     {@code Thread priority}: minimum,
     *     maximum, or normal thread priority.
     *   </li>
     *   <li>
     *     {@code Run Queue Type}: specifies whether run queue
     *     threads use a blocking, spinning, spin+park, or
     *     spin+yield technique to poll clients from the run
     *     queue.
     *   </li>
     *   <li>
     *     {@code Spin Limit}: if run queue type is either
     *     spin+park or spin+yield, then this is the number of
     *     times the dispatcher thread will poll the run queue
     *     before parking or yielding.
     *   </li>
     *   <li>
     *     {@code Park Time}: if run queue type is spin+park,
     *     then this is the number of nanoseconds the dispatcher
     *     thread will park after the spin limit is reached. When
     *     park returns, the thread will go back to spinning on
     *     poll.
     *   </li>
     *   <li>
     *     {@code Quantum}: nanosecond run quantum assigned to
     *     each client for this dispatcher group. If a client
     *     exhausts this quantum upon completing a task, the
     *     client is placed back on the end of the FIFO run queue
     *     and its quantum is reset to this value. Note that eBus
     *     does not preempt the client. It checks for quantum
     *     exhaustion when the client task returns.
     *   </li>
     *   <li>
     *     {@code Group thread count}: number of dispatcher
     *     threads in this group.
     *   </li>
     *   <li>
     *     {@code Default group flag}: {@code true} if this is
     *     the default dispatcher group. If a client is not
     *     assigned to any particular group, then it is assigned
     *     to the default group. If this flag is {@code true},
     *     then the {@code Classes} parameter is ignored. There
     *     may be only one default dispatcher group.
     *   </li>
     *   <li>
     *     {@code Classes}: a list of application classes.
     *     Clients which are instances of these classes are
     *     assigned to this dispatcher group. This parameter is
     *     valid only if this is not the default dispatcher
     *     group.
     *   </li>
     * </ul>
     * <h1>Special Dispatchers</h1>
     * There are two special, pre-defined eBus
     * {@code Dispatcher} names:
     * {@link DispatcherType#SWING "swing"} and
     * {@link DispatcherType#JAVAFX "javafx"} (case insensitive).
     * These {@code Dispatcher}s use the Swing/JavaFX GUI thread
     * to deliver eBus messages to a client. This means that the
     * client callback is free to update the GUI because the
     * callback code is running on the GUI thread.
     * <p>
     * Only two properties are supported by special
     * {@code Dispatchers}: {@link #DEFAULT_KEY} and
     * {@link #CLASSES_KEY}. All other properties are ignored by
     * special {@code Dispatchers}. The reason is that the
     * underlying GUI threads are implemented and configured by
     * the GUI package and cannot be altered by eBus.
     * </p>
     * <p>
     * It may be useful to define
     * the GUI thread Dispatcher as the default Dispatcher and
     * then create a separate eBus {@code Dispatcher} for
     * non-GUI classes. This way, an application class updating
     * the display will be assigned to the GUI thread without
     * needing to add that class to the GUI Dispatcher classes
     * property.
     * </p>
     */
    public static final class Dispatcher
        implements Comparable<Dispatcher>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * The unique dispatcher name.
         */
        private final String mName;

        /**
         * The dispatcher type. Defines the dispatch method
         * handle.
         */
        private final DispatcherType mType;

        /**
         * Specifies how the dispatcher acquires the next client
         * from the run queue: blocking, spinning, spin+park, or
         * spin+yield.
         */
        private final ThreadType mRunQueueType;

        /**
         * If {@link #mRunQueueType} is
         * {@link ThreadType#SPINPARK spin+park} or
         * {@link ThreadType#SPINYIELD spin+yield}, then spin
         * this many times on trying to acquire the next client
         * before parking/yielding.
         */
        private final long mSpinLimit;

        /**
         * If {@link #mRunQueueType} is
         * {@link ThreadType#SPINPARK spin+park}, then park for
         * this many nanoseconds before returning to spinning.
         */
        private final long mParkTime;

        /**
         * The dispatcher threads run at this priority. Must be
         * &ge; {@link Thread#MIN_PRIORITY} and
         * &le; {@link Thread#MAX_PRIORITY}.
         */
        private final int mPriority;

        /**
         * The nanosecond run quantum assigned to each client
         * in this run queue.
         */
        private final long mQuantum;

        /**
         * The dispatcher has this many threads.
         */
        private final int mNumThreads;

        /**
         * {@code true} if this dispatcher is the default for all
         * {@link EClient} instances not assigned to another,
         * specific dispatcher.
         */
        private final boolean mIsDefault;

        /**
         * Contains the classes assigned to this dispatcher. Will
         * be an empty array if {@link #mIsDefault} is
         * {@code true}.
         */
        private final Class<?>[] mClasses;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a Dispatcher group configuration based on
         * {@code builder}'s current setting.
         * @param builder contains the Dispatcher configuration
         * parameters.
         */
        /* package */ Dispatcher(final DispatcherBuilder builder)
        {
            this.mName = builder.name();
            this.mType = builder.dispatcherType();
            this.mRunQueueType = builder.threadType();
            this.mSpinLimit = builder.spinLimit();
            this.mParkTime = builder.parkTime();
            this.mPriority = builder.priority();
            this.mQuantum = builder.quantum();
            this.mNumThreads = builder.numberThreads();
            this.mIsDefault = builder.isDefault();
            this.mClasses = builder.classes();
        } // end of Dispatcher(DispatcherBuilder)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Comparable Interface Implementation.
        //

        @Override
        public int compareTo(
            final Dispatcher o)
        {
            return (mName.compareTo(o.mName));
        } // end of compareTo(Dispatcher)

        //
        // end of Comparable Interface Implementation.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Object Method Overrides.
        //

        @Override
        public boolean equals(final Object o)
        {
            boolean retcode = (this == o);

            if (!retcode)
            {
                retcode = (mName.equals(((Dispatcher) o).mName));
            }

            return (retcode);
        } // end of equals(Object)

        @Override
        public int hashCode()
        {
            return (mName.hashCode());
        } // end of hashCode()

        @Override
        public String toString()
        {
            final Formatter retval = new Formatter();

            retval.format("[%s]%n", mName);
            retval.format("       priority: %s%n", mPriority);
            retval.format("     is default: %b%n", mIsDefault);

            return (retval.toString());
        } // end of toString()

        //
        // end of Object Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get methods.
        //

        /**
         * Returns the unique dispatcher name.
         * @return dispatcher thread name.
         */
        public String name()
        {
            return (mName);
        } // end of name()

        /**
         * Returns the dispatch method type.
         * @return dispatch method type.
         */
        public DispatcherType dispatchType()
        {
            return (mType);
        } // end of dispatchType()

        /**
         * Returns the thread type which defines how the next
         * client is acquired from the run queue.
         * @return run queue-acquisition type.
         */
        public ThreadType runQueueType()
        {
            return (mRunQueueType);
        } // end of runQueueType()

        /**
         * Returns the spin limit used by
         * {@link ThreadType#SPINPARK spin+park} and
         * {@link ThreadType#SPINYIELD spin+yield} thread types.
         * @return spin limit.
         */
        public long spinLimit()
        {
            return (mSpinLimit);
        } // end of spinLimit()

        /**
         * Returns the park time used by
         * {@link ThreadType#SPINPARK spin+park} thread type.
         * @return nanosecond park time.
         */
        public long parkTime()
        {
            return (mParkTime);
        } // end of parkTime()

        /**
         * Returns {@code true} if this user-defined dispatcher
         * is the default dispatcher.
         * @return {@code true} if the default dispatcher.
         */
        public boolean isDefault()
        {
            return (mIsDefault);
        } // end of isDefault()

        /**
         * Returns the classes assigned to this dispatcher.
         * Returned array may be empty but will not be
         * {@code null}.
         * @return dispatcher class array.
         */
        public Class<?>[] classes()
        {
            return (Arrays.copyOf(mClasses, mClasses.length));
        } // end of classes()

        /**
         * Returns the dispatcher thread priority.
         * @return thread priority.
         */
        public int priority()
        {
            return (mPriority);
        } // end of priority()

        /**
         * Returns the client run-time quantum assigned by this
         * dispatcher.
         * @return client run-time quantum.
         */
        public long quantum()
        {
            return (mQuantum);
        } // end of quantum()

        /**
         * Returns the number of threads in this dispatcher.
         * @return dispatcher thread count.
         */
        public int numberThreads()
        {
            return (mNumThreads);
        } // end of numberThreads()

        //
        // end of Get methods.
        //-------------------------------------------------------
    } // end of class Dispatcher

    /**
     * Base class for {@link ServerBuilder} and
     * {@link ConnectionBuilder}, containing the properties
     * common to both.
     *
     * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
     */
    public static abstract class AbstractBuilder
    {
    //-----------------------------------------------------------
    // Member data.
    //

        /**
         * Unique service or connection name.
         */
        protected String mName;

        /**
         * Server-accepted or remote connection type is either
         * plain text TCP or SSL/TLS secure TCP.
         */
        private ConnectionType mConnectionType;

        /**
         * Maximum input buffer size. May be overridden by
         * {@link javax.net.ssl.SSLSession#getApplicationBufferSize()}
         * when using a secure TCP connection.
         */
        protected int mInputBufferSize;

        /**
         * Maximum output buffer size. May be override by
         * {@link javax.net.ssl.SSLSession#getPacketBufferSize()}
         * when using a secure TCP connection.
         */
        protected int mOutputBufferSize;

        /**
         * Code and decode messages in this byte order.
         */
        protected ByteOrder mByteOrder;

        /**
         * Maximum eBus message outbound queue size for remote
         * connections. When the queue outbound message count
         * exceeds this size, the remote connection is
         * automatically closed and the outbound messages
         * discarded.
         */
        protected int mMsgQueueSize;

        /**
         * Send a heartbeat message after this many milliseconds
         * of inbound message inactivity.
         */
        protected long mHbDelay;

        /**
         * The number of milliseconds the far-end has to respond
         * to a heartbeat. If the response is not received after
         * this many milliseconds, the remote connection is
         * automatically closed.
         */
        protected long mHbReplyDelay;

        /**
         * The SSL/TLS context used for a secure TCP connection.
         */
        protected SSLContext mSSLContext;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new abstract configuration builder.
         */
        protected AbstractBuilder()
        {
            mName = null;
            mConnectionType = ConnectionType.TCP;
            mInputBufferSize = 0;
            mOutputBufferSize = 0;
            mByteOrder = ByteOrder.LITTLE_ENDIAN;
            mMsgQueueSize = 0;
            mHbDelay = 0;
            mHbReplyDelay = 0;
            mSSLContext = null;
        } // end of AbstractBuilder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the configuration name. Returns {@code null}
         * if name is not set.
         * @return configuration name.
         */
        public final String name()
        {
            return (mName);
        } // end of name()

        /**
         * Returns the underlying channel type.
         * @return channel type.
         */
        public final ConnectionType connectionType()
        {
            return (mConnectionType);
        } // end of connectionType()

        /**
         * Returns the configured input buffer size.
         * @return input buffer size.
         */
        public final int inputBufferSize()
        {
            return (mInputBufferSize);
        } // end of inputBufferSize()

        /**
         * Returns the configured output buffer size.
         * @return output buffer size.
         */
        public final int outputBufferSize()
        {
            return (mOutputBufferSize);
        } // end of outputBufferSize()

        /**
         * Returns the configured byte order.
         * @return byte order.
         */
        public final ByteOrder byteOrder()
        {
            return (mByteOrder);
        } // end of byteOrder()

        /**
         * Returns the configured message queue size.
         * @return message queue size.
         */
        public final int messageQueueSize()
        {
            return (mMsgQueueSize);
        } // end of messageQueueSize()

        /**
         * Returns the configured heartbeat delay.
         * @return heartbeat delay.
         */
        public final long heartbeatDelay()
        {
            return (mHbDelay);
        } // end of heartbeatDelay()

        /**
         * Returns the configured heartbeat reply delay.
         * @return heartbeat reply delay.
         */
        public final long heartbeatReplyDelay()
        {
            return (mHbReplyDelay);
        } // end of heartbeatReplyDelay()

        /**
         * Returns the configured SSL context. Returns
         * {@code null} if {@link #connectionType()} is
         * <em>not</em> {@link ConnectionType#SECURE_TCP}.
         * @return SSL context instance.
         */
        public final SSLContext sslContext()
        {
            return (mSSLContext);
        } // end of sslContext()

        //
        // end of Get Methods.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the service name.
         * @param name service name.
         * @throws NullPointerException
         * if {@code name} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code name} is empty.
         */
        protected final void setName(final String name)
        {
            Objects.requireNonNull(name, "name is null");

            if (name.isEmpty())
            {
                throw (new IllegalArgumentException("name is empty"));
            }

            mName = name;

            return;
        } // end of setName(String)

        /**
         * Sets the channel type to the given value.
         * @param connType underlying channel is of this type.
         */
        protected final void setConnectionType(final ConnectionType connType)
        {
            mConnectionType =
                Objects.requireNonNull(
                    connType, "connType is null");
            return;
        } // end of setConnectionType(ConnectionType)

        /**
         * Sets the input buffer size for all accepted TCP
         * connections. If {@code size} is zero, then the default
         * input buffer size is used.
         * @param size accepted connection input buffer size.
         * @throws IllegalArgumentException
         * if {@code size} &lt; zero.
         */
        protected final void setInputBufferSize(final int size)
        {
            if (size < 0)
            {
                throw (
                    new IllegalArgumentException(
                        "input buffer size < 0"));
            }

            mInputBufferSize = size;

            return;
        } // end of setInputBufferSize(int)

        /**
         * Sets the output buffer size for all accept TCP
         * connections. If {@code size} is zero, then the default
         * output buffer size is used.
         * @param size accepted connection output buffer size.
         * @throws IllegalArgumentException
         * if {@code size} &lt; zero.
         */
        protected final void setOutputBufferSize(final int size)
        {
            if (size < 0)
            {
                throw (
                    new IllegalArgumentException(
                        "output buffer size < 0"));
            }

            mOutputBufferSize = size;

            return;
        } // end of setOutputBufferSize(int)

        /**
         * Sets the byte order used by accepted TCP connections.
         * @param byteOrder accepted connections serialize and
         * de-serialize messages using this byte order.
         * @throws NullPointerException
         * if {@code byteOrder} is {@code nuill}.
         */
        protected final void setByteOrder(final ByteOrder byteOrder)
        {
            Objects.requireNonNull(byteOrder, "byteOrder is null");

            mByteOrder = byteOrder;

            return;
        } // end of setByteOrder(ByteOrder)

        /**
         * Sets the maximum queue size for accepted TCP
         * connections. When the message queue size reaches this
         * maximum, the TCP connection is automatically closed.
         * If zero, then there is no maximum queue size and the
         * queue is allowed to grow indefinitely.
         * @param size message queue maximum size.
         * @throws IllegalArgumentException
         * if {@code size} &lt; zero.
         */
        protected final void setMessageQueueSize(final int size)
        {
            if (size < 0)
            {
                throw (
                    new IllegalArgumentException(
                        "message queue size < 0"));
            }

            mMsgQueueSize = size;

            return;
        } // end of setMessageQueueSize(int)

        /**
         * Sets the heartbeat delay used for accepted TCP
         * connections. If no message is received after
         * {@code delay} milliseconds, then a heartbeat is sent.
         * If {@code delay} is zero, then heartbeating is turned
         * off.
         * @param delay millisecond heartbeat rate.
         * @throws IllegalArgumentException
         * if {@code delay} &lt; zero.
         */
        protected final void setHeartbeatDelay(final long delay)
        {
            if (delay < 0)
            {
                throw (
                    new IllegalArgumentException(
                        "heartbeat delay < 0"));
            }

            mHbDelay = delay;

            return;
        } // end of setHeartbeatDelay(long)

        /**
         * Sets the heartbeat reply delay used for accepted TCP
         * connections. The remote eBus application has this
         * many milliseconds to reply to a heartbeat. If no reply
         * is received after {@code delay} milliseconds, the TCP
         * connection is closed. This value is ignored if the
         * heartbeat delay is zero.
         * @param delay millisecond heartbeat reply delay.
         * @throws IllegalArgumentException
         * if {@code delay} &lt; zero.
         */
        protected final void setHeartbeatReplyDelay(final long delay)
        {
            if (delay < 0)
            {
                throw (
                    new IllegalArgumentException(
                        "heartbeat reply delay < 0"));
            }

            mHbReplyDelay = delay;

            return;
        } // end of setHeartbeatReplyDelay(long)

        /**
         * Sets the SSL context instance. Required if connection
         * type is {@link ConnectionType#SECURE_TCP}. If an SSL
         * context is provided for non-secure connection types,
         * then validation will fail.
         * @param context SSL context.
         * @throws NullPointerException
         * if {@code context} is {@code null}.
         */
        protected final void setSSLContext(final SSLContext context)
        {
            mSSLContext =
                Objects.requireNonNull(
                    context, "context is null");
            return;
        } // end of setSSLContext(SSLContext)

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Validates builder arguments are correctly set,
         * throwing the appropriate exception if validation
         * fails.
         * @throws NullPointerException
         * if service name or connection type is not set or if
         * SSL context not provided for a secure connection.
         * @throws IllegalArgumentException
         * if SSL context provided for a non-secure connection.
         */
        protected void validate()
        {
            Objects.requireNonNull(
                mName, "service name is null");
            Objects.requireNonNull(
                mConnectionType, "connection type is null");

            // Check if an SSL context was provided for a secure
            // TCP connection.
            if (mConnectionType == ConnectionType.SECURE_TCP)
            {
                Objects.requireNonNull(
                    mSSLContext,
                    "SSL context not provided for secure connection");
            }
            // Check if an SSL context was provided for a
            // non-secure connection.
            else if (mSSLContext != null)
            {
                throw (
                    new IllegalArgumentException(
                        "SSL context provided for non-secure connection"));
            }

            return;
        } // end of validate()
    } // end of class AbstractBuilder

    /**
     * Constructs an {@link EConfigure.Service} instance based on
     * the parameters set via this builder's API. The minimally
     * allowed configuration is the service name, connection type
     * and port. The remaining parameters are set to the
     * following defaults:
     * <ul>
     *   <li>
     *     connection type: {@link ConnectionType#TCP} - plain
     *     text TCP connection.
     *   </li>
     *   <li>
     *     address filter: {@code null} - no address filter
     *     applied.
     *   </li>
     *   <li>
     *     input buffer size: zero - use system default buffer
     *     size.
     *   </li>
     *   <li>
     *     output buffer size: zero - use system default buffer
     *     size.
     *   </li>
     *   <li>
     *     byte order: little-endian.
     *   </li>
     *   <li>
     *     maximum message queue size: zero - unlimited queue
     *     size.
     *   </li>
     *   <li>
     *     service selector thread:
     *     {@code AsyncChannel.defaultSelector}.
     *   </li>
     *   <li>
     *     accepted connection select thread:
     *     {@code AsyncChannel.defaultSelector}.
     *   </li>
     *   <li>
     *     heartbeat millisecond delay: zero - heartbeating off.
     *   </li>
     *   <li>
     *     heartbeat millisecond reply delay: zero - wait
     *     indefinitely for a reply.
     *   </li>
     * </ul>
     * <p>
     * If either the service name or port are not set, then
     * {@link #build()} will throw an exception.
     * </p>
     * <h1>Example building an {@code EServer}</h1>
     * <pre><code>final AddressFilter filter = ...;
final SSLContext secureContext = ...;
final EConfigure.ServerBuilder builder = EConfigure.serverBuilder();

EServer.openServer(builder.name("AppServer")
                          .port(6789)
                          .connectionType(EConfigure.ConnectionType.SECURE_TCP)
                          .sslContext(secureContext)
                          .addressFilter(filter)
                          .inputBufferSize(1_024)
                          .outputBufferSize(1_024)
                          .byteOrder(ByteOrder.BIG_ENDIAN)
                          .messageQueueSize(10_000)
                          .serviceSelector("svcSelector")
                          .connectionSelector("connSelector")
                          .heartbeatDelay(60_000L)
                          .heartbeatReplyDelay(30_000L)
                          .build());</code></pre>
     *
     * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
     *
     * @see ConnectionBuilder
     */
    public static final class ServerBuilder
        extends AbstractBuilder
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private int mPort;
        private AddressFilter mAddressFilter;
        private String mServiceSelector;
        private String mConnSelector;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new server configuration builder.
         */
        private ServerBuilder()
        {
            mPort = 0;
            mAddressFilter = null;
            mServiceSelector =
                (ENetConfigure.defaultSelector()).name();
            mConnSelector = mServiceSelector;
        } // end of ServerBuilder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the configured TCP port.
         * @return TCP port.
         */
        public int port()
        {
            return (mPort);
        } // end of port()

        /**
         * Returns the configured address filter. Returns
         * {@code null} if the address filter is not set.
         * @return address filter.
         */
        public AddressFilter addressFilter()
        {
            return (mAddressFilter);
        } // end of addressFilter()

        /**
         * Returns the network selector information used for the
         * eBus service.
         * @return network selector information.
         */
        public String serviceSelector()
        {
            return (mServiceSelector);
        } // end of serviceSelector()

        /**
         * Returns the network selector information used for
         * accepted connections.
         * @return network selector information.
         */
        public String connectionSelector()
        {
            return (mConnSelector);
        } // end of connectionSelector()

        //
        // end of Get Methods.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the service name.
         * @param name service name.
         * @return {@code this} service builder.
         * @throws NullPointerException
         * if {@code name} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code name} is empty.
         */
        public ServerBuilder name(final String name)
        {
            setName(name);

            return (this);
        } // end of name(String)

        /**
         * Sets the underlying channel type.
         * @param connType channel type.
         * @return {@code this} service builder.
         * @throws NullPointerException
         * if {@code connType} is {@code null}.
         */
        public ServerBuilder connectionType(final ConnectionType connType)
        {
            setConnectionType(connType);

            return (this);
        } // end of connectionType(ConnectionType)

        /**
         * Sets the service TCP port.
         * @param port service TCP port.
         * @return {@code this} service builder.
         * @throws IllegalArgumentException
         * if {@code port} is not a valid TCP address.
         */
        public ServerBuilder port(final int port)
        {
            if (port < MIN_PORT || port > MAX_PORT)
            {
                throw (
                    new IllegalArgumentException(
                        "invalid port (" + port + ")"));
            }

            mPort = port;

            return (this);
        } // end of port(int)

        /**
         * Sets the optional service address filter.
         * @param filter the optional address filter. May be
         * {@code null}.
         * @return {@code this} service builder.
         */
        public ServerBuilder addressFilter(final AddressFilter filter)
        {
            mAddressFilter = filter;
            return (this);
        } // end of addressFilter(AddressFilter)

        /**
         * Sets the input buffer size for all accepted TCP
         * connections. If {@code size} is zero, then the default
         * input buffer size is used.
         * @param size accepted connection input buffer size.
         * @return {@code this} service builder.
         * @throws IllegalArgumentException
         * if {@code size} &lt; zero.
         */
        public ServerBuilder inputBufferSize(final int size)
        {
            setInputBufferSize(size);

            return (this);
        } // end of inputBufferSize(int)

        /**
         * Sets the output buffer size for all accept TCP
         * connections. If {@code size} is zero, then the default
         * output buffer size is used.
         * @param size accepted connection output buffer size.
         * @return {@code this} service builder.
         * @throws IllegalArgumentException
         * if {@code size} &lt; zero.
         */
        public ServerBuilder outputBufferSize(final int size)
        {
            setOutputBufferSize(size);

            return (this);
        } // end of outputBufferSize(int)

        /**
         * Sets the byte order used by accepted TCP connections.
         * @param byteOrder accepted connections serialize and
         * de-serialize messages using this byte order.
         * @return {@code this} service builder.
         * @throws NullPointerException
         * if {@code byteOrder} is {@code nuill}.
         */
        public ServerBuilder byteOrder(final ByteOrder byteOrder)
        {
            setByteOrder(byteOrder);

            return (this);
        } // end of byteOrder(ByteOrder)

        /**
         * Sets the maximum queue size for accepted TCP
         * connections. When the message queue size reaches this
         * maximum, the TCP connection is automatically closed.
         * If zero, then the default maximum queue size is used.
         * @param size message queue maximum size.
         * @return {@code this} service builder.
         * @throws IllegalArgumentException
         * if {@code size} &lt; zero.
         */
        public ServerBuilder messageQueueSize(final int size)
        {
            setMessageQueueSize(size);

            return (this);
        } // end of messageQueueSize(int)

        /**
         * Sets the selector used for the service connection.
         * @param selector eBus selector name.
         * @return {@code this} service builder.
         * @throws NullPointerException
         * if {@code name} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code name} is an empty string or not a known
         * selector.
         */
        public ServerBuilder serviceSelector(final String selector)
        {
            Objects.requireNonNull(
                selector, "service selector is null");

            if (selector.isEmpty())
            {
                throw (
                    new IllegalArgumentException(
                        "service selector is empty"));
            }

            if (!ENetConfigure.isKnownSelector(selector))
            {
                throw (
                    new IllegalArgumentException(
                        "\"" +
                        selector +
                        "\" unknown service selector"));
            }

            mServiceSelector = selector;

            return (this);
        } // end of serviceSelector(String)

        /**
         * Sets the selector used for accepted TCP connections.
         * @param selector eBus selector name.
         * @return {@code this} service builder.
         * @throws NullPointerException
         * if {@code name} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code name} is an empty string or not a known
         * selector.
         */
        public ServerBuilder connectionSelector(final String selector)
        {
            Objects.requireNonNull(
                selector, "connection selector is null");

            if (selector.isEmpty())
            {
                throw (
                    new IllegalArgumentException(
                        "connection selector is empty"));
            }

            if (!ENetConfigure.isKnownSelector(selector))
            {
                throw (
                    new IllegalArgumentException(
                        "\"" +
                        selector +
                        "\" unknown connection selector"));
            }

            mConnSelector = selector;

            return (this);
        } // end of connectionSelector(String)

        /**
         * Sets the heartbeat delay used for accepted TCP
         * connections. If no message is received after
         * {@code delay} milliseconds, then a heartbeat is sent.
         * If {@code delay} is zero, then heartbeating is turned
         * off.
         * @param delay millisecond heartbeat rate.
         * @return {@code this} service builder.
         * @throws IllegalArgumentException
         * if {@code delay} &lt; zero.
         */
        public ServerBuilder heartbeatDelay(final long delay)
        {
            setHeartbeatDelay(delay);

            return (this);
        } // end of heartbeatDelay(long)

        /**
         * Sets the heartbeat reply delay used for accepted TCP
         * connections. The remote eBus application has this
         * many milliseconds to reply to a heartbeat. If no reply
         * is received after {@code delay} milliseconds, the TCP
         * connection is closed. This value is ignored if the
         * heartbeat delay is zero.
         * @param delay millisecond heartbeat reply delay.
         * @return {@code this} service builder.
         * @throws IllegalArgumentException
         * if {@code delay} &lt; zero.
         */
        public ServerBuilder heartbeatReplyDelay(final long delay)
        {
            setHeartbeatReplyDelay(delay);

            return (this);
        } // end of heartbeatReplyDelay(long)

        /**
         * Sets the SSL/TLS context used by a secure location.
         * This should be called only if the connection type
         * is set to {@link ConnectionType#SECURE_TCP}.
         * @param context secure TCP context.
         * @return {@code this} connection builder.
         * @throws NullPointerException
         * if {@code context} is {@code null}.
         */
        public ServerBuilder sslContext(final SSLContext context)
        {
            setSSLContext(context);

            return (this);
        } // end of sslContext(SSLContext)

        /**
         * Returns the eBus service configuration built from the
         * previously set parameters.
         * @return an eBus service configuration.
         * @throws NullPointerException
         * if any service name is not set.
         * @throws IllegalStateException
         * if service port is not set.
         */
        public EConfigure.Service build()
        {
            validate();

            return (new EConfigure.Service(this));
        } // end of build()

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Validates the builder parameters. Called for effect
         * only.
         * @throws NullPointerException
         * if a required parameter is not set.
         * @throws IllegalArgumentException
         * if a parameter is not set to a valid value with
         * respect to another parameter.
         */
        @Override
        protected void validate()
        {
            super.validate();

            if (mPort == 0)
            {
                throw (
                    new IllegalArgumentException(
                        "service port not set"));
            }

            return;
        } // end of validate()
    } // end of class ServerBuilder

    /**
     * Constructs an {@link EConfigure.RemoteConnection} instance
     * based on the parameters set via the builder's API. The
     * minimally allowed configuration is the connection name and
     * address. The remaining parameters are set to the following
     * defaults:
     * <ul>
     *   <li>
     *     connection type: {@link ConnectionType#TCP} - plain
     *     text TCP connection.
     *   </li>
     *   <li>
     *     bind port: {@code ERemoteApp.ANY_PORT}.
     *   </li>
     *   <li>
     *     input buffer size: zero - use system default buffer
     *     size.
     *   </li>
     *   <li>
     *     output buffer size: zero - use system default buffer
     *     size.
     *   </li>
     *   <li>
     *     byte order: little-endian.
     *   </li>
     *   <li>
     *     maximum message queue size: zero - unlimited queue
     *     size.
     *   </li>
     *   <li>
     *     connection selector thread:
     *     {@code AsyncChannel.defaultSelector}.
     *   </li>
     *   <li>
     *     reconnect flag and delay: reconnection is turned off and
     *     delay set to zero.
     *   </li>
     *   <li>
     *     heartbeat millisecond delay: zero - heartbeating off.
     *   </li>
     *   <li>
     *     heartbeat millisecond reply delay: zero - wait
     *     indefinitely for a reply.
     *   </li>
     * </ul>
     * <h1>Example building an {@code ERemoteApp}</h1>
     * <pre><code>final InetSocketAddress address = new InetSocketAddress(InetAddress.getLocalHost(), 12345);
final SSLContext secureContext = ...;
final EConfigure.ConnectionBuilder builder = EConfigure.connectionBuilder();

ERemoteApp.openConnection(builder.name("Conn0")
                                 .address(address)
                                 .bindPort(0)
                                 .connectionType(EConfigure.ConnectionType.SECURE_TCP)
                                 .sslContext(secureContext)
                                 .inputBufferSize(4_996)
                                 .outputBufferSize(8_192)
                                 .byteOrder(ByteOrder.BIG_ENDIAN)
                                 .messageQueueSize(0)
                                 .selector("connSelector")
                                 .reconnect(true)
                                 .reconnectDelay(500L)
                                 .heartbeatDelay(0L)
                                 .heartbeatReplyDelay(0L)
                                 .build());</code></pre>
     *
     * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
     *
     * @see ServerBuilder
     */
    public static final class ConnectionBuilder
        extends AbstractBuilder
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private InetSocketAddress mAddress;
        private int mBindPort;
        private String mSelector;
        private boolean mReconnectFlag;
        private long mReconnectTime;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new eBus connection configuration builder.
         */
        private ConnectionBuilder()
        {
            mAddress = null;
            mBindPort = ANY_PORT;
            mSelector = (ENetConfigure.defaultSelector()).name();
            mReconnectFlag = false;
            mReconnectTime = 0L;
        } // end of ConnectionBuilder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the configured Internet address and port.
         * Returns {@code null} if the address is not set.
         * @return Internet address and port.
         */
        public InetSocketAddress address()
        {
            return (mAddress);
        } // end of address()

        /**
         * Returns the configured bind port.
         * @return bind port.
         */
        public int bindPort()
        {
            return (mBindPort);
        } // end of bindPort()

        /**
         * Returns the configured network selector information.
         * @return network selector information.
         */
        public String selector()
        {
            return (mSelector);
        } // end of selector()

        /**
         * Returns {@code true} if the connection is configured
         * to re-establish a lost connect.
         * @return reconnection flag.
         */
        public boolean reconnect()
        {
            return (mReconnectFlag);
        } // end of reconnect()

        /**
         * Returns the configured millisecond reconnect delay.
         * @return millisecond reconnect delay.
         */
        public long reconnectDelay()
        {
            return (mReconnectTime);
        } // end of reconnectDelay()

        //
        // end of Get Methods.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the connection name.
         * @param name connection name.
         * @return {@code this} connection builder.
         * @throws NullPointerException
         * if {@code name} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code name} is empty.
         */
        public ConnectionBuilder name(final String name)
        {
            setName(name);

            return (this);
        } // end of name(String)

        /**
         * Sets the underlying channel type.
         * @param connType channel type.
         * @return {@code this} service builder.
         * @throws NullPointerException
         * if {@code connType} is {@code null}.
         */
        public ConnectionBuilder connectionType(final ConnectionType connType)
        {
            setConnectionType(connType);

            return (this);
        } // end of connectionType(ConnectionType)

        /**
         * Set the connection address.
         * @param address connection address.
         * @return {@code this} connection builder.
         * @throws NullPointerException
         * if {@code address} is {@code null}.
         */
        public ConnectionBuilder address(final InetSocketAddress address)
        {
            Objects.requireNonNull(address, "address is null");

            mAddress = address;

            return (this);
        } // end of address(InetSocketAddress)

        /**
         * Bind the connection local port to this value. If
         * {@code port} is set to {@link ENetConfigure#ANY_PORT},
         * then the connection local port is set to any available
         * value.
         * @param port connection local bind port.
         * @return {@code this} connection builder.
         * @throws IllegalArgumentException
         * if {@code port} is either &lt;
         * {@link ENetConfigure#ANY_PORT} or &gt;
         * {@link ENetConfigure#MAX_PORT}.
         */
        public ConnectionBuilder bindPort(final int port)
        {
            if (port < ANY_PORT || port > MAX_PORT)
            {
                throw (
                    new IllegalArgumentException(
                        "invalid bind port (" + port + ")"));
            }

            mBindPort = port;

            return (this);
        } // end of bindPort(int)

        /**
         * Sets the connection input buffer size. If {@code size}
         * is zero, then the default input buffer size is used.
         * @param size accepted connection input buffer size.
         * @return {@code this} service builder.
         * @throws IllegalArgumentException
         * if {@code size} &lt; zero.
         */
        public ConnectionBuilder inputBufferSize(final int size)
        {
            setInputBufferSize(size);

            return (this);
        } // end of inputBufferSize(int)

        /**
         * Sets the connection output buffer size. If
         * {@code size} is zero, then the default output buffer
         * size is used.
         * @param size connection output buffer size.
         * @return {@code this} connection builder.
         * @throws IllegalArgumentException
         * if {@code size} &lt; zero.
         */
        public ConnectionBuilder outputBufferSize(final int size)
        {
            setOutputBufferSize(size);

            return (this);
        } // end of outputBufferSize(int)

        /**
         * Sets the connection byte order.
         * @param byteOrder connection serializes and
         * de-serializes messages using this byte order.
         * @return {@code this} connection builder.
         * @throws NullPointerException
         * if {@code byteOrder} is {@code nuill}.
         */
        public ConnectionBuilder byteOrder(final ByteOrder byteOrder)
        {
            setByteOrder(byteOrder);

            return (this);
        } // end of byteOrder(ByteOrder)

        /**
         * Sets the connection maximum queue size. When the
         * message queue size reaches this maximum, the
         * connection is automatically closed. If zero, then the
         * message queue is allowed to grow without limit.
         * @param size message queue maximum size.
         * @return {@code this} connection builder.
         * @throws IllegalArgumentException
         * if {@code size} &lt; zero.
         */
        public ConnectionBuilder messageQueueSize(final int size)
        {
            setMessageQueueSize(size);

            return (this);
        } // end of messageQueueSize(int)

        /**
         * Sets the selector used for the connection.
         * @param selector eBus selector name.
         * @return {@code this} connection builder.
         * @throws NullPointerException
         * if {@code name} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code name} is an empty string or not a known
         * selector.
         */
        public ConnectionBuilder selector(final String selector)
        {
            Objects.requireNonNull(selector, "selector is null");

            if (selector.isEmpty())
            {
                throw (new IllegalArgumentException("selector is empty"));
            }

            if (!ENetConfigure.isKnownSelector(selector))
            {
                throw (
                    new IllegalArgumentException(
                        "\"" + selector + "\" unknown selector"));
            }

            mSelector = selector;

            return (this);
        } // end of selector(String)

        /**
         * Sets the reconnect flag to the given value.
         * {@code true} means the connection will be
         * re-established if lost; {@code false} means a lost
         * connection is left down.
         * @param flag establish lost connection flag.
         * @return {@code this} connection builder.
         *
         * @see #reconnectDelay(long)
         */
        public ConnectionBuilder reconnect(final boolean flag)
        {
            mReconnectFlag = flag;

            return (this);
        } // end of reconnect(boolean)

        /**
         * Sets the reconnect delay to the given value. This value
         * is used on if the reconnect flag is {@code true}.
         * @param time reconnect a lost connect after this
         * millisecond delay.
         * @return {@code this} connection builder.
         *
         * @see #reconnect(boolean)
         */
        public ConnectionBuilder reconnectDelay(final long time)
        {
            if (time < 0L)
            {
                throw (
                    new IllegalArgumentException(
                        "reconnect time < 0"));
            }

            mReconnectTime = time;

            return (this);
        } // end of reconnectDelay(long)

        /**
         * Sets the heartbeat delay used for the connection.
         * If no message is received after {@code delay}
         * milliseconds, then a heartbeat is sent. If
         * {@code delay} is zero, then heartbeating is turned
         * off.
         * @param delay millisecond heartbeat rate.
         * @return {@code this} connection builder.
         * @throws IllegalArgumentException
         * if {@code delay} &lt; zero.
         */
        public ConnectionBuilder heartbeatDelay(final long delay)
        {
            setHeartbeatDelay(delay);

            return (this);
        } // end of heartbeatDelay(long)

        /**
         * Sets the heartbeat reply delay used for the
         * connection. The remote eBus application has this many
         * milliseconds to reply to a heartbeat. If no reply is
         * received after {@code delay} milliseconds, the
         * connection is closed. This value is ignored if the
         * heartbeat delay is zero (heartbeating is off).
         * @param delay millisecond heartbeat reply delay.
         * @return {@code this} connection builder.
         * @throws IllegalArgumentException
         * if {@code delay} &lt; zero.
         */
        public ConnectionBuilder heartbeatReplyDelay(final long delay)
        {
            setHeartbeatReplyDelay(delay);

            return (this);
        } // end of heartbeatReplyDelay(long)

        /**
         * Sets the SSL/TLS context used by a secure location.
         * This should be called only if the connection type
         * is set to {@link ConnectionType#SECURE_TCP}.
         * @param context secure TCP context.
         * @return {@code this} connection builder.
         * @throws NullPointerException
         * if {@code context} is {@code null}.
         */
        public ConnectionBuilder sslContext(final SSLContext context)
        {
            setSSLContext(context);

            return (this);
        } // end of sslContext(SSLContext)

        /**
         * Returns the eBus connection configuration built from
         * the previously set parameters.
         * @return an eBus connection configuration.
         * @throws NullPointerException
         * if either connection name or address is not set.
         * @throws IllegalStateException
         * if reconnect flag is set to {@code true} but the
         * reconnect delay is not set.
         */
        public EConfigure.RemoteConnection build()
        {
            validate();

            return (new EConfigure.RemoteConnection(this));
        } // end of build()

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Validates the builder parameters. Called for effect
         * only.
         * @throws NullPointerException
         * if a required parameter is not set.
         * @throws IllegalArgumentException
         * if a parameter is not set to a valid value with
         * respect to another parameter.
         */
        @Override
        protected void validate()
        {
            super.validate();

            Objects.requireNonNull(
                mName, "connection name not set");
            Objects.requireNonNull(
                mAddress, "address not set");

            // If the reconnect flag is set, the the reconnect delay
            // must also be set.
            if (mReconnectFlag && mReconnectTime <= 0L)
            {
                throw (
                    new IllegalArgumentException(
                        "reconnect time == 0"));
            }

            return;
        } // end of validate()
    } // end of class ConnectionBuilder

    /**
     * Constructs an {@link EConfigure.Dispatcher} configuration
     * instance based on the parameters set via the builder's
     * API. The minimally allowed configuration is the Dispatcher
     * name and Dispatcher type if that type is either
     * {@link DispatcherType#SWING} or
     * {@link DispatcherType#JAVAFX}. For a
     * {@link DispatcherType#EBUS} Dispatcher type, then the
     * thread type must be provided. If this Dispatcher is
     * <em>not</em> marked as the default, then the classes
     * assigned to this Dispatcher must be provided.
     * <p>
     * The remaining properties are set to the following
     * defaults:
     * </p>
     * <ul>
     *   <li>
     *     spin limit: {@link EConfigure#DEFAULT_SPIN_LIMIT}.
     *   </li>
     *   <li>
     *     park time: {@link EConfigure#DEFAULT_PARK_TIME}.
     *   </li>
     *   <li>
     *     priority: {@link Thread#NORM_PRIORITY}.
     *   </li>
     *   <li>
     *     quantum: {@link EConfigure#DEFAULT_QUANTUM}.
     *   </li>
     *   <li>
     *     thread count: {@link EConfigure#DEFAULT_NUMBER_THREADS}.
     *   </li>
     *   <li>
     *     default Dispatcher: {@code false}.
     *   </li>
     * </ul>
     *
     * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
     */
    public static final class DispatcherBuilder
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private String mName;
        private DispatcherType mType;
        private ThreadType mRunQueueType;
        private long mSpinLimit;
        private long mParkTime;
        private int mPriority;
        private long mQuantum;
        private int mNumThreads;
        private boolean mIsDefault;
        private Class<?>[] mClasses;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new dispatcher builder instance used to
         * construct an {@link EConfigure.Dispatcher} instance.
         */
        private DispatcherBuilder()
        {
            mName = null;
            mType = null;
            mRunQueueType = null;
            mSpinLimit = 0L;
            mParkTime = 0L;
            mPriority = Thread.NORM_PRIORITY;
            mQuantum = EConfigure.DEFAULT_QUANTUM;
            mNumThreads = EConfigure.DEFAULT_NUMBER_THREADS;
            mIsDefault = false;
            mClasses = new Class<?>[0];
        } // end of DispatcherBuilder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the Dispatcher name. Returns {@code null} if
         * the name is not set.
         * @return Dispatcher name.
         */
        public String name()
        {
            return (mName);
        } // end of name()

        /**
         * Returns the Dispatcher type. Returns {@code null} if
         * the Dispatcher type is not set.
         * @return Dispatcher type.
         */
        public DispatcherType dispatcherType()
        {
            return (mType);
        } // end of dispatcherType()

        /**
         * Returns the Dispatcher thread type. Returns
         * {@code null} if not yet set.
         * @return Dispatcher thread type.
         */
        public ThreadType threadType()
        {
            return (mRunQueueType);
        } // end of threadType()

        /**
         * Returns the spin limit. This setting is valid only if
         * the Dispatch type is {@link DispatcherType#EBUS} and
         * the thread type is either {@link ThreadType#SPINPARK}
         * or {@link ThreadType#SPINYIELD}.
         * @return spin limit.
         */
        public long spinLimit()
        {
            return (mSpinLimit);
        } // end of spinLimit()

        /**
         * Returns the nanosecond park time. This setting is
         * valid only if the Dispatch type is
         * {@link DispatcherType#EBUS} and the thread type is
         * {@link ThreadType#SPINPARK}.
         * @return nanosecond park time.
         */
        public long parkTime()
        {
            return (mParkTime);
        } // end of parkTime()

        /**
         * Returns the Dispatcher thread priority.
         * @return thread priority.
         */
        public int priority()
        {
            return (mPriority);
        } // end of priority()

        /**
         * Returns the eBus client run quantum.
         * @return run quantum.
         */
        public long quantum()
        {
            return (mQuantum);
        } // end of quantum()

        /**
         * Returns the number of Dispatcher threads.
         * @return Dispatcher thread count.
         */
        public int numberThreads()
        {
            return (mNumThreads);
        } // end of numberThreads()

        /**
         * Returns {@code true} if the Dispatcher is configured
         * to be the default Dispatcher.
         * @return {@code true} if default Dispatcher.
         */
        public boolean isDefault()
        {
            return (mIsDefault);
        } // end of isDefault()

        /**
         * Returns the array of classes assigned to this
         * Dispatcher. Does not return {@code null}. Returned
         * array is a copy of this builder's class array.
         * @return assigned class array.
         */
        public Class<?>[] classes()
        {
            return (Arrays.copyOf(mClasses, mClasses.length));
        } // end of classes()

        //
        // end of Get Methods.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the dispatcher name.
         * @param name dispatcher name.
         * @return {@code this} dispatcher builder.
         * @throws NullPointerException
         * if {@code name} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code name} is empty.
         */
        public DispatcherBuilder name(final String name)
        {
            Objects.requireNonNull(name, "name is null");

            if (name.isEmpty())
            {
                throw (
                    new IllegalArgumentException(
                        "name is empty"));
            }

            mName = name;

            return (this);
        } // end of name(String)

        /**
         * Sets the dispatcher type.
         * @param type dispatcher type.
         * @return {@code this} dispatcher builder.
         * @throws NullPointerException
         * if {@code type} is {@code null}.
         */
        public DispatcherBuilder dispatcherType(final DispatcherType type)
        {
            mType =
                Objects.requireNonNull(
                    type, "dispatcher type is null");

            return (this);
        } // end of dispatcherType(DispatcherType)

        /**
         * Sets the run queue thread type. This setting is
         * ignored if the Dispatcher type is
         * {@link DispatcherType#SWING} or
         * {@link DispatcherType#JAVAFX}.
         * @param type thread type.
         * @return {@code this} dispatcher builder.
         * @throws NullPointerException
         * if {@code type} is {@code null}.
         */
        public DispatcherBuilder threadType(final ThreadType type)
        {
            mRunQueueType =
                Objects.requireNonNull(
                    type, "thread type is null");

            return (this);
        } // end of threadType(ThreadType)

        /**
         * Sets the {@link ThreadType#SPINPARK} or
         * {@link ThreadType#SPINYIELD} spin limit. This setting
         * is ignored for any other run queue thread type.
         * @param limit spin limit.
         * @return {@code this} dispatcher builder.
         * @throws IllegalArgumentException
         * if {@code limit} &lt; zero.
         */
        public DispatcherBuilder spinLimit(final long limit)
        {
            if (limit < 0L)
            {
                throw (
                    new IllegalArgumentException(
                        "limit < zero"));
            }

            mSpinLimit = limit;

            return (this);
        } // end of spinLimit(long)

        /**
         * Sets the {@link ThreadType#SPINPARK} nanosecond park
         * time. This setting is ignored for any other run queue
         * thread type.
         * @param time nanosecond park time.
         * @return {@code this} dispatcher builder.
         * @throws IllegalArgumentException
         * if {@code time} &lt; zero.
         */
        public DispatcherBuilder parkTime(final long time)
        {
            if (time < 0L)
            {
                throw (
                    new IllegalArgumentException("time < zero"));
            }

            mParkTime = time;

            return (this);
        } // end of parkTime(long)

        /**
         * Sets the run queue thread priority. Must be &ge;
         * {@link Thread#MIN_PRIORITY} and &le;
         * {@link Thread#MAX_PRIORITY}. This setting is ignored
         * if the Dispatcher type is {@link DispatcherType#SWING}
         * or {@link DispatcherType#JAVAFX}.
         * @param priority assigned thread priority for run queue
         * threads.
         * @return {@code this} dispatcher builder.
         * @throws IllegalArgumentException
         * if {@code priority} &lt; zero or &gt;
         * {@code Thread.MAX_PRIORITY}.
         */
        public DispatcherBuilder priority(final int priority)
        {
            if (priority < 0 || priority > Thread.MAX_PRIORITY)
            {
                throw (
                    new IllegalArgumentException(
                        "priority out of bounds"));
            }

            mPriority = priority;

            return (this);
        } // end of priority(int)

        /**
         * Sets the run quantum assigned to each eBus client.
         * @param quantum run quantum.
         * @return {@code this} dispatcher builder.
         * @throws IllegalArgumentException
         * if {@code quantum} &lt; zero.
         */
        public DispatcherBuilder quantum(final long quantum)
        {
            if (quantum < 0L)
            {
                throw (
                    new IllegalArgumentException(
                        "quantum < zero"));
            }

            mQuantum = quantum;

            return (this);
        } // end of quantum(long)

        /**
         * Sets the number of threads used by the run queue. This
         * setting is ignored if the Dispatcher type is
         * {@link DispatcherType#SWING} or
         * {@link DispatcherType#JAVAFX}.
         * @param numThreads number of run queue threads.
         * @return {@code this} dispatcher builder.
         * @throws IllegalArgumentException
         * if {@code numThreads} &lt; zero.
         */
        public DispatcherBuilder numberThreads(final int numThreads)
        {
            if (numThreads < 0)
            {
                throw (
                    new IllegalArgumentException(
                        "numThreads < zero"));
            }

            mNumThreads = numThreads;

            return (this);
        } // end of numberThreads(int)

        /**
         * If {@code flag} is {@code true}, then marks the
         * dispatcher as the default dispatcher.
         * @param flag {@code true} marks the dispatcher as the
         * default.
         * @return {@code this} dispatcher builder.
         */
        public DispatcherBuilder isDefault(final boolean flag)
        {
            mIsDefault = flag;

            return (this);
        } // end of isDefault(boolean)

        /**
         * Lists the classes assigned to the dispatcher. If the
         * default dispatcher, this list is ignored and
         * {@code classes} may be {@code null} or empty. If the
         * dispatcher is <em>not</em> default, then
         * {@code classes} must be defined as a non-{@code null},
         * non-empty array.
         * <p>
         * This parameter is checked when the dispatcher is
         * {@link #build() built} since the validation depends on
         * whether the dispatcher is default or not.
         * </p>
         * @param classes eBus client classes assigned to the
         * Dispatcher.
         * @return {@code this} dispatcher builder.
         */
        public DispatcherBuilder classes(final Class<?>[] classes)
        {
            if (classes == null)
            {
                mClasses = new Class<?>[0];
            }
            else
            {
                mClasses =
                    Arrays.copyOf(classes, classes.length);
            }

            return (this);
        } // end of classes(Class<?>)

        /**
         * Returns the eBus Dispatcher configuration built from
         * the previously set parameters.
         * @return an eBus Dispatcher configuration.
         * @throws NullPointerException
         * if either the Dispatcher name, Dispatcher type, or run
         * queue thread type are not set.
         * @throws IllegalArgumentException
         * if the Dispatcher is not marked as default and no
         * classes were provided.
         */
        public EConfigure.Dispatcher build()
        {
            validate();

            return (new EConfigure.Dispatcher(this));
        } // end of build()

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Validates the builder parameters. Called for effect
         * only.
         * @throws NullPointerException
         * if a required parameter is not set.
         * @throws IllegalArgumentException
         * if a parameter is not set to a valid value with
         * respect to another parameter.
         */
        private void validate()
        {
            Objects.requireNonNull(mName, "name is null");
            Objects.requireNonNull(mType,
                                   "dispatcher type not set");
            Objects.requireNonNull(mRunQueueType,
                                   "thread type not set");

            // Is this a spin+park Dispatcher?
            if (mRunQueueType == ThreadType.SPINPARK)
            {
                // Yes. Are spin limit and park time set?
                if (mSpinLimit <= 0L)
                {
                    throw (
                        new IllegalArgumentException(
                            "spin limit not set for spin+park thread type"));
                }

                if (mParkTime <= 0L)
                {
                    throw (
                        new IllegalArgumentException(
                            "park limit not set for spin+park thread type"));
                }
            }
            // Is this a spin+yield Dispatcher?
            else if (mRunQueueType == ThreadType.SPINYIELD)
            {
                // Yes. Is spin limit set?
                if (mSpinLimit <= 0L)
                {
                    throw (
                        new IllegalArgumentException(
                            "spin limit not set for spin+yield thread type"));
                }
            }

            // Is this a default dispatcher?
            // Are classes provided?
            if (!mIsDefault &&
                (mClasses == null || mClasses.length == 0))
            {
                // Non-default dispatcher but no classes given.
                throw (
                    new IllegalArgumentException(
                        "classes not set for non-default dispatcher"));
            }

            return;
        } // end of validate()
    } // end of class DispatcherBuilder
} // end of class EConfigure
