package net.sf.doolin.oxml.annotation;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import net.sf.doolin.oxml.OXMLContext;
import net.sf.doolin.oxml.OXMLReaderOutput;
import net.sf.doolin.oxml.action.PropertyOXMLAction;

/**
 * The <code>Collection</code> annotation defines that a property is a potential
 * collection of fixed values and those values must be collected while the OXML
 * script is executed. Those values will be stored finally in the
 * {@link OXMLReaderOutput#getCollections()} map.
 * 
 * The <code>Collection</code> annotation must be associated with one of the
 * setter or getter of the property to collect values from.
 * 
 * @see PropertyOXMLAction
 * @see OXMLContext#collect(String, Object)
 */
@Documented
@Retention(RetentionPolicy.RUNTIME)
@Target( { ElementType.METHOD, ElementType.TYPE })
public @interface Collection {

	/**
	 * If <code>true</code>, the bean name, which contains the property, will
	 * added in the collection name.
	 * 
	 * This property is only used for annotated properties.
	 * 
	 * @return <code>true</code> if the bean name must be included in the
	 *         collection name
	 */
	boolean beanName() default false;

	/**
	 * List of properties to collect on an instance. If empty (default), a
	 * {@link Object#toString() toString()} call is applied on the instance.
	 * 
	 * This property is only used for annotated classes.
	 * 
	 * @return List of property names
	 */
	String[] collectedProperties() default {};

	/**
	 * Name of the collection. Will default to the property name if not defined.
	 * 
	 * @return the collection name
	 */
	String name() default "";

}
