/*
 * Created on Sep 17, 2007
 */
package net.sf.doolin.oxml.action;

import net.sf.doolin.oxml.OXML;
import net.sf.doolin.oxml.OXMLContext;
import net.sf.doolin.oxml.OXMLFactory;
import net.sf.doolin.oxml.adapter.DefaultOXMLAdapter;
import net.sf.doolin.oxml.adapter.OXMLAdapter;
import net.sf.doolin.oxml.annotation.Collection;
import net.sf.doolin.util.Utils;
import net.sf.doolin.util.xml.DOMUtils;
import net.sf.jstring.LocalizableException;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import java.io.IOException;
import java.lang.reflect.Method;

/**
 * <code>property</code> action.
 * 
 * @author Damien Coraboeuf
 */
public class PropertyOXMLAction extends AbstractOXMLAction {

	private String node;

	private String name;

	private String value;

	private String setter;

	private OXMLAdapter adapter = new DefaultOXMLAdapter();

	public void parse(Element e) throws IOException {
		this.name = DOMUtils.getAttribute(e, "name", true, null);
		this.node = DOMUtils.getAttribute(e, "node", false, null);
		this.value = DOMUtils.getAttribute(e, "value", false, null);
		this.setter = DOMUtils.getAttribute(e, "setter", false, null);
		//
		if (this.value == null && StringUtils.isBlank(this.node)) {
			this.node = StringUtils.replace(this.name, ".", "/");
		}
		// Adapter as a node
		Element eAdapter = DOMUtils.getElement(e, OXML.NS, "adapter");
		if (eAdapter != null) {
			this.adapter = OXMLFactory.getInstance().create(eAdapter);
		}
		// Adapter as an attribute
		else {
			Class<? extends OXMLAdapter> adapterClass = DOMUtils
					.getClassAttribute(e, "adapter", null);
			if (adapterClass != null) {
				this.adapter = Utils.newInstance(adapterClass);
			}
		}
	}

	public void process(OXMLContext context) {
		// Direct value
		if (this.value != null) {
			// Gets the current instance
			Object instance = context.instancePeek();
			// Sets the property
			try {
				setProperty(instance, this.value);
			} catch (Exception ex) {
				throw new RuntimeException("Cannot set property " + this.name,
						ex);
			}
		}
		// Adapter
		else {
			// Get the current node
			Node currentNode = context.nodePeek();
			// Gets the current instance
			Object instance = context.instancePeek();
			// Adapter
			Object propertyValue = this.adapter.adapt(currentNode, this.node,
					context);
			if (propertyValue != null) {
				try {
					// Property type
					Class<?> propertyType = PropertyUtils.getPropertyType(
							instance, this.name);
					// Enumeration ?
					if (propertyType.isEnum()) {
						// Enumeration handling
						propertyValue = toEnum(instance, propertyValue);
					}
					// Look for annotations
					else if (context.getConfig().isCollectEnabled()) {
						// Collection annotation
						Collection collection = Utils.getPropertyAnnotation(
								Collection.class, instance, this.name);
						if (collection != null) {
							String collectionName = collection.name();
							if (StringUtils.isBlank(collectionName)) {
								collectionName = this.name;
							}
							// Includes the bean name ?
							if (collection.beanName()) {
								// Bean name
								String beanClassName = instance.getClass()
										.getName();
								String beanName = StringUtils
										.substringAfterLast(beanClassName, ".");
								// New collection name
								collectionName += "@" + beanName;
							}
							// Collects the value
							context.collect(collectionName, propertyValue);
						}
					}
				} catch (RuntimeException ex) {
					throw ex;
				} catch (Exception ex) {
					throw new LocalizableException(
							"PropertyOXMLAction.PropertyInspectError", ex,
							this.name);
				}
			}
			// Sets the property
			try {
				if (propertyValue != null) {
					setProperty(instance, propertyValue);
				}
			} catch (Exception ex) {
				throw new RuntimeException("Cannot set property " + this.name,
						ex);
			}
			// Collection of the value
		}
	}

	/**
	 * Sets the property on a instance, using the parameters of this action. If
	 * no <code>setter</code> is defined, the property is directly set on the
	 * instance using the property <code>name</code>. If a <code>setter</code>
	 * has been defined, a sub-instance is first accessed using the
	 * <code>name</code> property and the <code>setter</code> is then called on
	 * this sub-instance.
	 * 
	 * @param instance
	 *            Object to set the property on
	 * @param propertyValue
	 *            Property value to set
	 * @throws Exception
	 *             In case the property cannot be set
	 * @see Utils#getProperty(Object, String)
	 */
	protected void setProperty(Object instance, Object propertyValue)
			throws Exception {
		if (StringUtils.isBlank(this.setter)) {
			BeanUtils.setProperty(instance, this.name, propertyValue);
		} else {
			Object propertyHolder = Utils.getProperty(instance, this.name);
			Utils.callMethod(propertyHolder, this.setter, propertyValue);
		}
	}

	/**
	 * Converts, if possible, the property value to a corresponding enumeration
	 * 
	 * @param instance
	 *            Current instance that will host the property value
	 * @param propertyValue
	 *            Raw property value
	 * @return Initial property or property converted to an enumeration
	 */
	protected Object toEnum(Object instance, Object propertyValue) {
		if (propertyValue != null) {
			try {
				// Property type
				Class<?> propertyType = PropertyUtils.getPropertyType(instance,
						this.name);
				// Enumeration ?
				if (propertyType.isEnum()) {
					// Enumeration class
					@SuppressWarnings("unchecked")
					Class<Enum<?>> enumClass = (Class<Enum<?>>) propertyType;
					// Property value as a string
					String propertyString = propertyValue.toString();
					// Converts to constant format
					String propertyEnumString = StringUtils
							.upperCase(propertyString);
					// Gets the enum value
					Method valueOfMethod = enumClass.getMethod("valueOf",
							String.class);
					Enum<?> enumValue = (Enum<?>) valueOfMethod.invoke(null,
							propertyEnumString);
					propertyValue = enumValue;
				}
			} catch (RuntimeException ex) {
				throw ex;
			} catch (Exception ex) {
				throw new LocalizableException(
						"PropertyOXMLAction.EnumConversionError", ex,
						propertyValue);
			}
		}
		return propertyValue;
	}

}
