package net.sf.doolin.oxml;

import java.io.IOException;
import java.util.Map;

import net.sf.doolin.oxml.action.OXMLAction;

import org.w3c.dom.Document;

/**
 * OXML reader.
 * 
 * @param <T>
 *            Expected type of the root object
 * @author Damien Coraboeuf
 */
public class OXMLReader<T> {

	private final OXMLConfig config;
	private final Map<String, Object> contextMap;

	/**
	 * Initialization of the reader.
	 * 
	 * @param cfg
	 *            Configuration to use
	 */
	public OXMLReader(OXMLConfig cfg) {
		this(cfg, null);
	}

	/**
	 * Initialization of the reader.
	 * 
	 * @param cfg
	 *            Configuration to use
	 * @param contextMap
	 *            Map of objects for the context
	 */
	public OXMLReader(OXMLConfig cfg, Map<String, Object> contextMap) {
		this.config = cfg;
		this.contextMap = contextMap;
	}

	/**
	 * Reads an XML source and returns the root object.
	 * 
	 * @param source
	 *            XML source
	 * @return Root object
	 * @throws IOException
	 *             If the source cannot be read.
	 */
	public T read(OXMLSource source) throws IOException {
		return read(source, new OXMLReaderOutput(), null);
	}

	/**
	 * Reads an XML source and returns the root object.
	 * 
	 * @param source
	 *            XML source
	 * @param output
	 *            Output of meta information after the reading is complete
	 * @param contextConfig
	 *            Context configuration (can be <code>null</code> if default
	 *            configuration is to be used).
	 * @return Root object
	 * @throws IOException
	 *             If the source cannot be read.
	 */
	public T read(OXMLSource source, OXMLReaderOutput output,
			OXMLContextConfig contextConfig) throws IOException {
		try {
			// Gets the DOM source
			Document dom = source.getSource();
			// Get the root action of the configuration
			OXMLAction action = this.config.getRootAction();
			// Creates the OXMLContext
			OXMLContext context = new OXMLContext(dom);
			// Context configuration
			if (contextConfig != null) {
				context.setConfig(contextConfig);
			}
			// Context
			if (this.contextMap != null) {
				context.contextPutAll(this.contextMap);
			}
			// Executes the action
			action.process(context);
			// OK
			@SuppressWarnings("unchecked")
			T root = (T) context.getTargetRoot();
			// Collections of data
			context.fill(output);
			// OK
			return root;
		} finally {
			// Closes the source
			source.close();
		}
	}

}
