package net.sf.doolin.oxml;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

import net.sf.doolin.oxml.action.SequenceOXMLAction;
import net.sf.doolin.util.Utils;
import net.sf.doolin.util.xml.DOMUtils;
import net.sf.doolin.util.xml.XPathUtils;

import org.apache.commons.lang.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Execution context
 * 
 * @author Damien Coraboeuf
 */
public class OXMLContext {

	private final Document dom;

	private Object root;

	private final XPathUtils xpath;

	private final Stack<Node> nodeStack;

	private final Map<String, Object> instanceMap;

	private final Stack<Object> instanceStack;

	private OXMLContextConfig config = new OXMLContextConfig();

	private Map<String, Set<Object>> collectionMap;

	private Map<String, SequenceOXMLAction> sequenceMap;

	private String packageName;

	/**
	 * Constructor.
	 * 
	 * @param document
	 *            Source document
	 */
	public OXMLContext(Document document) {
		this.dom = document;
		this.xpath = new XPathUtils();
		this.nodeStack = new Stack<Node>();
		this.nodeStack.push(this.dom);
		this.instanceMap = new HashMap<String, Object>();
		this.instanceStack = new Stack<Object>();
		this.sequenceMap = new HashMap<String, SequenceOXMLAction>(0);
		this.collectionMap = new HashMap<String, Set<Object>>();
	}

	/**
	 * Checks if the <code>node</code> has been used.
	 * 
	 * @param node
	 *            Node to check
	 * @param output
	 *            Output to fill
	 */
	protected void checkUsed(Node node, OXMLReaderOutput output) {
		if (!nodeUsed(node)) {
			// XPath to the node
			String x = DOMUtils.getXPath(node);
			// Creates a warning
			String message = x + " has not been used.";
			output.addWarning(message);
		}
		// Children nodes
		if (node.getNodeType() == Node.ELEMENT_NODE) {
			Element e = (Element) node;
			NodeList childNodes = e.getChildNodes();
			int length = childNodes.getLength();
			for (int i = 0; i < length; i++) {
				Node childNode = childNodes.item(i);
				switch (childNode.getNodeType()) {
				case Node.ELEMENT_NODE:
				case Node.ATTRIBUTE_NODE:
					checkUsed(childNode, output);
					break;
				}
			}
		}
	}

	/**
	 * Collects a value.
	 * 
	 * @param collectionName
	 *            Name of the collection
	 * @param collectionItem
	 *            Item to add into the collection
	 */
	public void collect(String collectionName, Object collectionItem) {
		// Gets the collection
		Set<Object> collection = this.collectionMap.get(collectionName);
		// Creates the collection
		if (collection == null) {
			collection = new HashSet<Object>();
			this.collectionMap.put(collectionName, collection);
		}
		// Adds the item
		collection.add(collectionItem);
	}

	/**
	 * Puts all the instances in the context instance map.
	 * 
	 * @param context
	 *            Index of instance to load in the context
	 */
	public void contextPutAll(Map<String, Object> context) {
		this.instanceMap.putAll(context);
	}

	/**
	 * Fills the output.
	 * 
	 * @param output
	 *            Output to fill
	 */
	public void fill(OXMLReaderOutput output) {
		// Fills collections
		fillCollections(output.getCollections());
		// Fills warnings
		fillWarnings(output);
	}

	/**
	 * Merges the collections.
	 * 
	 * @param collections
	 *            Collections to fill
	 */
	protected void fillCollections(Map<String, Set<Object>> collections) {
		for (Map.Entry<String, Set<Object>> entry : this.collectionMap
				.entrySet()) {
			String collectionName = entry.getKey();
			Set<Object> sourceCollection = entry.getValue();
			Set<Object> targetCollection = collections.get(collectionName);
			if (targetCollection == null) {
				targetCollection = new HashSet<Object>(sourceCollection);
				collections.put(collectionName, targetCollection);
			} else {
				targetCollection.addAll(sourceCollection);
			}
		}
	}

	/**
	 * Fills the warnings.
	 * 
	 * @param output
	 *            Output to fill
	 */
	protected void fillWarnings(OXMLReaderOutput output) {
		// Checks used marks
		Element eRoot = this.dom.getDocumentElement();
		checkUsed(eRoot, output);
	}

	/**
	 * Gets the config.
	 * 
	 * @return the config
	 */
	public OXMLContextConfig getConfig() {
		return this.config;
	}

	/**
	 * Returns a list of nodes from the current node using an XPath expression.
	 * 
	 * @param path
	 *            XPath expression
	 * @return Node list
	 * @see XPathUtils#selectNodeList(org.w3c.dom.Node, String)
	 */
	public NodeList getNodeList(String path) {
		Node node = this.nodeStack.peek();
		NodeList nodeList = this.xpath.selectNodeList(node, path);
		int nodeCount = nodeList.getLength();
		for (int i = 0; i < nodeCount; i++) {
			Node forNode = nodeList.item(i);
			markUsed(forNode);
		}
		return this.xpath.selectNodeList(node, path);
	}

	/**
	 * Returns the generated target.
	 * 
	 * @return Target
	 */
	public Object getTargetRoot() {
		return this.root;
	}

	/**
	 * Returns the XPath utility.
	 * 
	 * @return XPath utility.
	 */
	public XPathUtils getXpath() {
		return this.xpath;
	}

	/**
	 * Gets a registered instance.
	 * 
	 * @param id
	 *            ID for the instance
	 * @return Registered instance or <code>null</code> if not registered.
	 */
	public Object instanceGet(String id) {
		return this.instanceMap.get(id);
	}

	/**
	 * Peeks an instance from the stack
	 * 
	 * @return Top instance
	 * @see Stack#peek()
	 */
	public Object instancePeek() {
		return this.instanceStack.peek();
	}

	/**
	 * Pops an instance from the stack
	 * 
	 * @return Popped instance
	 * @see Stack#pop()
	 */
	public Object instancePop() {
		return this.instanceStack.pop();
	}

	/**
	 * Pushes an instance onto the stack.
	 * 
	 * @param instance
	 *            Instance to push
	 * @see Stack#push(Object)
	 */
	public void instancePush(Object instance) {
		this.instanceStack.push(instance);
	}

	/**
	 * Registers an instance.
	 * 
	 * @param id
	 *            ID for the instance
	 * @param instance
	 *            Instance to register
	 */
	public void instancePut(String id, Object instance) {
		this.instanceMap.put(id, instance);
	}

	/**
	 * Marks the node as being used.
	 * 
	 * @param node
	 *            Node to mark (if <code>null</code>, nothing is node)
	 */
	public void markUsed(Node node) {
		if (node != null) {
			node.setUserData("OXML", true, null);
			// Parent node
			Node parentNode = node.getParentNode();
			if (parentNode != null && !nodeUsed(parentNode)) {
				markUsed(parentNode);
			}
		}
	}

	/**
	 * New instance.
	 * 
	 * @param className
	 *            the class name
	 * 
	 * @return the object
	 */
	public Object newInstance(String className) {
		// Completion ?
		if (!StringUtils.contains(className, '.')
				&& StringUtils.isNotBlank(this.packageName)) {
			className = this.packageName + "." + className;
		}
		// Creates the instance
		return Utils.newInstance(className);
	}

	/**
	 * Peeks a node from the stack
	 * 
	 * @return Top node
	 * @see Stack#peek()
	 */
	public Node nodePeek() {
		Node node = this.nodeStack.peek();
		// Marks the node as being used
		markUsed(node);
		// OK
		return node;
	}

	/**
	 * Pops a node from the stack
	 * 
	 * @return Popped node
	 * @see Stack#pop()
	 */
	public Node nodePop() {
		return this.nodeStack.pop();
	}

	/**
	 * Pushes a node onto the stack.
	 * 
	 * @param node
	 *            Node to push
	 * @see Stack#push(Object)
	 */
	public void nodePush(Node node) {
		this.nodeStack.push(node);
	}

	/**
	 * Node used.
	 * 
	 * @param node
	 *            the node
	 * 
	 * @return true, if successful
	 */
	protected boolean nodeUsed(Node node) {
		if (node != null) {
			Boolean used = (Boolean) node.getUserData("OXML");
			return (used != null && used);
		} else {
			return true;
		}
	}

	/**
	 * Gets a sequence.
	 * 
	 * @param id
	 *            Sequence ID
	 * 
	 * @return Sequence
	 */
	public SequenceOXMLAction sequenceGet(String id) {
		SequenceOXMLAction sequence = this.sequenceMap.get(id);
		if (sequence != null) {
			return sequence;
		} else {
			throw new IllegalStateException(
					"Cannot find any sequence with ID = " + id);
		}
	}

	/**
	 * Registers a sequence.
	 * 
	 * @param sequence
	 *            the sequence to register
	 */
	public void sequencePut(SequenceOXMLAction sequence) {
		this.sequenceMap.put(sequence.getId(), sequence);
	}

	/**
	 * Sets the config.
	 * 
	 * @param config
	 *            the new config
	 */
	public void setConfig(OXMLContextConfig config) {
		this.config = config;
	}

	/**
	 * Sets the root package name to use when loading classes whose only a
	 * simple name is given
	 * 
	 * @param packageName
	 *            Package name
	 */
	public void setPackageName(String packageName) {
		this.packageName = packageName;
	}

	/**
	 * Sets an instance as being the root.
	 * 
	 * @param instance
	 *            Instance
	 */
	public void setRoot(Object instance) {
		if (this.root == null) {
			this.root = instance;
		} else {
			throw new IllegalStateException(
					"The root instance has already been set.");
		}
	}

}
