package net.sf.cuf.examples.model.portal;

import net.sf.cuf.fw.Dc;
import net.sf.cuf.fw.Application;
import net.sf.cuf.ui.builder.SwingXMLBuilder;
import net.sf.cuf.examples.model.AppData;
import net.sf.cuf.fw2.DialogApplication;
import net.sf.cuf.fw2.DialogDc;
import net.sf.cuf.fw2.AbstractAppDc;
import net.sf.cuf.fw2.ShowStatusEvent;
import net.sf.cuf.fw2.DialogDescription;
import net.sf.cuf.fw2.DialogManager;
import net.sf.cuf.fw2.CloseDialogEvent;
import net.sf.cuf.fw2.OpenDialogEvent;
import net.sf.cuf.ui.SwingConnectionManager;

import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.io.InputStream;

/**
 * The portal dialog manages the main window of the application and coordinates all
 * dialogs contained in that window.
 */
public class PortalDc extends AbstractAppDc implements DialogManager.DialogCallback
{
    /** our peer presentation component */
    private PortalPc            mPc;
    /** the builder that created the UI */
    private SwingXMLBuilder     mBuilder;
    /** our application */
    private DialogApplication   mApp;

    /** the key of the know dialogs map */
    static final String KNOWN_DIALOGS= "knownDialogs";

    public void init(final Dc pParent, final Map<String, ? super Object> pArgs)
    {
        super.init(pParent, pArgs);

        // set up link to our app
        mApp= (DialogApplication) pArgs.get(Application.APPLICATION_KEY);

        // load our Pc and its UI, this also loads the AboutDc
        InputStream in= Thread.currentThread().getContextClassLoader().getResourceAsStream(AppData.XML_APP_UI_NAME);
        Map<String, Object>  nonVisual= new HashMap<>();
        nonVisual.put("PortalDc", this);
        mBuilder= SwingXMLBuilder.create(in, nonVisual);

        // load and init the DialogDc's
        mDialogManager.setCallback(this);
        Map<String, DialogDescription> knownDialogs= DialogDescription.loadKnownDialogs(AppData.UC_DIALOGS);
        DialogDescription.initDialogs(knownDialogs, mApp, this, pArgs);

        // init the AboutDc
        DialogDescription description= new DialogDescription();
        AboutDc aboutDc= (AboutDc)mBuilder.getNonVisualObject("AboutDc");
        aboutDc.init(this, pArgs);
        description.setDc(aboutDc);
        description.setDialogId("About");
        knownDialogs.put(description.getDialogId(), description);

        // set up the dialog manager
        mDialogManager.setKnownDialogs(knownDialogs);

        // remove the About dialog from the known dialogs, because that one isn't handle by our Pc
        knownDialogs.remove("About");

        // now initialize our Pc
        mPc= (PortalPc)mBuilder.getNonVisualObject("PortalPc");
        Map<String, Object> args= new HashMap<>(pArgs);
        args.put(KNOWN_DIALOGS, knownDialogs);
        mPc.init(this, args);
    }

    public void dialogOpened(final Dc pDc, final Map<String, Object> pArgs)
    {
        if ( (pDc instanceof DialogDc) && !(pDc instanceof AboutDc))
        {
            // show the UI
            mPc.showSubDialog(((DialogDc)pDc).getVisualPresentation());
        }
    }

    public void dialogClosed(final Dc pDc, final boolean pWasClosed, final Map<String, Object> pArgs)
    {
        if ( pWasClosed && (pDc instanceof DialogDc) && !(pDc instanceof AboutDc))
        {
            // clear status line
            ShowStatusEvent e= new ShowStatusEvent(this);
            postAppEvent(e);

            // hide the UI
            mPc.hideSubDialog();
        }
    }

    public boolean canPassivate()
    {
        // check if all sub dialogs can be passivated
        return mDialogManager.canPassivate();
    }

    public void doActivate(final Map<String, ? super Object> pArgs)
    {
        // show the frame
        mPc.show();
    }

    public void doPassivate(final Map<String, ? super Object> pArgs)
    {
        // hide the frame
        mPc.hide();
    }

    public void dispose(final Map<String, ? super Object> pArgs)
    {
        // dispose our children first
        mDialogManager.dispose(pArgs);

        // disconnect listeners
        SwingConnectionManager.removeConnection(this);
        mBuilder.dispose();
        mPc.dispose(pArgs);

        super.dispose(pArgs);
    }


    /**
     * Called to close the window / quit the application.
     */
    public void quit()
    {
        CloseDialogEvent close= new CloseDialogEvent(this, this);
        postAppEvent(close);
    }

    /**
     * Called to add a new top-level window.
     */
    public void newWindow()
    {
        OpenDialogEvent open= new OpenDialogEvent(this, "AppDc");
        postAppEvent(open);
    }

    /**
     * Called to show the about dialog.
     */
    public void about()
    {
        OpenDialogEvent open= new OpenDialogEvent(this, "About");
        postAppEvent(open);
    }

    /**
     * Called by our presentation tier to close the current and open a new dialog.
     * @param pNewDialogId the id of the new dialog
     * @return true if the dialog could be opened (= the current one could be closed)
     */
    boolean activateDialog(final String pNewDialogId)
    {
        // we don't ask our dialog manager via mDialogManager.canPassivate() if
        // we can passivate, because the close event below will call canPassivate anyway
        // we assume that there is only one relevant active dialog
        boolean canPassivate= true;

        List<Dc> activeDialogs= mDialogManager.getActiveDialogs();
        for (Dc activeDialog : activeDialogs)
        {
            DialogDc dialogDc = (DialogDc) activeDialog;
            CloseDialogEvent close = new CloseDialogEvent(this, dialogDc);
            postAppEvent(close);
            canPassivate = canPassivate && close.getWasClosed();
        }

        if (canPassivate)
        {
            OpenDialogEvent open = new OpenDialogEvent(this, pNewDialogId);
            postAppEvent(open);
            return true;
        }
        else
        {
            // return false, this will re-select the last selected button
            return false;
        }
    }
}
