package net.sf.cuf.examples.model.uc1;

import net.sf.cuf.fw.Dc;
import net.sf.cuf.fw.Application;
import net.sf.cuf.ui.builder.SwingXMLBuilder;
import net.sf.cuf.fw2.DialogPc;
import net.sf.cuf.fw2.AbstractDialogPc;
import net.sf.cuf.fw2.MenuToolBarEvent;
import net.sf.cuf.ui.table.OptimalColumnWidthSupport;
import net.sf.cuf.ui.table.TableProperties;
import net.sf.cuf.model.ValueModel;
import net.sf.cuf.model.IndexedAdapter;
import net.sf.cuf.model.SelectionInList;
import net.sf.cuf.model.DelegateAccess;
import net.sf.cuf.model.ui.ToolTipAdapter;
import net.sf.cuf.model.ui.LOVCellEditor;
import net.sf.cuf.state.State;
import net.sf.cuf.state.StateExpression;
import net.sf.cuf.state.ui.SwingDocumentState;
import net.sf.cuf.examples.model.AppData;
import net.sf.cuf.examples.model.portal.PortalPc;

import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.JToolBar;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.table.TableColumn;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import java.util.Map;
import java.util.List;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Date;
import java.awt.event.ActionEvent;
import java.awt.Component;

/**
 * UseCase1Pc is the presentation component of a typical CRUD (Create, Read,
 * Update, Delete) business application.
 * The presentation component has a quite complex presentation state build
 * on the following assumptions:
 * o there is one operation to load (all) data, the data (a List with objects)
 *   is stored in the ValueModel AppData.TABLE_DATA
 * o parts of the data are displayed in a list, there is no initial
 *   selection after a load
 * o the data is dirty if a row is selected in the table and the
 *   data in the textfields is has been changed by the user
 * o whenever the user selects a row, the data of that row is copied to
 *   the textfields below the table
 * o any changes to the row are buffered, and reflected in the table
 *   only after a accept
 * o as long as the data is changed ("dirty"), the table is disabled
 *   and the dialog can't be passivated
 * o the new operation is only possible if there is no selected row or
 *   the data in the selected row is not dirty
 * o the delete operation is only possible if there is a selected row and
 *   the data in the selected row is not dirty
 * o the accept operation is only possible when the data is dirty and
 *   can be saved
 * o the cancel operation is only possible when the data is dirty
 * o the load and save operation is only possible when the data is not dirty
 */
public class UseCase1Pc extends AbstractDialogPc implements DialogPc, SwingXMLBuilder.Backlink
{
    /** the builder that created us */
    private SwingXMLBuilder         mBuilder;
    /** our application */
    private Application             mApp;
    /** list of our toobar buttons */
    private List<Component>         mToolbarButtons;
    /** map ouf our menu entries */
    private Map<String, Object>     mMenuMap;
    /** map ouf the index of our menu entries */
    private Map<String, Integer>    mMenuIndexMap;

    /** the name of the table */
    private static final String TABLE_NAME    = "UC1/ScrollPane/Table";

    /**
     * Callback method for the Builder.
     * @param pBuilder the builder that created us
     */
    public void setSwingXMLBuilder(final SwingXMLBuilder pBuilder)
    {
        mBuilder= pBuilder;
    }

    /**
     * Called from the DialogDc peer in the initialized, active or passive
     * state to get the visual representation for this dialog.
     * @return an object that represents the visual representation of this dialog
     */
    public Object getVisualPresentation()
    {
        return mBuilder.getComponentByName("UC1");
    }

    /**
     * init both the UI as well as the data binding
     * @param pDc our peer Dc
     * @param pArgs our arguments
     */
    @Override
    public void init(final Dc pDc, final Map<String, ? super Object> pArgs)
    {
        super.init(pDc, pArgs);

        mApp= (Application)pArgs.get(Application.APPLICATION_KEY);

        // init toobar/menubar info for activation/passivation
        JToolBar toolbar= (JToolBar)mBuilder.getComponentByName("ToolbarUC1");
        mToolbarButtons = new ArrayList<Component>();
        for (int i = 0, n = toolbar.getComponentCount(); i < n; i++)
        {
            mToolbarButtons.add(toolbar.getComponent(i));
        }
        JMenu     rowMenu = (JMenu)    mBuilder.getComponentByName("MenuUC1/Row");
        JMenuItem loadData= (JMenuItem)mBuilder.getComponentByName("MenuUC1/File/LoadDataMenu");
        mMenuMap          = new HashMap<String, Object>();
        mMenuMap.put     (PortalPc.SEPARATOR,        rowMenu);
        mMenuMap.put     (PortalPc.SEPARATOR+"File", loadData);
        mMenuIndexMap     = new HashMap<String, Integer>();
        mMenuIndexMap.put(PortalPc.SEPARATOR, -2);

        // i need to test this adapter somewhere
        new ToolTipAdapter((ValueModel)mBuilder.getNonVisualObject("FirstNameAspect"),
                           mBuilder.getByShortName("FirstName"));

        /*
         * UI fine tuning
         */
        // add excel-like width adjustment
        // because we can't use a SortingTable due to the ValueModel's table adapter,
        // we can't use the cool SortingTable stuff besides the OptimalColumnWidthSupport
        JTable table= (JTable)mBuilder.getComponentByName(TABLE_NAME);
        new OptimalColumnWidthSupport(table);
        table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        // we're inside a scrollbar, so don't auto resize
        table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);

        // load the last table settings
        TableProperties.load(table, mApp.getProperties(), AppData.DEFAULT_TABLE_SETTINGS);

        /*
         * models and connections
         */

        // whenever the selection changes, we update the buffered row
        // WARNING: we _must_ not register to SelectionInList.selectionHolder(), because
        // the CurrentRow value model may get its callback after us, so that the data
        // of the current row is not valid when we copy it in our selectionChanged method
        // WARNING: when using the "listSelectionAction" in modeltest.xml, we have a problem
        // because we disable the table during the callback: the "isAdjusting= true" event
        // is therefore never sent
        ValueModel currentRow= (ValueModel)mBuilder.getNonVisualObject("CurrentRow");
        currentRow.onChangeSend(this, "selectionChanged");

        // state holding if the data as dirty, used by the following debugging code
        StateExpression dataDirty= (StateExpression)mBuilder.getNonVisualObject("DataDirty");
        dataDirty.addChangeListener(new ChangeListener(){
            public void stateChanged(final ChangeEvent pEvent)
            {
                System.out.print  ("data dirty changed, new value= ");
                System.out.print  (((State)pEvent.getSource()).isEnabled());
                System.out.println("  reason: "+((State)pEvent.getSource()).getChangeReason());
            }
        });

        TableColumn lastNameColumn = table.getColumnModel().getColumn(2);
        List<Date> keys= new ArrayList<Date>();
        keys.add(new Date(79, 1, 1));
        keys.add(new Date(69, 1, 1));
        keys.add(new Date(66, 1, 1));
        List<String> items= new ArrayList<String>();
        items.add("Seuring");
        items.add("Steinberger");
        items.add("Biedermann");
        LOVCellEditor cellEditor= new LOVCellEditor(keys, items);
        lastNameColumn.setCellEditor(cellEditor);
        lastNameColumn.setCellRenderer(cellEditor);

    }
    // init is over now ;-)

    /**
     * Register toolbar/menubar via an AppEvent.
     */
    void doActivate()
    {
        MenuToolBarEvent e= new MenuToolBarEvent(this, mToolbarButtons, mMenuMap, mMenuIndexMap);
        postAppEvent(e);
    }

    /**
     * Unregister toolbar/menubar via an AppEvent and save our table settings.
     */
    void doPassivate()
    {
        MenuToolBarEvent e= new MenuToolBarEvent(this, mToolbarButtons, mMenuMap, true);
        postAppEvent(e);

        JTable table= (JTable)mBuilder.getComponentByName(TABLE_NAME);
        TableProperties.store(table, mApp.getProperties(), AppData.DEFAULT_TABLE_SETTINGS);
    }

    /*
     * start of specific methods
     */

    /**
     * Callback from our new action
     * @param pEvent not used
     */
    public void newRow(final ActionEvent pEvent)
    {
        SelectionInList selectionInList= (SelectionInList)mBuilder.getNonVisualObject("SelectionInList");
        selectionInList.addItem(new UserData());
        focusTable();
    }

    /**
     * Callback from our delete action.
     * @param pEvent not used
     */
    public void deleteRow(final ActionEvent pEvent)
    {
        SelectionInList selectionInList= (SelectionInList)mBuilder.getNonVisualObject("SelectionInList");
        selectionInList.removeItem(selectionInList.getIndex());
        focusTable();
    }

    /**
     * Small helper to focus the table.
     */
    private void focusTable()
    {
        // set the focus on the table
        JTable table= (JTable)mBuilder.getComponentByName(TABLE_NAME);
        table.requestFocusInWindow();
    }

    /**
     * Callback from our acceopt action, we write the current buffer to the list.
     * @param pEvent not used
     */
    public void acceptRow(final ActionEvent pEvent)
    {
        // write the buffered value models
        silentDocumentStates(true);
        ValueModel bufferTrigger= (ValueModel)mBuilder.getNonVisualObject("CurrentRowBufferTrigger");
        bufferTrigger.setValueForced(Boolean.TRUE);
        syncDocumentStates();
        silentDocumentStates(false);
    }

    /**
     * Callback from our cancel action, we re-read the buffered row.
     * @param pEvent not used
     */
    public void cancelRow(final ActionEvent pEvent)
    {
        // re-read the buffered value models
        silentDocumentStates(true);
        ValueModel bufferTrigger= (ValueModel)mBuilder.getNonVisualObject("CurrentRowBufferTrigger");
        bufferTrigger.setValueForced(Boolean.FALSE);
        syncDocumentStates();
        silentDocumentStates(false);
    }

    /**
     * Callback from current row that it's content changed, we
     * read the current row in our current row buffer.
     * @param pEvent not used
     */
    public void selectionChanged(final ChangeEvent pEvent)
    {
        // read the buffered value models
        silentDocumentStates(true);
        ValueModel bufferTrigger= (ValueModel)mBuilder.getNonVisualObject("CurrentRowBufferTrigger");
        bufferTrigger.setValueForced(Boolean.FALSE);
        syncDocumentStates();
        silentDocumentStates(false);
    }

    /**
     * Small helper that set the document states silent/active, to avoid
     * "is dirty" broadcasts.
     * @param pSilent true to silent the states
     */
    private void silentDocumentStates(final boolean pSilent)
    {
        SwingDocumentState firstNameState = (SwingDocumentState) mBuilder.getNonVisualObject("FirstNameState");
        SwingDocumentState middleNameState= (SwingDocumentState) mBuilder.getNonVisualObject("MiddleNameState");
        SwingDocumentState lastNameState  = (SwingDocumentState) mBuilder.getNonVisualObject("LastNameState");
        SwingDocumentState bornState      = (SwingDocumentState) mBuilder.getNonVisualObject("BornState");

        firstNameState. setSilent(pSilent);
        middleNameState.setSilent(pSilent);
        lastNameState.  setSilent(pSilent);
        bornState.      setSilent(pSilent);
    }

    /**
     * Small helper that sets the compare content of our document states.
     */
    private void syncDocumentStates()
    {
        UserData value;
        // when the user de-selects the table, we provide whatever the default
        // constructer gives us
        IndexedAdapter currentRow  = (IndexedAdapter)mBuilder.getNonVisualObject("CurrentRow");
        if (currentRow.isIndexInList().booleanValue())
        {
            value= (UserData)currentRow.getValue();
        }
        else
        {
            value= new UserData();
        }

        SwingDocumentState firstNameState = (SwingDocumentState) mBuilder.getNonVisualObject("FirstNameState");
        SwingDocumentState middleNameState= (SwingDocumentState) mBuilder.getNonVisualObject("MiddleNameState");
        SwingDocumentState lastNameState  = (SwingDocumentState) mBuilder.getNonVisualObject("LastNameState");
        SwingDocumentState bornState      = (SwingDocumentState) mBuilder.getNonVisualObject("BornState");

        firstNameState. setCompareContent(value.getFirstName());
        middleNameState.setCompareContent(value.getMiddleName());
        lastNameState.  setCompareContent(value.getLastName());
        DelegateAccess bornConverter= (DelegateAccess)mBuilder.getNonVisualObject("BornConverter");
        bornState.      setCompareContent((String)bornConverter.getValue(value));
    }
}
