package net.sf.cuf.examples.model.uc1;

import net.sf.cuf.fw.Dc;
import net.sf.cuf.fw.Application;
import net.sf.cuf.ui.builder.SwingXMLBuilder;
import net.sf.cuf.model.ValueModel;
import net.sf.cuf.fw2.DialogDc;
import net.sf.cuf.fw2.AbstractDialogDc;
import net.sf.cuf.fw2.BlockEvent;
import net.sf.cuf.fw2.ShowStatusEvent;
import net.sf.cuf.xfer.Dispatch;
import net.sf.cuf.xfer.Response;

import net.sf.cuf.state.State;
import net.sf.cuf.examples.model.AppData;
import net.sf.cuf.xfer.SwingDispatcher;

import javax.swing.event.ChangeEvent;
import java.util.Map;
import java.util.List;
import java.util.HashMap;
import java.io.InputStream;

/**
 * UseCase1Dc is the presentation component of a typical CRUD (Create, Read,
 * Update, Delete) business application.
 * It uses the net.sf.cuf.xfer library to do asynchronous data loading
 * and uses the BlockEvent and ShowStatusEvent to provide the user with
 * feedback.
 * The quite complicated UI work is done in the presentation component (UseCase1Pc).
 */
public class UseCase1Dc extends AbstractDialogDc implements DialogDc
{
    /** the builder that created our Pc and UI */
    private SwingXMLBuilder mBuilder;
    /** our peer dialog component */
    private UseCase1Pc      mPc;
    /** our application */
    private Application     mApp;
    /** counter to similate new data */
    private int             mDummyDataCounter= 0;
    /** our dispatcher */
    private Dispatch        mDispatcher;

    /**
     * This method is called by the parent dialog controller during
     * the initialisation phase of an application.
     *
     * @param pParent our non-null parent
     * @param pArgs variable argument list modeled as a key/value map, never null.
     */
    public void init(final Dc pParent, final Map<String, ? super Object> pArgs)
    {
        super.init(pParent, pArgs);

        // create our dispatcher
        mDispatcher= new SwingDispatcher();

        // set up link to our app
        mApp= (Application) pArgs.get(Application.APPLICATION_KEY);

        // load our Pc and its UI
        InputStream in= Thread.currentThread().getContextClassLoader().getResourceAsStream(AppData.XML_APP_UC1_NAME);
        Map<String, Object> nonVisual= new HashMap<String, Object> ();
        nonVisual.put("UseCase1Dc", this);
        nonVisual.put(AppData.TABLE_DATA, mApp.getAppModel().get(AppData.TABLE_DATA));
        mBuilder= SwingXMLBuilder.create(in, nonVisual);
        mPc     = (UseCase1Pc)mBuilder.getNonVisualObject("UseCase1Pc");
        mPc.init(this, pArgs);
    }

    /**
     * Called from the dialog coordinater in the initialized, active or passive
     * state to get the visual representation for this dialog.
     * @return an object that represents the visual representation of this dialog
     */
    public Object getVisualPresentation()
    {
        return mPc.getVisualPresentation();
    }

    /**
     * We can only passivate if there is no dirty data
     * @return true if there is no data to save
     */
    public boolean canPassivate()
    {
        State dataDirty= (State)mBuilder.getNonVisualObject("DataDirty");
        return !dataDirty.isEnabled();
    }

    /**
     * Called from the dialog coordinator to tell the dialog to switch
     * to the active state.
     * @param pArgs arguments, key is a String, value is any suitable objekt for the key
     */
    public void doActivate(final Map<String, ? super Object> pArgs)
    {
        System.out.println("UseCase1Dc.doActivate");
        mPc.doActivate();
    }

    /**
     * Called from the dialog coordinator to tell the dialog to switch
     * to the passive state.
     * @param pArgs arguments, key is a String, value is any suitable objekt for the key
     */
    public void doPassivate(final Map<String, ? super Object> pArgs)
    {
        System.out.println("UseCase1Dc.doPassivate");
        mPc.doPassivate();
    }

    /**
     * Called from the dialog coordinator to tell the dialog to cleanup all
     * resources. We only call dispose of our Pc.
     * @param pArgs arguments, key is a String, value is any suitable objekt for the key
     */
    public void dispose(final Map<String, ? super Object> pArgs)
    {
        mPc.dispose(pArgs);
    }

    /**
     * Method for our presentation to initiate the data loading.
     * We block the gui, set the status and dispatch a LoadDataRequest in an
     * other thread.
     */
    public void loadData()
    {
        BlockEvent block= new BlockEvent(this, true);
        postAppEvent(block);

        ShowStatusEvent status= new ShowStatusEvent(this, "Lade Daten, das dauert etwas ...");
        postAppEvent(status);

        LoadDataRequest loadDataRequest= new LoadDataRequest(mDummyDataCounter);
        loadDataRequest.setDispatchTarget(this, "dataLoaded");
        mDummyDataCounter++;

        mDispatcher.asyncDispatchInEDT(loadDataRequest);
    }

    /**
     * Callback from the Request issued in loadData(). We change the value model
     * holding the data and the status line and unblock the gui.
     * @param pResponse the response containing either the result or an error
     */
    public void dataLoaded(final Response pResponse)
    {
        String statusText;
        if (pResponse.isValid())
        {
            ValueModel tableData= (ValueModel)mApp.getAppModel().get(AppData.TABLE_DATA);
            tableData.setValue(pResponse.getResult());
            statusText= "Daten geladen";
        }
        else
        {
            statusText= "Daten konnten nicht geladen werden!";
        }

        ShowStatusEvent status= new ShowStatusEvent(this, statusText);
        postAppEvent(status);

        BlockEvent unblock= new BlockEvent(this, false);
        postAppEvent(unblock);
    }

    /**
     * Method for our presentation to initiate the data saving.
     * We block the gui, set the status and dispatch a SaveDataRequest in an
     * other thread.
     */
    public void saveData()
    {
        BlockEvent block= new BlockEvent(this, true);
        postAppEvent(block);

        ShowStatusEvent status= new ShowStatusEvent(this, "Speichere Daten, das dauert etwas ...");
        postAppEvent(status);

        ValueModel tableData= (ValueModel)mApp.getAppModel().get(AppData.TABLE_DATA);
        SaveDataRequest saveDataRequest= new SaveDataRequest((List)tableData.getValue());
        saveDataRequest.setDispatchTarget(this, "dataSaved");

        mDispatcher.asyncDispatchInEDT(saveDataRequest);
    }

    /**
     * Callback from the Request issued in saveData(). We change the status line
     * and unblock the gui.
     * @param pResponse the response containing either the result or an error
     */
    public void dataSaved(final Response pResponse)
    {
        String statusText;
        if (pResponse.isValid())
        {
            statusText= "Daten gespeichert";
        }
        else
        {
            statusText= "Daten konnten nicht gespeichert werden!";
        }

        ShowStatusEvent status= new ShowStatusEvent(this, statusText);
        postAppEvent(status);

        BlockEvent unblock= new BlockEvent(this, false);
        postAppEvent(unblock);
    }

    /**
     * Callback to test ValueModel callbacks via XML.
     * @param pEvent the change event
     */
    public void monitorVMChange(final ChangeEvent pEvent)
    {
        ValueModel source= (ValueModel)pEvent.getSource();
        System.out.println("(new) index value = " + source.getValue());
    }
}
