package net.sf.cuf.examples.model.portal;

import net.sf.cuf.fw.Dc;
import net.sf.cuf.fw.Application;
import net.sf.cuf.ui.builder.SwingXMLBuilder;
import net.sf.cuf.fw2.AppPc;
import net.sf.cuf.fw2.AbstractSwingAppPc;
import net.sf.cuf.fw2.DialogDescription;
import net.sf.cuf.ui.SwingConnectionManager;
import net.sf.cuf.examples.model.AppData;

import javax.swing.JFrame;
import javax.swing.JMenuBar;
import javax.swing.JToolBar;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JToggleButton;
import javax.swing.Icon;
import javax.swing.JScrollPane;
import javax.swing.JMenuItem;
import java.util.Map;
import java.awt.Dimension;
import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;

/**
 * The portal dialog manages the main window of the application and coordinates all
 * dialogs contained in that window.
 */
public class PortalPc extends AbstractSwingAppPc implements AppPc, SwingXMLBuilder.Backlink
{
    /** the builder that created us */
    private SwingXMLBuilder mBuilder;
    /** our application */
    private Application     mApp;
    /** our dialog peer */
    private PortalDc        mDc;
    /** null or the toggle button for the current sub-dialog */
    private JToggleButton   mCurrentDialogButton;
    /** marker if we are inside a callback */
    private boolean         mInsideCB;

    /**
     * Callback method for the Builder.
     * @param pBuilder the builder that created us
     */
    public void setSwingXMLBuilder(final SwingXMLBuilder pBuilder)
    {
        mBuilder= pBuilder;
    }

    /**
     * init both the UI as well as the data binding
     * @param pDc our peer Dc
     * @param pArgs our arguments
     */
    @Override
    public void init(final Dc pDc, final Map<String, ? super Object> pArgs)
    {
        super.init(pDc, pArgs);

        mDc = (PortalDc)   pDc;
        mApp= (Application)pArgs.get(Application.APPLICATION_KEY);

        // set the status label
        mStatusLabel= (JLabel)mBuilder.get("Frame/Panel/Status/StatusText");

        // build inital menu map
        mToolBar= (JToolBar) mBuilder.get("Frame/Toolbar");
        mMenuBar= (JMenuBar) mBuilder.get("Frame/MenuBar");
        mMenuMap.put("/File",   (JMenuItem)mBuilder.get("Frame/MenuBar/File"));
        mMenuMap.put("/Window", (JMenuItem)mBuilder.get("Frame/MenuBar/Window"));

        // set the preffered size, otherwise we have problems with the
        // a missing toolbar
        mToolBar.setPreferredSize(new Dimension(0, 40));

        /*
         * create the use case button bar
         */

        // helper for GridBag layouting
        GridBagConstraints  buttonGBC= new GridBagConstraints();
        buttonGBC.gridx  = 0;
        buttonGBC.fill   = GridBagConstraints.BOTH;
        buttonGBC.insets = new Insets(3, 4, 3, 4);

        JPanel  useCasePanel= (JPanel)mBuilder.get("Frame/Panel/UCScrollPane/UCSelectPanel");
        useCasePanel.setBackground( ColorFactory.getControlShadow() );
        useCasePanel.setLayout(new GridBagLayout());

        ButtonGroup buttonGroup= new ButtonGroup();

        Map<String, DialogDescription> knownDialogs= (Map<String, DialogDescription>)pArgs.get(PortalDc.KNOWN_DIALOGS);
        for (final DialogDescription description : knownDialogs.values())
        {
            // create a toggle button for this description
            Icon icon = mBuilder.getIconCache().getIcon(description.getDialogIcon(), "");
            String text = description.getDialogName();
            JToggleButton button = new JToggleButton(text, icon);
            button.setName(description.getDialogId());
            button.setHorizontalTextPosition(JButton.CENTER);
            button.setVerticalTextPosition(JButton.BOTTOM);
            button.setFocusPainted(false);
            button.setOpaque(false);
            button.setBackground(ColorFactory.getControlShadow());
            button.setBorder(new CrossHairBorder());

            // only one button can be selected
            buttonGroup.add(button);

            // create callback
            SwingConnectionManager.createActionConnection(button, this, "ucSelected");

            // add toggle button
            buttonGBC.gridy = useCasePanel.getComponentCount();
            useCasePanel.add(button, buttonGBC);
        }

        // add spacer at the end
        JPanel spacer= new JPanel();
        spacer.setName("Spacer");
        spacer.setOpaque(false);
        GridBagConstraints spacerGBC= new GridBagConstraints();
        spacerGBC.gridx  = 0;
        spacerGBC.weightx= 1.0;
        spacerGBC.weighty= 1.0;
        spacerGBC.gridy  = useCasePanel.getComponentCount();
        useCasePanel.add(spacer, spacerGBC);

        // set the preffered size, otherwise we have problems with the
        // scrollbar after the first selection
        JScrollPane scrollPane= (JScrollPane)mBuilder.get("Frame/Panel/UCScrollPane");
        scrollPane.setPreferredSize(new Dimension(
                (int)scrollPane.getPreferredSize().getWidth()+6, 0));

        /*
         * set the UI size
         */
        JFrame frame  = (JFrame)mBuilder.getContainerByName("Frame");
        mFrameOrDialog= frame;
        int x         = Integer.parseInt(AppData.DEFAULT_X);
        int y         = Integer.parseInt(AppData.DEFAULT_Y);
        int width     = Integer.parseInt(AppData.DEFAULT_WIDTH);
        int height    = Integer.parseInt(AppData.DEFAULT_WIDTH);
        //noinspection EmptyCatchBlock
        try
        {
            x     = Integer.parseInt(mApp.getProperty(AppData.DEFAULT_X_KEY,      AppData.DEFAULT_X));
            y     = Integer.parseInt(mApp.getProperty(AppData.DEFAULT_Y_KEY,      AppData.DEFAULT_Y));
            width = Integer.parseInt(mApp.getProperty(AppData.DEFAULT_WIDTH_KEY,  AppData.DEFAULT_WIDTH ));
            height= Integer.parseInt(mApp.getProperty(AppData.DEFAULT_HEIGHT_KEY, AppData.DEFAULT_HEIGHT));
        }
        catch (NumberFormatException ignored)
        {
        }
        frame.setLocation(x, y);
        frame.setSize    (width, height);
    }

    @Override
    public void dispose(final Map<String, ? super Object> pArgs)
    {
        // save current size etc.
        JFrame frame = (JFrame)mBuilder.getContainerByName("Frame");
        int    width = frame.getWidth();
        int    height= frame.getHeight();
        int    x     = frame.getX();
        int    y     = frame.getY();
        mApp.setProperty(AppData.DEFAULT_X_KEY,      Integer.toString(x));
        mApp.setProperty(AppData.DEFAULT_Y_KEY,      Integer.toString(y));
        mApp.setProperty(AppData.DEFAULT_WIDTH_KEY,  Integer.toString(width));
        mApp.setProperty(AppData.DEFAULT_HEIGHT_KEY, Integer.toString(height));

        // dispose the frame
        frame.dispose();

        // disconnect listeners
        SwingConnectionManager.removeConnection(this);

        super.dispose(pArgs);
    }


    /**
     * Method for our dialog component to set the right visual presentation.
     * @param pVisualPresentation the visual presentation that should set
     */
    void showSubDialog(final Object pVisualPresentation)
    {
        // add the sub-dialog panel
        Component  child  = (Component)pVisualPresentation;
        JPanel     ucPanel= (JPanel)mBuilder.get("Frame/Panel/SubDialogPanel");
        ucPanel.add(child);
        ucPanel.revalidate();
        ucPanel.repaint();
    }

    /**
     * Hide the sub dialog.
     */
    void hideSubDialog()
    {
        // remove the sub-dialog panel
        JPanel ucPanel= (JPanel)mBuilder.get("Frame/Panel/SubDialogPanel");
        ucPanel.removeAll();
        ucPanel.revalidate();
        ucPanel.repaint();
    }

    /**
     * Show the app frame.
     */
    void show()
    {
        JFrame frame = (JFrame)mBuilder.getContainerByName("Frame");
        frame.setVisible(true);
    }

    /**
     * Hide the app frame.
     */
    void hide()
    {
        JFrame frame = (JFrame)mBuilder.getContainerByName("Frame");
        frame.setVisible(false);
    }

    /**
     * Callback of one of the use case buttons.
     * @param pEvent the event
     */
    public void ucSelected(final ActionEvent pEvent)
    {
        // check if we are not inside a callback
        if (mInsideCB)
            return;

        mInsideCB= true;
        try
        {
            JToggleButton newDialogButton= (JToggleButton)pEvent.getSource();
            // ignore clicks on the same UC icon
            //noinspection ObjectEquality
            if (newDialogButton==mCurrentDialogButton)
            {
                return;
            }
            String  newDialogId= newDialogButton.getName();
            boolean activated  = mDc.activateDialog(newDialogId);

            // set the old button if we could not avtivate the new dialog
            if (!activated)
            {
                if (mCurrentDialogButton!=null)
                    mCurrentDialogButton.setSelected(true);
            }
            else
            {
                mCurrentDialogButton= newDialogButton;
            }
        }
        finally
        {
            mInsideCB= false;
        }
    }
}
