package net.sf.cuf.examples.model;

import net.sf.cuf.fw2.UserProperties;
import net.sf.cuf.ui.SwingDecorator;
import net.sf.cuf.fw.Dc;
import net.sf.cuf.fw2.AbstractDialogSwingApplication;
import net.sf.cuf.fw2.DialogApplication;
import net.sf.cuf.fw2.DialogDescription;
import net.sf.cuf.fw2.OpenDialogEvent;
import net.sf.cuf.fw2.DialogManager;
import net.sf.cuf.model.ValueHolder;
import com.jgoodies.looks.Options;

import javax.swing.UIManager;
import javax.swing.SwingUtilities;
import java.io.IOException;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.util.Map;
import java.util.ArrayList;

/**
 * This example application demonstrates most of the features of the
 * net.sf.cuf.fw2, net.sf.cuf.xfer, net.sf.cuf.state
 * and net.sf.cuf.ui.model classes.
 * The class is mainly responsible for the starting and stopping of
 * the application.
 */
public class Main extends AbstractDialogSwingApplication implements DialogApplication, DialogManager.DialogCallback
{
    /**
     * Called to start an application.
     * @param pArgs the arguments of the static main entry point.
     */
    public void doStart(final String[] pArgs)
    {
        super.doStart(pArgs);

        // load properties
        loadProperties();

        /*
         * The following tweaks were taken from
         * http://wiki.java.net/bin/view/Javadesktop/SecretSwingProperties and
         * http://www.javadesktop.org/forums/thread.jspa?threadID=7311&tstart=255
         */

        // relayout if component size changes
        Toolkit.getDefaultToolkit().setDynamicLayout(true);

        // do not erase the backgound during moving windows "gray fog"
        System.setProperty("sun.awt.noerasebackground", "true");

        // keep awt working set in memory (only relevant on Windows, needs Java 5.0)
        // for details see http://support.microsoft.com/default.aspx?scid=kb;en-us;293215
        // and http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=5043070
        System.setProperty("sun.awt.keepWorkingSetOnMinimize", "true");

        // use anti-alias text on Linux
        if (System.getProperty("os.name", "unknown").toLowerCase().contains("linux"))
        {
            System.setProperty("swing.aatext", "true");
        }

        // init Look&Feel
        try
        {
            // check if the user set a specifc LAF from the command line
            String defaultLAF= System.getProperty("swing.defaultlaf");
            if (defaultLAF==null)
            {
                // jgoodies code start
                UIManager.put(Options.USE_SYSTEM_FONTS_APP_KEY, Boolean.TRUE);
                Options.setDefaultIconSize(new Dimension(18, 18));
                defaultLAF= Options.getCrossPlatformLookAndFeelClassName();
                // jgoodies code end, an alternativ is the following line:
                //defaultLAF= UIManager.getSystemLookAndFeelClassName();
            }
            UIManager.setLookAndFeel(defaultLAF);
        }
        catch (Exception ignored) { }

        // init SwingDecorator (only needed by SortingTable)
        SwingDecorator.addBundle("i18n/sortingtable");

        // init our appplication domain models
        ValueHolder tableListHolder= new ValueHolder();
        tableListHolder.setValue(new ArrayList());
        getAppModel().put(AppData.TABLE_DATA, tableListHolder);

        // load the (top-level) dialogs we know
        Map<String, DialogDescription> knownDialogs= DialogDescription.loadKnownDialogs(AppData.APP_DIALOG);
        mDialogManager.setKnownDialogs(knownDialogs);
        mDialogManager.setCallback(this);

        // create top-level Dc and init Dc's/Pc's in the Swing's event dispatch thread (EDT)
        SwingUtilities.invokeLater(new Runnable() {
            public void run()
            {
                OpenDialogEvent openDialogEvent= new OpenDialogEvent(this, AppData.APPDC_ID);
                postAppEvent(openDialogEvent);
            }
        });

    }

    public void dialogOpened(final Dc pDc, final Map pArgs)
    {
    }

    public void dialogClosed(final Dc pDc, final boolean pWasClosed, final Map pArgs)
    {
        // if there is no dialog left, we quit the application
        if (mDialogManager.getActiveDialogs().isEmpty())
        {
            System.out.println("no active dialogs left, will stop");
            doStop();
        }
    }

    /**
     * Called to stop an application. Will call saveProperties() and
     * exits then.
     */
    public void doStop()
    {
        try
        {
            // persist our appplication domain models
            // (no domain ojbects are available in this simple application)

            // save the application properties
            saveProperties();
        }
        finally
        {
            // we must not return
            System.exit(0);
        }
    }

    /**
     * Small helper to load the properties.
     */
    private void loadProperties()
    {
        try
        {
            mAppProperties = UserProperties.read(null, null, AppData.USER_PROPERTIES);
        }
        catch (IOException e)
        {
            System.err.println("loadProperties problems: " + e);
        }
    }

    /**
     * Save the application state to a persistence storage.
     */
    private void saveProperties()
    {
        try
        {
            UserProperties.write(mAppProperties, null, AppData.USER_PROPERTIES);
        }
        catch (IOException e)
        {
            System.err.println("saveProperties problems: " + e);
        }
    }

    /**
     * Main entry point to the application.
     * @param pArgs the arguments
     */
    public static void main(final String[] pArgs)
    {
        DialogApplication app= new Main();
        app.doStart(pArgs);
    }
}
