package net.sf.cuf.fw2;

import net.sf.cuf.appevent.AppEvent;

/**
 * A BlockEvent is used to by a Dc/Pc to signal "the user should not make any
 * further input".
 */
public class BlockEvent extends AppEvent
{
    /** true if the event is sent to block the GUI, false if the GUI should
      * be unblocked */
    private final boolean mBlock;

    /** time we estimate the block will take, -1 if we don't know */
    private final long    mEstimatedBlockDuration;

    private final boolean mReFocus;

    public static BlockEvent createBlockEvent(final Object pSource)
    {
        return new BlockEvent(pSource, true);
    }

    public static BlockEvent createUnBlockEvent(final Object pSource)
    {
        return new BlockEvent(pSource, false);
    }

    public static BlockEvent createNoRefocusUnBlockEvent(final Object pSource)
    {
        return new BlockEvent(pSource, false, -1, false);
    }

    /**
     * Create a non-consumed event to block or unblock the GUI.
     * @param pSource null or event source
     * @param pBlock flag if we should block (true) or unblock (false)
     */
    public BlockEvent(final Object pSource, final boolean pBlock)
    {
        this(pSource, pBlock, -1, true);
    }

    /**
     * Create a non-consumed event to block the GUI.
     * @param pSource null or event source
     * @param pEstimatedBlockDuration -1 or the estimated duration of the block
     */
    public BlockEvent(final Object pSource, final long pEstimatedBlockDuration)
    {
        this(pSource, true, pEstimatedBlockDuration, true);
    }

    /**
     * Common constructor code
     * @param pSource null or event source
     * @param pBlock flag if we should block (true) or unblock (false)
     * @param pEstimatedBlockDuration -1 or the estimated duration of the block
     * @param pReFocus flag if we should re-focus on unblock or not, ignored if pBlock is true
     */
    public BlockEvent(final Object pSource, final boolean pBlock, final long pEstimatedBlockDuration, final boolean pReFocus)
    {
        super(pSource);
        if (pEstimatedBlockDuration<0 && pEstimatedBlockDuration!=-1)
        {
            throw new IllegalArgumentException("block duration must be -1 or >=0 and not "+pEstimatedBlockDuration);
        }
        mBlock                 = pBlock;
        mEstimatedBlockDuration= pEstimatedBlockDuration;
        mReFocus               = pReFocus;
    }

    /**
     * Returns true if the event is sent to block the GUI, false if the GUI should
     * be unblocked.
     * @return true if GUI should be blocked
     */
    public boolean isBlock()
    {
        return mBlock;
    }

    /**
     * Returns the estimated time the block will take, -1 if we don't know.
     * @return estimated block time
     */
    public long getEstimatedBlockDuration()
    {
        return mEstimatedBlockDuration;
    }

    /**
     * @return true if we should refocus on unblocking
     */
    public boolean isReFocus()
    {
        return mReFocus;
    }
}
