package net.sf.cuf.fw2;

import net.sf.cuf.appevent.AppEvent;
import net.sf.cuf.appevent.AppEventSupport;
import net.sf.cuf.appevent.AppEventManager;
import net.sf.cuf.appevent.BindEvent;
import net.sf.cuf.appevent.UnbindEvent;
import net.sf.cuf.fw.Dc;
import net.sf.cuf.fw.Application;

import java.util.Map;
import java.util.List;
import java.util.Arrays;

/**
 * The AbstractAppDc implements the common stuff (AppEvent handling)
 * of the AppDc interface.
 * A project specific AppDc is usally derived from AbstractAppDc.
 */
public abstract class AbstractAppDc implements AppDc
{
    /** Our app event manager. */
    protected AppEventManager   mAppEventManger;

    /** The next object in the AppEvent responder chain. */
    protected AppEventSupport   mAppEventSupport;

    /** The manager object that opens/closes dialogs. */
    protected DialogManager     mDialogManager;

    /** A list of the events we do not forward but handle via our own AppEventManager. */
    protected static final List<Class<? extends AppEvent>> BLOCKED_EVENTS= Arrays.asList(MenuToolBarEvent.class,
                                                                        ShowStatusEvent.class,
                                                                        BlockEvent.class);

    /**
     * This method is called by the Application during
     * the initialisation phase of an application.
     *
     * @param pParent null, because we are the top Dc
     * @param pArgs variable argument list modeled as a key/value map, never null.
     */
    public void init(final Dc pParent, final  Map<String, ? super Object>  pArgs)
    {
        Object application= pArgs.get(Application.APPLICATION_KEY);
        mAppEventSupport= (AppEventSupport) application;
        mAppEventManger = new AppEventManager();
        mDialogManager  = new DialogManager((Application)application);
    }

    public Dc getActiveDialog()
    {
        return mDialogManager.getActiveDialog();
    }

    public List<Dc> getActiveDialogs()
    {
        return mDialogManager.getActiveDialogs();
    }

    /**
     * Process/Dispatch an AppEvent.
     *
     * @param pAppEvent event that should be routed/processes
     * @throws IllegalArgumentException if pAppEvent is null
     */
    public void postAppEvent(final AppEvent pAppEvent)
    {
        // special handling to capture MenuToolBarEvent, BlockEvent and ShowStatusEvent
        if (pAppEvent instanceof BindEvent)
        {
            if (BLOCKED_EVENTS.contains(((BindEvent)pAppEvent).getTriggerClass()))
                mAppEventManger.postAppEvent(pAppEvent);
        }
        else if (pAppEvent instanceof UnbindEvent)
        {
            if (BLOCKED_EVENTS.contains(((UnbindEvent)pAppEvent).getTriggerClass()))
                mAppEventManger.postAppEvent(pAppEvent);
        }
        else if (BLOCKED_EVENTS.contains(pAppEvent.getClass()))
        {
            mAppEventManger.postAppEvent(pAppEvent);
        }

        pAppEvent.forward();
        if (!pAppEvent.isConsumed())
        {
            mDialogManager.postAppEvent(this, mAppEventSupport, pAppEvent);
        }
    }

    /**
     * Called from the application to to cleanup all resources.
     * The base implementation does nothing.
     * @param pArgs arguments, key is a String, value is any suitable objekt for the key
     */
    public void dispose(final Map<String, ? super Object> pArgs)
    {
    }
}
