package net.sf.cuf.appevent;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

/**
 * A BindEvent object is used to bind AppEvent classes to (any number of)
 * event receivers.
 * @see UnbindEvent
 */
public class BindEvent<T extends AppEvent> extends AppEvent
{
    /** callback method */
    private AppEventListener<T>         mMethodToCallBack;
    /** trigger class */
    private Class<T>                    mAppEventToBindTo;
    /** -1 or number of outstanding hops */
    private int                         mHopCount;

    /**
     * The constructor creates a BindEvent with the handed values and an
     * unbound hop-count.
     *
     * @param pSource           sender object, will receive the callback
     * @param pMethodToCallBack method ot call back for the given AppEvent class
     * @param pAppEventToBindTo trigger class, must be a subclass of AppEvent
     * @throws IllegalArgumentException if one of the arguments is null
     *         or the method is not known/not public.
     */
    public BindEvent(final Object pSource, final AppEventListener<T> pMethodToCallBack,
                     final Class<T> pAppEventToBindTo)
    {
        this(pSource, pMethodToCallBack, pAppEventToBindTo, 0);
    }

    /**
     * The constructor creates a BindEvent with the handed values.
     *
     * @param pSource           sender object, will receive the callback
     * @param pMethodToCallBack method ot call back for the given AppEvent class
     * @param pAppEventToBindTo trigger class, must be a subclass of AppEvent
     * @param pHopCount         number of hops this event gets forwarded
     *                          before it consumes itself. If pHopCount is 0,
     *                          there is no hop limit.
     * @throws IllegalArgumentException if one of the arguments is null,
     *         pHopCount is negative or the method is not known/not public.
     */
    public BindEvent(final Object pSource, final AppEventListener<T> pMethodToCallBack,
                     final Class<T> pAppEventToBindTo, final int pHopCount)
    {
        super(pSource);
        if (pSource==null)
            throw new IllegalArgumentException("pSource must not be null");
        if (pMethodToCallBack==null)
            throw new IllegalArgumentException("pMethodToCallBack must not be null");
        if (pAppEventToBindTo==null)
            throw new IllegalArgumentException("pAppEventToBindTo must not be null");
        if (!AppEvent.class.isAssignableFrom(pAppEventToBindTo))
            throw new IllegalArgumentException(pAppEventToBindTo.getName() + " is not derived from AppEvent");
        if (pHopCount<0)
            throw new IllegalArgumentException("pHopCount must be >= 0");
        mMethodToCallBack= pMethodToCallBack;
        mAppEventToBindTo= pAppEventToBindTo;
        mHopCount        = pHopCount;
        if (mHopCount==0)
        {
            mHopCount= -1;
        }
    }

    /**
     * The constructor creates a BindEvent with the handed values and an
     * unbound hop-count.
     *
     * @param pSource           sender object, will receive the callback
     * @param pMethodToCallBack method name of pSource
     * @param pAppEventToBindTo trigger class, must be a subclass of AppEvent
     * @throws IllegalArgumentException if one of the arguments is null
     *         or the method is not known/not public.
     */
    public BindEvent(final Object pSource, final String pMethodToCallBack,
                     final Class<T> pAppEventToBindTo)
    {
        this(pSource, pMethodToCallBack, pAppEventToBindTo, 0);
    }

    /**
     * The constructor creates a BindEvent with the handed values.
     *
     * @param pSource           sender object, will receive the callback
     * @param pMethodToCallBack method name of pSource
     * @param pAppEventToBindTo trigger class, must be a subclass of AppEvent
     * @param pHopCount         number of hops this event gets forwarded
     *                          before it consumes itself. If pHopCount is 0,
     *                          there is no hop limit.
     * @throws IllegalArgumentException if one of the arguments is null,
     *         pHopCount is negative or the method is not known/not public.
     */
    public BindEvent(final Object pSource, final String pMethodToCallBack,
                     final Class<T> pAppEventToBindTo, final int pHopCount)
    {
        super(pSource);
        if (pSource==null)
            throw new IllegalArgumentException("pSource must not be null");
        if (pMethodToCallBack==null)
            throw new IllegalArgumentException("pMethodToCallBack must not be null");
        if (pAppEventToBindTo==null)
            throw new IllegalArgumentException("pAppEventToBindTo must not be null");
        if (!AppEvent.class.isAssignableFrom(pAppEventToBindTo))
            throw new IllegalArgumentException(pAppEventToBindTo.getName() + " is not derived from AppEvent");
        if (pHopCount<0)
            throw new IllegalArgumentException("pHopCount must be >= 0");
        if (!Modifier.isPublic(pSource.getClass().getModifiers()))
            throw new IllegalArgumentException("pSource must be an object from a public class");

        // create Method Object
        try
        {
            Method methodToCallBack= pSource.getClass().getMethod(pMethodToCallBack, pAppEventToBindTo);
            if (!Modifier.isPublic(methodToCallBack.getModifiers()))
            {
                throw new IllegalArgumentException(pMethodToCallBack+" isn't a public method");
            }
            mMethodToCallBack= pAppEvent -> {
                try
                {
                    Object[] args = {pAppEvent};

                    // do it: call the method with the incoming AppEvent
                    methodToCallBack.invoke(pSource, args);
                }
                catch (IllegalAccessException iae)
                {
                    // this should never happen, because we checked it
                    // during creation of the BindEvent
                    throw new IllegalStateException(iae.toString(), iae);

                }
                catch (InvocationTargetException ite)
                {
                    // map the cause of e to a IllegalArgumentException, if it is
                    // not a RuntimeException or an Error
                    Throwable cause = ite.getTargetException();

                    if (cause instanceof RuntimeException)
                    {
                        throw (RuntimeException) cause;
                    }
                    else if (cause instanceof Error)
                    {
                        throw (Error) cause;
                    }

                    throw new IllegalArgumentException(cause != null ? cause.getMessage() : ite.getMessage(), cause);
                }

            };
        }
        catch (NoSuchMethodException e)
        {
            throw new IllegalArgumentException(e.getMessage(), e);
        }

        // remember other stuff
        mAppEventToBindTo= pAppEventToBindTo;
        mHopCount        = pHopCount;
        if (mHopCount==0)
        {
            mHopCount= -1;
        }
    }


    /**
     * During the propagation in the chain of responsibility, at each hop
     * the forward() method should be called according to the AppEventSupport
     * "contract".
     * We use this to consume ourselves if we had a hop count.
     */
    public void forward()
    {
        if (mHopCount<0)
        {
            return;
        }

        if (mHopCount < 1)
        {
            consume();
        }
        mHopCount--;
    }

    /**
     * Returns the method to call back.
     *
     * @return the method to call back
     */
    public AppEventListener getMethodToCallBack()
    {
        return mMethodToCallBack;
    }

    /**
     * Returns the AppEvent derived class that the target is interested in.
     *
     * @return trigger class
     */
    public Class<T> getTriggerClass()
    {
        return mAppEventToBindTo;
    }
}
