package net.sf.cuf.xfer;

/**
 * The dispatcher class provides methods to produce and dispatch
 * a response object from a request object. Because this implementation
 * has no state, it may be used as a static instance for dispatching.
 */
public abstract class AbstractDispatcher implements Dispatch
{
    /**
     * Create a (stateless) dispatcher.
     */
    public AbstractDispatcher()
    {
    }

    /**
     * Generate a response and and dispatch it in a new (separate) thread.
     * @param pRequest the request
     */
    @Override
    public <T> void asyncDispatch(final Request<T> pRequest)
    {
        Thread load= new Thread(() -> syncDispatch(pRequest));
        load.start();
    }

    /**
     * Generate a response in a new (separate) thread and dispatch the
     * response in the EDT (event dispatch thread).
     * @param pRequest the request
     */
    @Override
    public <T> void asyncDispatchInEDT(final Request<T> pRequest)
    {
        Thread load= new Thread(new DispatchRunnable<>(pRequest));
        load.start();
    }

    /**
     * Generate a response (includes delegate handling) but doesn't dispatch the response.
     * @param pRequest the request
     * @return the response
     */
    @Override
    public <T> Response<T> syncExecute(final Request<T> pRequest)
    {
        RequestDelegate<T> delegate= pRequest.getDelegate();
        Response<T>        response;
        if (delegate!=null)
        {
            response= delegate.execute(pRequest);
        }
        else
        {
            response= pRequest.execute();
        }
        return response;
    }

    /**
     * Generate a response and and dispatch it in the current thread.
     * @param pRequest the request
     */
    @Override
    public <T> void syncDispatch(final Request<T> pRequest)
    {
        DispatchTarget<T> target = pRequest.getDispatchTarget();
        Response<T> response = syncExecute(pRequest);
        syncDispatch(target, response);
    }

    /**
     * Small helper to do common work.
     * @param pTarget the target for dispatching
     * @param pResponse the response to dispatch
     */
    private <T> void syncDispatch(final DispatchTarget<T> pTarget, final Response<T> pResponse)
    {
        pTarget.callback(pResponse);
    }

    /**
     * The callback method to switch to the "right" UI thread.
     * @param pRunnable the runnable containing the result, must not be null
     */
    protected abstract void doDispatch(final Runnable pRunnable);

    /**
     * The runnable handling the dispatch.
     */
    protected class DispatchRunnable<T> implements Runnable
    {
        /** the target, never null */
        private final Request<T> mRequest;
        /** the target, never null */
        private final DispatchTarget mTarget;
        /** the response, initially null */
        private Response mResponse;

        /**
         * Create a new runnable for dispatching the request
         * @param pRequest the request, must not be null, must have a target
         */
        public DispatchRunnable(final Request<T> pRequest)
        {
            mRequest= pRequest;
            mTarget = pRequest.getDispatchTarget();
            if (mTarget == null)
            {
                throw new IllegalArgumentException("target must not be null");
            }
        }

        public void run()
        {
            if (mResponse != null)
            {
                // in EDT
                syncDispatch(mTarget, mResponse);
            }
            else
            {
                // in new thread
                mResponse = syncExecute(mRequest);
                if (mResponse != null)
                {
                    doDispatch(this);
                }
            }
        }

    }
}
