package net.sf.cuf.fw2;

import java.util.Map;

/**
 * The SimpleLifeCycle interface describes a part of a more complex
 * lifecycle, and is used to encapsulate the common life cycle of
 * AppDc and DialogDc. <br>
 * The lifecycle that is described by the SimpleLifeCycle interface is
 * shown below:
 * <code>
 *               doActivate()          doPassivate()
 * [initialized] -----------&gt; [active] ------------&gt; [passive]
 *                                     &lt;------------
 *                                     doActivate()
 * </code>
 * The arguments of the {@link #doActivate(Map)} and {@link #doPassivate(Map)} calls may
 * also be used to return stuff to the caller.
 */
public interface SimpleLifeCycle
{
    /**
     * Key for the parameter of the {@link #doPassivate(Map)}, the value is the
     * optional flag (value type {@link Boolean}) which is true
     * if the method is called directly after a call to {@link #canPassivate()}.
     * In that case, the implementation should also use the force mode for
     * its child dialog hierarchy (if any) to avoid duplicate calls to the
     * {@link #canPassivate()}.
     */
    String FORCE_CLOSE_KEY = "net.sf.cuf.fw2.SimpleLifeCycle.forceClose";

    /**
     * Test to check if a dialog in the active state can be moved
     * to the passive state.<br>
     * The dialog may e.g. check if it has unsaved data and prompt the user
     * for information how to proceed.<br>
     * If the dialog returns true, it may be moved to the passive state by
     * a call to {@link #doPassivate(Map)}, if it returns false, it will remain in the
     * active state and no {@link #doPassivate(Map)} call will happen.
     * In the call to {@link #doPassivate(Map)} directly after this method returned true,
     * the additional value {@link #FORCE_CLOSE_KEY} is passed.
     * @return true if the dialog can be passivated
     */
    boolean canPassivate();

    /**
     * Called from the dialog coordinator to tell the dialog to switch
     * to the active state. If the dialog is an modal dialog, this method
     * returns after the dialog closed.
     * @param pArgs arguments, key is a String, value is any suitable object for the key
     */
    void    doActivate (Map<String, ? super Object> pArgs);

    /**
     * Called from the dialog coordinator to tell the dialog to switch
     * to the passive state. If the dialog is a modal dialog, this method is
     * called "inside" the {@link #doActivate(Map)} method.
     * If {@link #canPassivate()} was called directly before this method, then
     * the parameter will contain the flag {@link #FORCE_CLOSE_KEY} to indicate that
     * calls to the child dialog hierarchy (if any) should also use the force mode
     * to avoid duplicate calls to {@link #canPassivate()}.
     * @param pArgs arguments, key is a String, value is any suitable object for the key
     */
    void    doPassivate(Map<String, ? super Object> pArgs);
}
