package net.sf.cuf.xfer;

/**
 * This class provides the common stuff of all Request classes.
 * It provides delegate and dispatch target handling.
 * In most cases, a derived class must only implement the execute() method.
 */
public abstract class AbstractRequest<T> implements Request<T>
{
    /** null or our delegate */
    private RequestDelegate<T> mDelegate;

    /** null or our target */
    private DispatchTarget  mTarget;

    /**
     * Derived classes must call this constructor via super() if they want to get
     * the benefit of delegate handling.
     */
    public AbstractRequest()
    {
        mDelegate= null;
        mTarget  = null;

        String delegateClassName= System.getProperty(this.getClass().getName()+".delegate");
        if (delegateClassName != null)
        {
            //noinspection EmptyCatchBlock
            try
            {
                Object delegate;
                if (delegateClassName.equals(this.getClass().getName()))
                {
                    delegate= this;
                }
                else
                {
                    delegate= Class.forName(delegateClassName).newInstance();
                }
                if (delegate instanceof RequestDelegate)
                {
                    mDelegate= (RequestDelegate)delegate;
                }
            }
            catch (Exception ignored)
            {
                // if the delegate extraction does not work, silently ignore it
            }
        }
    }

    /**
     * Sets our dispatch target.
     * @param pTarget the target to set (null is o.k.)
     */
    public void setDispatchTarget(final DispatchTarget pTarget)
    {
        mTarget= pTarget;
    }

    /**
     * Sets our dispatch target.
     * @param pTargetObject target to call, must not be null
     * @param pMethodName method to call, must not be null and match the
     *        void <pMethodname>(Response pResponse) signature
     */
    public void setDispatchTarget(final Object pTargetObject, final String pMethodName)
    {
        DispatchTarget target= new DefaultDispatchTarget();
        target.setTarget(pTargetObject, pMethodName);
        setDispatchTarget(target);
    }

    /**
     * Returns our dispatch target.
     * @return dispatch target (may be null)
     */
    public DispatchTarget getDispatchTarget()
    {
        return mTarget;
    }

    /**
     * Returns our request delegate.
     * @return request delegate (may be null)
     */
    public RequestDelegate<T> getDelegate()
    {
        return mDelegate;
    }

    /**
     * A request object may be canceled if this method returns true.
     * @return always false
     */
    public boolean canBeCancelled()
    {
        return false;
    }

    /**
     * Cancel the current request; this implementation will always throw an
     * IllegalStateException.
     * @throws IllegalStateException always thrown
     */
    public void cancel() throws IllegalStateException
    {
        throw new IllegalStateException("this request can't be canceled");
    }

    /**
     * A AbstractRequest is equal to an other AbstractRequest if their
     * delegates and dispatch targets are equal.
     * @param pOther object we compare to
     * @return true if the objects are equal
     */
    public boolean equals(final Object pOther)
    {
        if (this == pOther)
            return true;
        if (!(pOther instanceof AbstractRequest))
            return false;

        final AbstractRequest abstractRequest = (AbstractRequest) pOther;

        if (mDelegate!=null ? !mDelegate.equals(abstractRequest.mDelegate) : abstractRequest.mDelegate!=null) return false;
        //noinspection RedundantIfStatement
        if (mTarget  !=null ? !mTarget.equals(abstractRequest.mTarget)     : abstractRequest.mTarget  !=null) return false;

        return true;
    }

    public int hashCode()
    {
        int result;
        result = (mDelegate != null ? mDelegate.hashCode() : 0);
        result = 29 * result + (mTarget != null ? mTarget.hashCode() : 0);
        return result;
    }

    /**
     * Returns a String representation of this AbstractRequest.
     * @return  a String representation of this AbstractRequest.
     */
    public String toString()
    {
        StringBuilder sb = new StringBuilder(super.toString());
        sb.append("[target=");
        sb.append(mTarget);
        sb.append(", delegate=");
        sb.append(mDelegate);
        sb.append(']');
        return sb.toString();
    }
}
