package net.sf.cuf.appevent;

import java.util.Map;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;

/**
 * A AppEventManager object can be used to dispatch events to registered listeners.
 * The binding between the event class and the listener is done via a BindEvent.
 * Usally a AppEventManger object is the last responder in a chain of responsabilty,
 * and called to dispatch the event if somebody has shown interest.
 */
public class AppEventManager implements AppEventSupport
{
    /**
     * Contains our binding state.
     * <code>
     * Map: key    Class appEventToBindTo,
     *      value  Map: key    Object target
     *                  value  Method methodToCallBack
     * </code>
     */
    private Map<Class<? extends AppEvent>, Map<Object, Method>> mBinding;

    /**
     * The constructor creates a new AppEventManager.
     */
    public AppEventManager()
    {
        mBinding= new HashMap<Class<? extends AppEvent>, Map<Object, Method>>();
    }

    /**
     * The method handles first BindEvent/UnbindEvent objects, and
     * dispatches all other objects to registered targets.
     * If no dispatcher is found, the event is not consumed.
     *
     * @param pAppEvent the event that should be dispatched
     * @throws IllegalArgumentException if pAppEvent is null
     */
    public void postAppEvent(final AppEvent pAppEvent)
    {
        if (pAppEvent==null)
        {
            throw new IllegalArgumentException("pAppEvent must not be null");
        }

        // fullfill AppEventSupport contract
        pAppEvent.forward();
        if (pAppEvent.isConsumed())
        {
            return;
        }

        // either bind ..
        if (pAppEvent instanceof BindEvent)
        {
            bind((BindEvent)pAppEvent);
        }
        // or unbind ..
        else if (pAppEvent instanceof UnbindEvent)
        {
            unbind((UnbindEvent)pAppEvent);
        }
        // or dispatch
        else
        {
            dispatch(pAppEvent);
        }
    }

    /**
     * Small helper method to check if a given Event-class is bound.
     *
     * @param  pAppEvent   event that should be checked
     * @return true, if the event class has one or more targets,
     *         false otherwise
     * @throws IllegalArgumentException if pAppEvent is null
     */
    public boolean isBound(final AppEvent pAppEvent)
    {
        if (pAppEvent==null)
        {
            throw new IllegalArgumentException("pAppEvent must not be null");
        }

        return mBinding.containsKey(pAppEvent.getClass());
    }

    /**
     * Adds a binding to this AppEventManager.
     *
     * @param pBind  event containing all necessary information
     * @throws IllegalArgumentException if pBind is null
     */
    private void bind(final BindEvent pBind)
    {
        // get stuff from event
        Object                    target          = pBind.getTarget();
        Method                    methodToCallBack= pBind.getMethod();
        Class<? extends AppEvent> appEventToBindTo= pBind.getTriggerClass();

        // get map for the trigger class
        Map<Object, Method> targets;
        if (mBinding.containsKey(appEventToBindTo))
        {
            targets= mBinding.get(appEventToBindTo);
        }
        else
        {
            // Warning: targets should respect the order, so it has to be
            // a LinkedHashMap
            targets= new LinkedHashMap<Object, Method>();
            mBinding.put(appEventToBindTo, targets);
        }

        // if there was already an entry, we overwrite it
        targets.put(target, methodToCallBack);

        // consume the event
        pBind.consume();
    }

    /**
     * Removes a binding from this AppEventManager.
     * If no binding is found, nothing happens.
     *
     * @param pUnbind event containing all necessary information
     */
    private void unbind(final UnbindEvent pUnbind)
    {
        // get stuff from event
        Object                    target          = pUnbind.getTarget();
        Class<? extends AppEvent> appEventToUnbind= pUnbind.getTriggerClass();

        // get map for the trigger class
        if (mBinding.containsKey(appEventToUnbind))
        {
            // remove target, if target is not in targets nothing bad happens
            Map<Object, Method> targets= mBinding.get(appEventToUnbind);
            targets.remove(target);

            // consume the event only if we knew the binding
            pUnbind.consume();
        }

    }

    /**
     * Dispatchs the event to on or more receivers.
     * If no suitable binding is found, nothing happens.
     * As soon as the event is consumed, dispatching stops.
     *
     * @param pAppEvent event that gets dispatched
     * @throws IllegalArgumentException if pAppEvent is null
     */
    private void dispatch(final AppEvent pAppEvent)
    {
        // check if we need to do work at all
        if (!isBound(pAppEvent))
        {
            return;
        }

        // copy targets so that a call to removeBinding() during
        // execution will not trigger a exception
        // Warning: targets should respect the order, so it has to be
        // a LinkedHashMap
        Map<Object, Method> targets= new LinkedHashMap<Object, Method>(mBinding.get(pAppEvent.getClass()));

        // iterate over all known targets until the event is consumed
        for (final Map.Entry<Object, Method> entry : targets.entrySet())
        {
            Object target          = entry.getKey();
            Method methodToCallBack= entry.getValue();
            try
            {
                Object[] args = {pAppEvent};

                // doit: call the method with the incoming AppEvent
                methodToCallBack.invoke(target, args);
            }
            catch (IllegalAccessException iae)
            {
                // this should never happen, because we checked it
                // during creation of the BindEvent
                IllegalStateException e = new IllegalStateException(iae.toString());
                e.initCause(iae);
                throw e;

            }
            catch (InvocationTargetException ite)
            {
                // map the cause of e to a IllegalArgumentException, if it is
                // not a RuntimeException or an Error
                Throwable cause = ite.getTargetException();

                if (cause instanceof RuntimeException)
                {
                    throw (RuntimeException) cause;
                }
                else if (cause instanceof Error)
                {
                    throw (Error) cause;
                }

                IllegalArgumentException e = new IllegalArgumentException(
                        cause != null ? cause.getMessage() : ite.getMessage());
                e.initCause(cause);
                throw e;
            }

            // check if the receiver consumed the event
            if (pAppEvent.isConsumed())
            {
                break;
            }
        }
    }
}
