package net.sf.cuf.fw2;

import net.sf.cuf.appevent.AppEvent;
import net.sf.cuf.fw.Dc;

import java.util.Map;
import java.util.Collections;
import java.util.HashMap;

/**
  * A OpenDialogEvent signals that a dialog should be opend.<br/>
 * For dialogs following the SimpleLifeCycle interface this will
 * trigger a call to the doActivate() method.
 * The dialog is referenced either by a Dc object or a symbolic name.
 * If the to be opened dialog is modal, the open event contains also the
 * information of the doPassivate() call.
 */
public class OpenDialogEvent extends AppEvent
{
    /** either a String or a Dc, never null */
    private Object                  mDialog;
    /** our argmuents, never null */
    private Map<String, Object>     mDialogArguments;
    /** our results, never null */
    private Map<String, Object>     mDialogResults;

    /**
     * Create a new open dialog event.
     * @param pSource the object requesting the open, must not be null
     * @param pDialogId the dialog id of the to be opend dialog
     */
    public OpenDialogEvent(final Object pSource, final String pDialogId)
    {
        this(pSource, pDialogId, null);
    }

    /**
     * Create a new open dialog event.
     * @param pSource the object requesting the open, must not be null
     * @param pDialogId the dialog id of the to be opend dialog
     * @param pDialogArguments arguments for the doPassivate() call, may be null
     */
    public OpenDialogEvent(final Object pSource, final String pDialogId, final Map<String, Object> pDialogArguments)
    {
        this(pSource, (Object)pDialogId, pDialogArguments);
    }

    /**
     * Create a new open dialog event.
     * @param pSource the object requesting the open, must not be null
     * @param pDialog the dialog object that should be opend, must not be null
     */
    public OpenDialogEvent(final Object pSource, final Dc pDialog)
    {
        this(pSource, pDialog, null);
    }

    /**
     * Create a new open dialog event.
     * @param pSource the object requesting the open, must not be null
     * @param pDialog the dialog object that should be opend, must not be null
     * @param pDialogArguments arguments for the doActivate() call, may be null
     */
    public OpenDialogEvent(final Object pSource, final Dc pDialog, final Map<String, Object> pDialogArguments)
    {
        this(pSource, (Object)pDialog, pDialogArguments);
    }

    /**
     * Create a new open dialog event.
     * @param pSource the object requesting the open, must not be null
     * @param pDialog the dialog object or dialog id that should be opend, must not be null
     * @param pDialogArguments arguments for the doActivate() call, may be null
     */
    private OpenDialogEvent(final Object pSource, final Object pDialog, final Map<String, Object> pDialogArguments)
    {
        super(pSource);

        if (pDialog==null)
            throw new IllegalArgumentException("dialog (name) must not be null");
        mDialog= pDialog;

        if (pDialogArguments==null)
            mDialogArguments= new HashMap<String, Object>();
        else
            mDialogArguments= pDialogArguments;

        mDialogResults= Collections.emptyMap();
    }

    /**
     * Checks if the open event contains a dialog id and not a Dc object.
     * @return true if the event was generated with a dialog id
     */
    public boolean hasDialogId()
    {
        return (mDialog instanceof String);
    }

    /**
     * The dialog id of this open event.
     * @return dialog id, never null
     * @throws IllegalStateException if this event contains no dialog string
     */
    public String getDialogId()
    {
        if (!hasDialogId())
        {
            throw new IllegalStateException("This OpenDialogEvent was created with no String but the Dc "+mDialog);
        }
        return (String)mDialog;
    }

    /**
     * The dialog Dc of this open event.
     * @return Dc object, never null
     * @throws IllegalStateException if this event contains no Dc object
     */
    public Dc getDialog()
    {
        if (hasDialogId())
        {
            throw new IllegalStateException("This OpenDialogEvent was created with no Dc but the String "+mDialog);
        }
        return (Dc)mDialog;
    }

    /**
     * Returns arguments for the doPassivate() call.
     * @return arguments, never null
     */
    public Map<String, Object> getDialogArguments()
    {
        return mDialogArguments;
    }

    /**
     * Returns the result of a open, this is only useful for modal dialogs, non-modal
     * dialogs will always return an empty map.
     * @return results, never null
     */
    public Map<String, Object> getDialogResults()
    {
        return mDialogResults;
    }

    /**
     * Set the results of the open, this should be used called after opening
     * a model dialog.
     * @param pDialogResults results, must not be null
     */
    public void setDialogResults(final Map<String, Object> pDialogResults)
    {
        if (pDialogResults==null)
            throw new IllegalArgumentException("dialog results must not be null");

        mDialogResults = pDialogResults;
    }
}
