package net.sf.cuf.fw2;

import net.sf.cuf.appevent.AppEvent;

import java.util.Map;
import java.util.List;
import java.util.Collections;
import java.awt.Component;

/**
 * A MenuToolBaarEvent can be used by a DialogPc to change the
 * Menu/Toolbar. </br>
 * The key in the menu map is hierarchical name starting with a /. The name
 * of a menu is not the displayed name but the widget name. The value in the
 * menu map is either JMenuItem or a List of JMenuItem objects. A fine tuning of the
 * insertion point is possible with an additional map that contains
 * positive (starting from the front, +1= left most menu) or negative
 * (starting from the end, -1= right most menu) position indices for
 * the used menu keys.
 * <P>
 * Note that if you add a menu entry with the same logical menu name as
 * an existing menu, the add request will be ignored. Thus, if two 
 * dialogs share entries in a menu that is not created outside of those
 * dialogs but implicitly from within the dialogs, you must separate the
 * JMenu and the JMenuItem in the event. If you add a JMenu that already contains
 * the JMenuItem only one of the dialogs will "win".
 * Also note that deleting a shared menu is not (yet) clearly defined.
 */
@SuppressWarnings({"CollectionsFieldAccessReplaceableByMethodCall"})
public class MenuToolBarEvent extends AppEvent
{
    /** list of tool bar button components, never null */
    private List<Component>      mToolBarButtons;
    /** map, key= logical menu name seperated by /,
     * value= either a List of JMenuItem objects, or a JMenuItem object, never null */
    private Map<String,Object>   mMenus;
    /** map, key= logical menu name seperated by /,
     * value= an Integer, never null */
    private Map<String, Integer> mMenuInsertionIndeces;
    /** flag if we should add (false) or delete (true) the menu/toolbar stuff */
    private boolean              mDelete;

    /**
     * Create a non-consumed event with toolbar buttons and menus to add/delete.
     * @param pSource null or event source
     * @param pToolBarButtons list of tool bar buttons, must not be null
     * @param pMenus map of menus, must not be null
     * @param pMenuInsertionIndeces map of menu insertion indices, must not be null
     * @param pDelete flag, if true we delete the toolbar buttons/menus
     */
    public MenuToolBarEvent(final Object pSource,
                            final List<Component> pToolBarButtons,
                            final Map<String, Object> pMenus,
                            final Map<String, Integer> pMenuInsertionIndeces,
                            final boolean pDelete)
    {
        super(pSource);
        if (pToolBarButtons==null)
        {
            throw new IllegalArgumentException("toolbar action list must not be null");
        }
        if (pMenus==null)
        {
            throw new IllegalArgumentException("menu action map must not be null");
        }
        if (pMenuInsertionIndeces==null)
        {
            throw new IllegalArgumentException("menu insertion map must not be null");
        }
        mToolBarButtons      = pToolBarButtons;
        mMenus               = pMenus;
        mMenuInsertionIndeces= pMenuInsertionIndeces;
        mDelete              = pDelete;
    }

    /**
     * Create a non-consumed event with no menu/toolbar to add.
     * @param pSource null or event source
     */
    public MenuToolBarEvent(final Object pSource)
    {
        this(pSource, Collections.EMPTY_LIST);
    }

    /**
     * Create a non-consumed event with toolbar buttons to add.
     * @param pSource null or event source
     * @param pToolBarButtons list of tool bar buttons, must not be null
     */
    public MenuToolBarEvent(final Object pSource, final List<Component> pToolBarButtons)
    {
        this(pSource, pToolBarButtons, false);
    }

    /**
     * Create a non-consumed event with toolbar buttons to add/delete.
     * @param pSource null or event source
     * @param pToolBarButtons list of tool bar buttons, must not be null
     * @param pDelete flag, if true we delete the buttons
     */
    public MenuToolBarEvent(final Object pSource, final List<Component> pToolBarButtons, final boolean pDelete)
    {
        this(pSource, pToolBarButtons, Collections.EMPTY_MAP, Collections.EMPTY_MAP, pDelete);
    }

    /**
     * Create a non-consumed event with menus to add.
     * @param pSource null or event source
     * @param pMenus map of menus, must not be null
     */
    public MenuToolBarEvent(final Object pSource, final Map<String, Object> pMenus)
    {
        this(pSource, Collections.EMPTY_LIST, pMenus, Collections.EMPTY_MAP, false);
    }

    /**
     * Create a non-consumed event with menus to add.
     * @param pSource null or event source
     * @param pMenus map of menus, must not be null
     * @param pMenuInsertionIndeces map of menu insertion indeces, must not be null
     */
    public MenuToolBarEvent(final Object pSource, final Map<String, Object> pMenus, final Map<String, Integer> pMenuInsertionIndeces)
    {
        this(pSource, Collections.EMPTY_LIST, pMenus, pMenuInsertionIndeces, false);
    }

    /**
     * Create a non-consumed event with menus to add.
     * @param pSource null or event source
     * @param pMenus map of menus, must not be null
     * @param pMenuInsertionIndeces map of menu insertion indeces, must not be null
     * @param pDelete flag, if true we delete the menus
     */
    public MenuToolBarEvent(final Object pSource, final Map<String, Object> pMenus, final Map<String, Integer> pMenuInsertionIndeces,
                            final boolean pDelete)
    {
        this(pSource, Collections.EMPTY_LIST, pMenus, pMenuInsertionIndeces, pDelete);
    }

    /**
     * Create a non-consumed event with menus to add/delete.
     * @param pSource null or event source
     * @param pMenus map of menus, must not be null
     * @param pDelete flag, if true we delete the menus
     */
    public MenuToolBarEvent(final Object pSource, final Map<String, Object> pMenus, final boolean pDelete)
    {
        this(pSource, Collections.EMPTY_LIST, pMenus, Collections.EMPTY_MAP, pDelete);
    }

    /**
     * Create a non-consumed event with toolbar buttons and menus to add.
     * @param pSource null or event source
     * @param pToolBarButtons list of tool bar buttons, must not be null
     * @param pMenus map of menus, must not be null
     */
    public MenuToolBarEvent(final Object pSource, final List<Component> pToolBarButtons, final Map<String, Object> pMenus)
    {
        this(pSource, pToolBarButtons, pMenus, Collections.EMPTY_MAP, false);
    }

    /**
     * Create a non-consumed event with toolbar buttons and menus to add/delete.
     * @param pSource null or event source
     * @param pToolBarButtons list of tool bar buttons, must not be null
     * @param pMenus map of menus, must not be null
     * @param pDelete flag, if true we delete the toolbar buttons/menus
     */
    public MenuToolBarEvent(final Object pSource, final List<Component> pToolBarButtons, final Map<String, Object> pMenus, final boolean pDelete)
    {
        this(pSource, pToolBarButtons, pMenus, Collections.EMPTY_MAP, pDelete);
    }

    /**
     * Create a non-consumed event with toolbar buttons and menus to add.
     * @param pSource null or event source
     * @param pToolBarButtons list of tool bar buttons, must not be null
     * @param pMenus map of menus, must not be null
     * @param pMenuInsertionIndeces map of menu insertion indeces, must not be null
     */
    public MenuToolBarEvent(final Object pSource, final List<Component> pToolBarButtons,
                            final Map<String, Object> pMenus, final Map<String, Integer> pMenuInsertionIndeces)
    {
        this(pSource, pToolBarButtons, pMenus, pMenuInsertionIndeces, false);
    }

    /**
     * Return the toolbar buttons.
     * @return list of toolbar buttons, never null
     */
    public List<Component> getToolBarButtons()
    {
        return mToolBarButtons;
    }

    /**
     * Returns the menu map.
     * @return map of logical position/list of menus or menu, never null
     */
    public Map<String, Object> getMenus()
    {
        return mMenus;
    }

    /**
     * Returns the menu insertion map.
     * @return map of logical position/insertion point, never null
     */
    public Map<String, Integer> getMenuInsertionIndeces()
    {
        return mMenuInsertionIndeces;
    }

    /**
     * Check if this event adds or removes stuff.
     * @return true if the menu/toolbar entries should be removed.
     */
    public boolean isDelete()
    {
        return mDelete;
    }
}
