package net.sf.cuf.appevent;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

/**
 * A BindEvent object is used to bind AppEvent classes to (any number of)
 * event receivers.
 * @see UnbindEvent
 */
public class BindEvent extends AppEvent
{
    /** callback method */
    private Method                      mMethodToCallBack;
    /** trigger class */
    private Class<? extends AppEvent>   mAppEventToBindTo;
    /** -1 or number of outstanding hops */
    private int                         mHopCount;

    /**
     * The constructor creates a BindEvent with the handed values and an
     * unbound hop-count.
     *
     * @param pSource           sender object, will receive the callback
     * @param pMethodToCallBack method name of pSource
     * @param pAppEventToBindTo trigger class, must be a subclass of AppEvent
     * @throws IllegalArgumentException if one of the arguments is null
     *         or the method is not known/not public.
     */
    public BindEvent(final Object pSource, final String pMethodToCallBack,
                     final Class<? extends AppEvent> pAppEventToBindTo)
    {
        this(pSource, pMethodToCallBack, pAppEventToBindTo, 0);
    }

    /**
     * The constructor creates a BindEvent with the handed values.
     *
     * @param pSource           sender object, will receive the callback
     * @param pMethodToCallBack method name of pSource
     * @param pAppEventToBindTo trigger class, must be a subclass of AppEvent
     * @param pHopCount         number of hops this event gets forwarded
     *                          before it consumes itself. If pHopCount is 0,
     *                          there is no hop limit.
     * @throws IllegalArgumentException if one of the arguments is null,
     *         pHopCount is negative or the method is not known/not public.
     */
    public BindEvent(final Object pSource, final String pMethodToCallBack,
                     final Class<? extends AppEvent> pAppEventToBindTo, final int pHopCount)
    {
        super(pSource);
        if (pSource==null)
            throw new IllegalArgumentException("pSource must not be null");
        if (pMethodToCallBack==null)
            throw new IllegalArgumentException("pMethodToCallBack must not be null");
        if (pAppEventToBindTo==null)
            throw new IllegalArgumentException("pAppEventToBindTo must not be null");
        if (!AppEvent.class.isAssignableFrom(pAppEventToBindTo))
            throw new IllegalArgumentException(pAppEventToBindTo.getName() + " is not derived from AppEvent");
        if (pHopCount<0)
            throw new IllegalArgumentException("pHopCount must be >= 0");
        if (!Modifier.isPublic(pSource.getClass().getModifiers()))
            throw new IllegalArgumentException("pSource must be an object from a public class");

        // create Method Object
        try
        {
            mMethodToCallBack= pSource.getClass().getMethod(pMethodToCallBack, pAppEventToBindTo);
        }
        catch (NoSuchMethodException e)
        {
            throw new IllegalArgumentException(e.getMessage());
        }
        if (!Modifier.isPublic(mMethodToCallBack.getModifiers()))
        {
            throw new IllegalArgumentException(pMethodToCallBack+" isn't a public method");
        }

        // remember other stuff
        mAppEventToBindTo= pAppEventToBindTo;
        mHopCount        = pHopCount;
        if (mHopCount==0)
        {
            mHopCount= -1;
        }
    }


    /**
     * During the propagation in the chain of responsibilty, at each hop
     * the forward() method should be called according to the AppEventSupport
     * "contract".
     * We use this to consume ourselfs if we had a hop count.
     */
    public void forward()
    {
        if (mHopCount<0)
        {
            return;
        }

        if (mHopCount < 1)
        {
            consume();
        }
        mHopCount--;
    }


    /**
     * Returns the target for the binding.
     *
     * @return target object
     */
    public Object getTarget()
    {
        return source;
    }

    /**
     * Returns the method object for the target object.
     *
     * @return method of the target object
     */
    public Method getMethod()
    {
        return mMethodToCallBack;
    }

    /**
     * Returns the AppEvent derived class that the target is interested in.
     *
     * @return trigger class
     */
    public Class<? extends AppEvent> getTriggerClass()
    {
        return mAppEventToBindTo;
    }
}
