package net.sf.cuf.fw2;

import net.sf.cuf.fw.Application;
import net.sf.cuf.fw.Dc;
import net.sf.cuf.appevent.AppEvent;
import net.sf.cuf.appevent.AppEventSupport;
import net.sf.cuf.appevent.AppEventUtil;

import java.util.Map;
import java.util.HashMap;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.ArrayList;

/**
 * This helper class manages the opening and closing of dialogs.
 * If a dialog is opened by id, it is also initialized.
 * If the dialog implements the SimpleLifeCycle, the doActivate()
 * and doPassivate() methods are called.
 * The helper also supports modal dialogs and holds a list of
 * all open dialogs. The DialogManager closes only dialogs that
 * where opened by him.
 * To allow further customizing of the open/close process, the
 * DialogManger provices callbacks after opening/closing a dialog.
 */
public class DialogManager implements Disposable
{
    /** never null, key= dialog id, value= DialogDescription object */
    private Map<String, DialogDescription>  mKnownDialogs;
    /** the application for newly created dialogs, never null */
    private Application                     mApp;
    /** never null, key= Dc object, value OpenDialogEvent */
    private Map<Dc,OpenDialogEvent>         mInActivateDialogs;
    /** list of all dialogs that implement the SimpleLifeCycle interface
      * that we opened and that are active */
    private List<Dc>                        mActiveDialogs;
    /** the current call back handler, may be null */
    private DialogCallback                  mCallback;

    /**
     * Creates a new dialog manager.
     * @param pApp the application for the init() call if a dialog is created
     */
    DialogManager(final Application pApp)
    {
        if (pApp==null)
            throw new IllegalArgumentException("Application must not be null");
        mKnownDialogs     = Collections.emptyMap();
        mApp              = pApp;
        mInActivateDialogs= new HashMap<Dc, OpenDialogEvent>();
        mActiveDialogs    = new ArrayList<Dc>();
    }

    /**
     * Sets the known dialogs of this DialogManager.
     * @param pKnownDialogs a map of all known dialogs, must not be null;
     *                      key= dialog id as string,
     *                      value= dialog description object
     */
    public void setKnownDialogs(final Map<String, DialogDescription> pKnownDialogs)
    {
        if (pKnownDialogs==null)
            throw new IllegalArgumentException("know dialog map must not be null");

        // check content
        Iterator<Map.Entry<String, DialogDescription>> i= pKnownDialogs.entrySet().iterator();
        try
        {
            while (i.hasNext())
            {
                Map.Entry<String, DialogDescription>         entry            = i.next();
                String            dialogID         = entry.getKey();
                DialogDescription dialogDescription= entry.getValue();
                if ((dialogID==null) || (dialogDescription==null))
                    throw new IllegalArgumentException("know dialog map must not contain null values");
            }
        }
        catch (ClassCastException e)
        {
            IllegalArgumentException iae= new IllegalArgumentException("invalid dialog map");
            iae.initCause(e);
            throw iae;
        }

        // copy content
        mKnownDialogs = new HashMap<String, DialogDescription>(pKnownDialogs);
    }

    /**
     * Set the callback target for dialog callbacks.
     * @param pCallback the callback object, may be null
     */
    public void setCallback(final DialogCallback pCallback)
    {
        mCallback= pCallback;
    }

    /**
     * Returns the last dialog that was opened by this DialogManager or null
     * if there is currently no active dialog.
     * @return null or a dialog object
     */
    public Dc getActiveDialog()
    {
        int size= mActiveDialogs.size();
        if (size>0)
            return mActiveDialogs.get(size-1);
        else
            return null;
    }

    /**
     * Returns a list of active dialog objects.
     * @return a copy of the active dialog list, never null
     */
    public List<Dc> getActiveDialogs()
    {
        return new ArrayList<Dc>(mActiveDialogs);
    }

    /**
     * Open a new dialog. This method doesn't return immediately if the opend
     * dialog is modal, in this case the method returns after the modal dialog
     * was closed.
     * @param pEvent the event describing the to be opened dialog
     */
    public void openDialog(final OpenDialogEvent pEvent)
    {
        openDialog(pEvent, null);
    }

    /**
     * Open a new dialog. This method doesn't return immediately if the opend
     * dialog is modal, in this case the method returns after the modal dialog
     * was closed.
     * @param pEvent the event describing the to be opened dialog
     * @param pParent null or the parent for the init() call if a dialog is created by dialog id
     */
    public void openDialog(final OpenDialogEvent pEvent, final Dc pParent)
    {
        Map<String, Object> args= new HashMap<String, Object>();
        args.put(Application.APPLICATION_KEY, mApp);
        args.putAll(pEvent.getDialogArguments());

        Dc dc;
        if (pEvent.hasDialogId())
        {
            // create dialog from scratch
            String            dialogId   = pEvent.getDialogId();
            DialogDescription description= mKnownDialogs.get(dialogId);
            if (description==null)
            {
                // if we don't know the dialog, we don't handle it
                return;
            }

            // check if we can re-use the Dc
            dc= description.getDc();
            if (dc==null)
            {
                // no, let's create a new one
                Class<? extends Dc>  c= description.getDialogClass();
                try
                {
                    dc = c.newInstance();
                    dc.init(pParent, args);
                }
                catch (Exception e)
                {
                    // should never happen
                    throw new RuntimeException(e);
                }
            }
        }
        else
        {
            // use existing dialog
            dc= pEvent.getDialog();
        }

        // activate the dialog
        if (dc instanceof SimpleLifeCycle)
        {
            // a modal dialog will block in doActivate(), and closeDialog() will
            // fill the result in the open event
            mInActivateDialogs.put(dc, pEvent);
            mActiveDialogs.add(dc);
            try
            {
                ((SimpleLifeCycle)dc).doActivate(args);
            }
            finally
            {
                mInActivateDialogs.remove(dc);
            }
        }

        // calback
        if (mCallback!=null)
        {
            mCallback.dialogOpened(dc, args);
        }

        // we opened the dialog, so we consume the event
        pEvent.consume();
    }

    /**
     * Closes an open dialog. If the dialog implements SimpleLifeCycle, it will
     * be checked first if the dialog can be passivated. We consume the event
     * only if we know the dialog (=handled the OpenDialogEvent for it).
     * @param pEvent the event describing the to be closed dialog
     */
    public void closeDialog(final CloseDialogEvent pEvent)
    {
        Dc dc= pEvent.getDialog();

        // we only close dialogs we opened
        if (!mActiveDialogs.contains(dc))
            return;

        Map<String, Object> args= new HashMap<String, Object>();
        boolean closed= true;

        // passivation is only usefull for dialogs that implement SimpleLifeCycle
        if (dc instanceof SimpleLifeCycle)
        {
            // check if we can close at all
            SimpleLifeCycle lifeCycle= (SimpleLifeCycle)dc;
            if (lifeCycle.canPassivate())
            {
                // the result is what ever doPassivate() + dispose() will put in args
                lifeCycle.doPassivate(args);
                if (pEvent.isDispose() && (dc instanceof Disposable))
                {
                    //noinspection CastConflictsWithInstanceof
                    ((Disposable)dc).dispose(args);
                }
                pEvent.setResult(args);

                // check if this closeDialog() closes a model dialog that
                // currently blocks in openDialog()
                if (mInActivateDialogs.containsKey(dc))
                {
                    OpenDialogEvent openEvent= mInActivateDialogs.get(dc);
                    openEvent.setDialogResults(args);
                }
            }
            else
            {
                closed= false;
            }
        }

        // we no longer manage this dialog
        if (closed)
        {
            mActiveDialogs.remove(dc);
        }

        // update event status
        pEvent.setWasClosed(closed);

        // calback
        if (mCallback!=null)
        {
            mCallback.dialogClosed(dc, closed, args);
        }

        // we closed or tried to close the dialog, so we consume the event
        pEvent.consume();
     }

    /**
     * Helper method that checks if all active dialogs can be passivated.
     * @return true if all dialogs can be passivated
     */
    public boolean canPassivate()
    {
        boolean canPassivate = true;

        // we can passivate if all active dialogs can passivate
        for (int i = 0, n = mActiveDialogs.size(); i < n; i++)
        {
            Object o= mActiveDialogs.get(i);
            if (o instanceof SimpleLifeCycle)
            {
                canPassivate= canPassivate && ((SimpleLifeCycle)o).canPassivate();
            }
        }

        return canPassivate;
    }

    /**
     * Called from the peer that coordinates our life cycle to tell us to
     * cleanup all resources. We forward this to all known (sub-)dialogs
     * and clear the known dialogs map.
     * @param pArgs arguments, key is a String, value is any suitable objekt for the key
     */
    public void dispose(final Map<String, ? super Object> pArgs)
    {
        for (final Object o : mKnownDialogs.values())
        {
            DialogDescription dialogDescription = (DialogDescription) o;
            Object dc = dialogDescription.getDc();
            if (dc instanceof Disposable)
            {
                ((Disposable) dc).dispose(pArgs);
            }
        }
        mKnownDialogs.clear();
        mActiveDialogs.clear();
    }

    /**
     * Filter for an AppEvent hierarchy: if a OpenDialogEvent or an CloseDialogEvent is posted,
     * we try to handle it. If we can't open or close a dialog, the event is not consumed and
     * routed further up in the hierarchy.
     * @param pDc the parent of the new dialog, may be null
     * @param pAppEventSupport the (logical) parent in the AppEvent chain of responsability
     * @param pAppEvent event that should be routed/processes, must not be null
     * @throws IllegalArgumentException if pAppEvent is null
     */
    public void postAppEvent(final Dc pDc, final AppEventSupport pAppEventSupport, final AppEvent pAppEvent)
    {
        if (pAppEvent instanceof OpenDialogEvent)
        {
            OpenDialogEvent openDialogEvent= (OpenDialogEvent)pAppEvent;
            openDialog(openDialogEvent, pDc);
        }
        else if (pAppEvent instanceof CloseDialogEvent)
        {
            CloseDialogEvent closeDialogEvent= (CloseDialogEvent)pAppEvent;
            closeDialog(closeDialogEvent);
        }

        pAppEvent.forward();
        if (!pAppEvent.isConsumed())
        {
            AppEventUtil.postAppEvent(pAppEventSupport, pAppEvent);
        }
    }

    /**
     * Callback interface for the opening/closing of a dialog.
     */
    public interface DialogCallback
    {
        /**
         * Called after a dialog was opened.
         * @param pDc the dialog
         * @param pArgs the arguments
         */
        void dialogOpened(Dc pDc, Map<String, Object> pArgs);

        /**
         * Called after a dialog was closed.
         * @param pDc the dialog
         * @param pWasClosed flag if the dialog could be closed
         * @param pArgs the arguments
         */
        void dialogClosed(Dc pDc, boolean pWasClosed, Map<String, Object> pArgs);
    }
}
