package net.sf.cuf.xfer;

import java.lang.reflect.InvocationTargetException;

/**
 * The dispatcher class provides methods to produce and dispatch
 * a response object from a request object. Because this implementation
 * has no state, it may be used as a static instance for dispatching.
 */
public abstract class AbstractDispatcher implements Dispatch
{
    /**
     * Create a (stateless) dispatcher.
     */
    public AbstractDispatcher()
    {
    }

    /**
     * Generate a response and and dispatch it in a new (separate) thread.
     * @param pRequest the request
     */
    public void asyncDispatch(final Request pRequest)
    {
        Thread load= new Thread() {
            public void run()
            {
                syncDispatch(pRequest);
            }
        };
        load.start();
    }

    /**
     * Generate a response in a new (separate) thread and dispatch the
     * response in the EDT (event dispatch thread).
     * @param pRequest the request
     */
    public void asyncDispatchInEDT(final Request pRequest)
    {
        Thread load= new Thread(new DispatchRunnable(pRequest));
        load.start();
    }

    /**
     * Generate a response (includes delegate handling) but doesn't dispatch the response.
     * @param pRequest the request
     * @return the response
     */
    public Response syncExecute(final Request pRequest)
    {
        RequestDelegate delegate= pRequest.getDelegate();
        Response        response;
        if (delegate!=null)
        {
            response= delegate.execute(pRequest);
        }
        else
        {
            response= pRequest.execute();
        }
        return response;
    }

    /**
     * Generate a response and and dispatch it in the current thread.
     * @param pRequest the request
     */
    public void syncDispatch(final Request pRequest)
    {
        DispatchTarget target  = pRequest.getDispatchTarget();
        Response       response= syncExecute(pRequest);
        syncDispatch(target, response);
    }

    /**
     * Small helper to do common work.
     * @param pTarget the target for dispatching
     * @param pResponse the response to dispatch
     */
    private void syncDispatch(final DispatchTarget pTarget, final Response pResponse)
    {
        try
        {
            Object[] args= {pResponse};

            // doit: call the method with the generated response
            pTarget.getTargetMethod().invoke(pTarget.getTargetObject(), args);
        }
        catch (IllegalAccessException e)
        {
            // map e to a IllegalArgumentException
            throw new IllegalArgumentException(e.getMessage());
        }
        catch (InvocationTargetException e)
        {
            // map the cause of e to a IllegalArgumentException, if it is
            // not a RuntimeException or an Error
            Throwable cause= e.getTargetException();

            if (cause instanceof RuntimeException)
            {
                throw (RuntimeException) cause;
            }
            if (cause instanceof Error)
            {
                throw (Error) cause;
            }

            throw new IllegalArgumentException(cause!=null ? cause.getMessage() : e.getMessage());
        }
    }

    /**
     * The callback method to switch to the "right" UI thread.
     * @param pRunnable the runnable containing the result, must not be null
     */
    protected abstract void doDispatch(final Runnable pRunnable);

    /**
     * The runnable handling the dispatch.
     */
    protected class DispatchRunnable implements Runnable
    {
        /** the target, never null */
        private final Request mRequest;
        /** the target, never null */
        private final DispatchTarget mTarget;
        /** the response, initially null */
        private Response mResponse;

        /**
         * Create a new runnable for dispatching the request
         * @param pRequest the request, must not be null, must have a target
         */
        public DispatchRunnable(final Request pRequest)
        {
            mRequest= pRequest;
            mTarget = pRequest.getDispatchTarget();
            if (mTarget == null)
            {
                throw new IllegalArgumentException("target must not be null");
            }
        }

        public void run()
        {
            if (mResponse != null)
            {
                // in EDT
                syncDispatch(mTarget, mResponse);
            }
            else
            {
                // in new thread
                mResponse = syncExecute(mRequest);
                if (mResponse != null)
                {
                    doDispatch(this);
                }
            }
        }

    }
}
