package net.sf.cuf.fw2;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import net.sf.cuf.fw.Application;
import net.sf.cuf.fw.Dc;

/**
 * Small helper class that stores the meta data about a dialog. It is
 * used to read all known dialogs from a properties file. The format
 * of the properties file is
 * <code>
 * dialog.id.&lt;running number&gt;=&lt;ID of the dialog&gt;
 * dialog.class.&lt;running number&gt;=&lt;class name of the dialog&gt;
 * dialog.name.&lt;running number&gt;=&lt;name of the dialog as shown to the user&gt;
 * dialog.icon.&lt;running number&gt;=&lt;icon of the dialog as shown to the user&gt;
 * dialog.&lt;myKey&gt;.&lt;running number&gt;=&lt;value for myKey&gt;
 * </code>
 * The dialog.icon and dialog.name properties are optional, as well as the individual
 * key value definitions. &lt;myKey&gt; may contain '.', as long as there is the identifying running
 * number at the end.
 * The individual key value definitions are stored in 
 * {@link #mParameters}.
 * <P>
 * The class optionally stores a dialog instance of a dialog, this is
 * usefull if the same dialog should be re-used.
 */
public class DialogDescription
{
    /**
     * the token with that the keys in the dialog configuration file must start
     */
    private static final String DIALOG_PREFIX = "dialog";
    
    /** 
     * the tokes for the explicit parameters 
     * used in the configuration file which are mapped to specific attributes
     */
    private static final String[] EXPLICIT_PARAMS =
    {
        "id",
        "class",
        "name",
        "icon"
    };

    /** the ID of the dialog, never null after the descriptin is complete */
    private String              mDialogId;
    /** the class name of the dialog, never null after the descriptin is complete */
    private Class<? extends Dc> mDialogClass;
    /** the name of the dialog as shown to the user, may be null */
    private String              mDialogName;
    /** the icon of the dialog as shown to the user, may be null */
    private String              mDialogIcon;
    /** the parameters, a map of {@link String} to {@link Object}, may be null. */
    private Map<String, Object> mParameters;
    /** a dialog instance, may be null */
    private Dc                  mDc;

    /**
     * Load known dialogs from a properties file.
     * @param pFileName the name of the properties file
     * @return a Map of DialogDescription objects, never null, key= DialogID, value= DialogDescription object
     */
    public static Map<String, DialogDescription> loadKnownDialogs(final String pFileName)
    {
        InputStream in= Thread.currentThread().getContextClassLoader().getResourceAsStream(pFileName);
        if (in==null)
        {
            throw new IllegalArgumentException("No input resource found for name "+pFileName);
        }

        Properties  props= new Properties();
        try
        {
            props.load(in);
        }
        catch (IOException e)
        {
            IllegalArgumentException iae= new IllegalArgumentException(
                    "Could not load input resource for name "+pFileName);
            iae.initCause(e);
            throw iae;
        }

        // check for each property key if it is "valid" (one of our tokens + any positive number)
        // create a DialogDescription (if needed) and store it on the right spot in dialogs
        List<DialogDescription> dialogs= new ArrayList<DialogDescription>();
        for (final Map.Entry<Object, Object> entry : props.entrySet())
        {
            String key  = (String)entry.getKey();
            String value= (String)entry.getValue();

            // The following code may seem weird, but it has its reason:
            // backwards compatibility. We now support custom parameters
            // and they may have dots in them. If we ever get the chance
            // to make a complete redesign, a format of dialog.<i>.name etc.
            // would be a lot better than dialog.name.<i>
            int firstDot = key.indexOf('.');
            int lastDot = key.lastIndexOf('.');
            if (firstDot>=0 && lastDot>=0 && firstDot<lastDot)
            {
                // We have at least 2 dots which means we can try to find
                // our tokens.
                //noinspection EmptyCatchBlock
                try
                {
                    // The last part is the dialog index:
                    int index = Integer.parseInt( key.substring( lastDot+1));
                    if (index<0)
                    {
                        break;
                    }
                    if (index>=dialogs.size())
                    {
                        int gap= index-dialogs.size();
                        for (int i= gap; i>=0; i--)
                        {
                            dialogs.add(new DialogDescription());
                        }
                    }
                    String token = key.substring( firstDot+1, lastDot);
                    if (DIALOG_PREFIX.equals( key.substring( 0, firstDot)))
                    {
                        DialogDescription dialogDescription= dialogs.get(index);

                        if (EXPLICIT_PARAMS[0].equals(token))
                        {
                            dialogDescription.setDialogId(value);
                        }
                        else if (EXPLICIT_PARAMS[1].equals(token))
                        {
                            dialogDescription.setDialogClassName(value);
                        }
                        else if (EXPLICIT_PARAMS[2].equals(token))
                        {
                            dialogDescription.setDialogName(value);
                        }
                        else if (EXPLICIT_PARAMS[3].equals(token))
                        {
                            dialogDescription.setDialogIcon(value);
                        }
                        else
                        {
                            // custom parameter
                            dialogDescription.addParameter(token, value);
                        }
                        
                    }
                }
                catch (NumberFormatException ignored)
                {
                }
            }
        }

        // remove garbage or "holes"
        for (Iterator<DialogDescription> i = dialogs.iterator(); i.hasNext();)
        {
            DialogDescription dialogDescription= i.next();
            if ((dialogDescription==null)        ||
                (!dialogDescription.isComplete())  )
            {
                i.remove();
            }
        }

        Map<String, DialogDescription> knownDialogs= new LinkedHashMap<String, DialogDescription>(dialogs.size());
        for (int i = 0, n = dialogs.size(); i < n; i++)
        {
            DialogDescription dialogDescription = dialogs.get(i);
            knownDialogs.put(dialogDescription.getDialogId(), dialogDescription);
        }
        return knownDialogs;
    }

    /**
     * Initialize the Dc's for a map of dialog descriptions.
     * <P>
     * The custom parameters from the {@link DialogDescription}
     * are added to those provided by the <code>pArgs</code>:
     * <BR>
     * First, the {@link DialogDescription#getParameters()} is used,
     * <BR>
     * then the parameters from <code>pArgs</code> are added,
     * <BR>
     * and finally the {@link Application} is added with the key {@link Application#APPLICATION_KEY}.
     * <BR>
     * This means that <code>pArgs</code> may overwrite the custom parameters
     * in the dialog description but may not overwrite the application parameter.
     * 
     * @param pKnownDialogs a Map of DialogDescription objects, never null,
     *                     key= DialogID, value= DialogDescription object
     * @param pApp    the application, must not be null
     * @param pParent the parent Dc for the init call, may be null
     * @param pArgs   arguments for the init call, may be null
     */
    public static void initDialogs(final Map<String, DialogDescription> pKnownDialogs, final Application pApp, final Dc pParent, final Map<String, ? super Object> pArgs)
    {
        if (pKnownDialogs==null)
            throw new IllegalArgumentException("know dialogs must not be null");
        if (pApp==null)
            throw new IllegalArgumentException("application must not be null");

        for (final Object o : pKnownDialogs.values())
        {
            DialogDescription description = (DialogDescription) o;

            // check if we can re-use the Dc
            Dc dc = description.getDc();
            if (dc == null)
            {
                // no, let's create a new one
                Class<? extends Dc> c = description.getDialogClass();
                try
                {
                    dc = c.newInstance();
                    Map<String, Object> args = new HashMap<String, Object>();
                    if (description.getParameters() != null)
                    {
                        args.putAll(description.getParameters());
                    }
                    args.putAll(pArgs);
                    args.put(Application.APPLICATION_KEY, pApp);

                    dc.init(pParent, args);
                    description.setDc(dc);
                }
                catch (Exception e)
                {
                    // can happen, e.g. when the init method throws an exception
                    throw new RuntimeException("could not initialize dialog with id " + description.mDialogId + ": " + e, e);
                }
            }
        }
    }

    /**
     * Returns true if we have a id and class
     * @return true if we are complete
     */
    public boolean isComplete()
    {
        return (mDialogId   != null)  &&
               (mDialogClass!= null);
    }

    /**
     * Return the dialog id.
     * @return the dialog id, may be null
     */
    public String getDialogId()
    {
        return mDialogId;
    }

    /**
     * Set the dialog id.
     * @param pDialogId the dialog id, must not be null
     */
    public void setDialogId(final String pDialogId)
    {
        if (pDialogId==null)
            throw new IllegalArgumentException("dialog id must not be null");
        mDialogId = pDialogId;
    }

    /**
     * Retrun the dialog class.
     * @return the dialog class, may be null
     */
    public Class<? extends Dc> getDialogClass()
    {
        return mDialogClass;
    }

    /**
     * Set the dialog class
     * @param pDialogClass the dialog class, must not be null
     */
    public void setDialogClass(final Class<? extends Dc> pDialogClass)
    {
        if (!Dc.class.isAssignableFrom(pDialogClass))
            throw new IllegalArgumentException("not a Dc class but a "+pDialogClass);
        mDialogClass= pDialogClass;
    }

    /**
     * Set the dialog class name.
     * @param pDialogClassName the dialog class, must not be null and must be a Dc derivate
     */
    public void setDialogClassName(final String pDialogClassName)
    {
        Class<? extends Dc> dialogClass;
        try
        {
            dialogClass= (Class<? extends Dc>)Thread.currentThread().getContextClassLoader().loadClass(pDialogClassName);
        }
        catch (Exception ignored)
        {
            IllegalArgumentException iae= new IllegalArgumentException("could not load class "+pDialogClassName);
            iae.initCause(ignored);
            throw iae;
        }
        setDialogClass(dialogClass);
    }

    /**
     * Return the dialog name.
     * @return null or the dialog name
     */
    public String getDialogName()
    {
        return mDialogName;
    }

    /**
     * Set the dialog name.
     * @param pDialogName the dialog name, may be null
     */
    public void setDialogName(final String pDialogName)
    {
        mDialogName = pDialogName;
    }

    /**
     * Return the dialog icon name.
     * @return null or the dialog icon name.
     */
    public String getDialogIcon()
    {
        return mDialogIcon;
    }

    /**
     * Set the dialog icon name.
     * @param pDialogIcon the dialog icon name, may be null
     */
    public void setDialogIcon(final String pDialogIcon)
    {
        mDialogIcon = pDialogIcon;
    }

    /**
     * Add a paraemter to the dialog
     * @param pKey the parameter to add, must not be null
     * @param pValue the value, must not be null
     */
    public void addParameter(final String pKey, final Object pValue)
    {
        if (pKey==null)
            throw new IllegalArgumentException("dialog parameter key must not be null");
        if (pValue==null)
            throw new IllegalArgumentException("dialog parameter value must not be null");
        if (mParameters==null)
        {
            mParameters= new HashMap<String, Object>();
        }
        mParameters.put(pKey, pValue);
    }

    /**
     * Return the parameters or null if there are none.
     * @return null or our parameters
     */
    public Map<String, Object> getParameters()
    {
        return mParameters;
    }

    /**
     * Set the parameters.
     * @param pParameters the parameter map, may be nul
     */
    public void setParameters(final Map<String, Object> pParameters)
    {
        mParameters = pParameters;
    }

    /**
     * Return the dialog of this description
     * @return null or the dialog Dc
     */
    public Dc getDc()
    {
        return mDc;
    }

    /**
     * Set the dialog for this description
     * @param pDc the dialog, may be null
     */
    public void setDc(final Dc pDc)
    {
        mDc = pDc;
    }
}
