package net.sf.cuf.csvviewfx.data;

import javafx.collections.FXCollections;

import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * A TreeFilteredObservableList generates its data from recursivly parsing a
 * directory tree.
 * Its rows are all found files/directories, its columns are path,
 * name, size and last modification date of the file/directory.
 */
public class TreeFilteredObservableList extends AbstractFilteredObservableList
{
    /** our table headers. */
    private static final String[] HEADER= {"path",  "name", "size", "last modified"};

    /**
     * Creates a new TreeFilteredObservableList by parsing the directories starting at the
     * handed tree.
     * @param pRootNode the root node, must not be null
     * @throws IllegalArgumentException if pRootNode is null
     */
    public TreeFilteredObservableList(final File pRootNode)
    {
        super();
        if (pRootNode==null)
        {
            throw new IllegalArgumentException("root file node must not be null");
        }

        // our columns are fixed
        mRowHeaders= new HashMap<>(HEADER.length);
        for (int i = 0, n = HEADER.length; i < n; i++)
        {
            mRowHeaders.put(i, HEADER[i]);
        }

        // traverseTree adds all data to the list
        mAllRows= new ArrayList<>();
        traverseTree(pRootNode);

        mFilteredRows= FXCollections.observableArrayList(mAllRows);
    }

    /**
     * Small helper to traverse a tree.
     * @param pCurrentNode the current node
     */
    private void traverseTree(final File pCurrentNode)
    {
        if (!pCurrentNode.canRead() || !pCurrentNode.exists())
        {
            return;
        }

        Map<String, String> row= new HashMap<>(HEADER.length);
        row.put(HEADER[0], pCurrentNode.getPath());
        row.put(HEADER[1], pCurrentNode.getName());
        row.put(HEADER[2], Long.toString(pCurrentNode.length()));
        row.put(HEADER[3], new Date(pCurrentNode.lastModified()).toString());
        mAllRows.add(row);

        if (pCurrentNode.isDirectory())
        {
            File[] children= pCurrentNode.listFiles();
            if (children!=null)
            {
                for (final File child : children)
                {
                    traverseTree(child);
                }
            }
        }
    }
}
