package net.sf.cuf.csvviewfx.util;

import javafx.beans.property.BooleanProperty;
import javafx.event.EventHandler;
import javafx.scene.control.TableColumn;
import javafx.scene.control.TablePosition;
import javafx.scene.control.TableView;
import javafx.scene.input.Clipboard;
import javafx.scene.input.ClipboardContent;
import javafx.scene.input.KeyCombination;
import javafx.scene.input.KeyEvent;

import java.util.List;

/**
 * This small helper class is used to provide either a row/rows or a cell
 * copy to the clipboard of the selected row of a JavaFX TableView.
 */
public class TableCopyAction implements EventHandler<KeyEvent>
{
    /** trigger for our copy */
    private static final KeyCombination COPY_ACTION= KeyCombination.valueOf("Shortcut+C");

    /** if true, we copy the selected row or rows, otherwise we copy the selected cell */
    private BooleanProperty mCopyRow;

    /**
     * Create a new table copy action.
     * @param pCopyRow the property determining if copy a row or a cell, must not be null
     */
    public TableCopyAction(final BooleanProperty pCopyRow)
    {
        if (pCopyRow==null)
        {
            throw new IllegalArgumentException("copy row property must not be null");
        }
        mCopyRow= pCopyRow;
    }

    /** {@inheritDoc} */
    @Override
    public void handle(final KeyEvent pKeyEvent)
    {
        // first check if there a copy action going on
        if (!COPY_ACTION.match(pKeyEvent))
        {
            return;
        }
        // now check if there is anything selected at all
        Object source= pKeyEvent.getSource();
        if (!(source instanceof TableView))
        {
            return;
        }
        TableView<?> tableView= (TableView<?>) source;
        if (tableView.getSelectionModel().getSelectedIndex()<0)
        {
            return;
        }

        // build the text
        StringBuilder sb = new StringBuilder();
        if (mCopyRow.getValue())
        {
            // copy whole row(s)
            List<Integer> selectedRows= tableView.getSelectionModel().getSelectedIndices();
            int rowCount   = selectedRows.size();
            int columnCount= tableView.getColumns().size();
            int rows       = 0;
            int columns    = 0;
            for (final Integer selectedRow : selectedRows)
            {
                for (final TableColumn<?, ?> tableColumn : tableView.getColumns())
                {
                    Object cellData= tableColumn.getCellData(selectedRow);
                    if (cellData!=null)
                    {
                        sb.append(cellData);
                    }
                    if (++columns<columnCount) sb.append('\t');
                }
                if (++rows<rowCount) sb.append('\n');
            }
        }
        else
        {
            // copy cell only
            if (!tableView.getSelectionModel().getSelectedCells().isEmpty())
            {
                TablePosition<?, ?> pos= tableView.getSelectionModel().getSelectedCells().get(0);
                int r = pos.getRow();
                int c = pos.getColumn();
                Object cellData = tableView.getColumns().get(c).getCellData(r);
                if (cellData!=null)
                {
                    sb.append(cellData);
                }
            }
        }

        // export text to clipboard
        Clipboard clipboard = Clipboard.getSystemClipboard();
        ClipboardContent content = new ClipboardContent();
        content.putString(sb.toString());
        clipboard.setContent(content);

        // we consider the handling done
        pKeyEvent.consume();
    }
}
