package net.sf.cuf.csvviewfx.data;

import javafx.collections.FXCollections;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * A CsvFileredObservableList generates its data from parsing a CSV input stream.
 */
public class CsvFileredObservableList extends AbstractFilteredObservableList
{
    /** default separator */
    private static final String CSV_SEPARATOR= ",";
    /** helper for all empty strings found during split-up */
    private static final String   EMPTY            = "";
    /** static pattern for a quote */
    private static final Pattern  CSV_QUOTE_PATTERN= Pattern.compile("\"\"");

    /**
     * Create an empty table model (no rows, no columns).
     */
    public CsvFileredObservableList()
    {
        init();
    }

    /**
     * Create a new table model by parsing the input stream as CSV data.
     * @param pCSVData the input we read our data from, may be null
     * @param pFirstRowIsHeader true if the first row contains the header
     * @param pCSVSeparator "comma" separator, may be null (then "," is taken)
     * @param pEncoding the encoding pf pCSVData, may be null if pCSVData is null
     * @throws IOException if we have IO problems
     */
    public CsvFileredObservableList(final InputStream pCSVData, final boolean pFirstRowIsHeader, final String pCSVSeparator, final String pEncoding) throws IOException
    {
        if (pCSVData==null)
        {
            init();
        }
        else
        {
            init(new InputStreamReader(pCSVData, pEncoding), pFirstRowIsHeader, pCSVSeparator);
        }
    }

    /**
     * Create a new table model by parsing the CSV data from the input reader.
     * @param pCSVData the reader we read the CSV data from, may be null
     * @param pFirstRowIsHeader true if the first row contains the header
     * @param pCSVSeparator "comma" separator, may be null
     * @throws IOException if we have IO problems
     */
    public CsvFileredObservableList(final Reader pCSVData, final boolean pFirstRowIsHeader, final String pCSVSeparator) throws IOException
    {
        if (pCSVData==null)
        {
            init();
        }
        else
        {
            init(pCSVData, pFirstRowIsHeader, pCSVSeparator);
        }
    }

    /**
     * Small helper method doing common constructor stuff if we have no real data.
     */
    private void init()
    {
        mRowHeaders  = Collections.emptyMap();
        mAllRows     = Collections.emptyList();
        mFilteredRows= FXCollections.observableArrayList(mAllRows);
    }

    /**
     * Small helper method doing common constructor stuff.
     * @param pCSVData the reader we get the data from, must not be null
     * @param pFirstRowIsHeader flag if the first row should be treated as a header
     * @param pCSVSeparator the separator of the data, may be null
     * @throws IOException on I/O problems
     */
    private void init(final Reader pCSVData, final boolean pFirstRowIsHeader, final String pCSVSeparator) throws IOException
    {
        if (pCSVData==null)
        {
            throw new IllegalArgumentException("pCSVReader must not be null");
        }

        String csvSeparator;
        if (pCSVSeparator==null)
        {
            csvSeparator= CSV_SEPARATOR;
        }
        else
        {
            csvSeparator= pCSVSeparator;
        }

        List<List<String>> data         = new ArrayList<>();
        LineNumberReader   reader       = new LineNumberReader(pCSVData);
        int                maxColumns   = 0;
        String           csvMainString  =
                "  \\G(?:^|"+csvSeparator+")\n"+
                "  (?:\n"+
                "     # Either a double-quoted field...\n"+
                "     \" # field's opening quote\n"+
                "      (  (?> [^\"]*+ ) (?> \"\" [^\"]*+ )*+  )\n"+
                "     \" # field's closing quote\n"+
                "   # ... or ...\n"+
                "   |\n"+
                "     # ... some non-quote/non-comma text ...\n"+
                "     ( [^\""+csvSeparator+"]*+ )\n"+
                "  )\n";
        Pattern csvMainPattern= Pattern.compile(csvMainString, Pattern.COMMENTS);
        Matcher csvQuote      = CSV_QUOTE_PATTERN.matcher(EMPTY);
        Matcher csvMain       = csvMainPattern. matcher(EMPTY);

        String line= reader.readLine();
        while (line!=null)
        {
            // parse line
            List<String> row= new ArrayList<>();
            csvMain.reset(line);
            while ( csvMain.find() )
            {
                // we'll fill this with $1 or $2 . . .
                String field;
                String first = csvMain.group(2);
                if ( first != null )
                {
                    field = first;
                }
                else
                {
                    // if $1, must replace paired double-quotes with one double quote
                    csvQuote.reset(csvMain.group(1));
                    field = csvQuote.replaceAll("\"");
                }
                // we can now work with field, but we optimize empty fields first
                if (EMPTY.equals(field))
                {
                    field= EMPTY;
                }
                row.add(field);
            }
            // trim trailing empty stuff
            if (!row.isEmpty() && row.get(row.size()-1).equals(EMPTY))
            {
                row.remove(row.size()-1);
            }

            if (pFirstRowIsHeader && mRowHeaders==null)
            {
                // in the first line are the headers + first line
                mRowHeaders= new HashMap<>();
                for (int i = 0, n = row.size(); i < n; i++)
                {
                    String header = row.get(i);
                    mRowHeaders.put(i, header);
                }
            }
            else
            {
                data.add(row);
            }
            maxColumns= Math.max(maxColumns, row.size());

            // get next line
            line= reader.readLine();
        }

        // add or create unnamed columns
        if (pFirstRowIsHeader)
        {
            for (int i = mRowHeaders.size(); i < maxColumns; i++)
            {
                mRowHeaders.put(i, Integer.toString(i));
            }
        }
        else
        {
            mRowHeaders= new HashMap<>();
            for (int i = 0; i < maxColumns; i++)
            {
                mRowHeaders.put(i, Integer.toString(i));
            }
        }

        // re-fit data to our map based backing
        mAllRows= new ArrayList<>(data.size());
        for (final List<String>row : data)
        {
            Map<String, String> mapRow= new HashMap<>(mRowHeaders.size());
            for (int i = 0, n = row.size(); i < n; i++)
            {
                String entry = row.get(i);
                mapRow.put(mRowHeaders.get(i), entry);
            }
            mAllRows.add(mapRow);
        }
        mFilteredRows= FXCollections.observableArrayList(mAllRows);
    }

}
