package net.sf.cuf.csvviewfx.about;

import javafx.scene.layout.StackPane;
import net.sf.cuf.fw.Dc;
import net.sf.cuf.fw.Pc;
import javafx.animation.FadeTransition;
import javafx.animation.SequentialTransition;
import javafx.animation.SequentialTransitionBuilder;
import javafx.animation.Transition;
import javafx.event.ActionEvent;
import javafx.event.EventHandler;
import javafx.fxml.FXML;
import javafx.scene.Scene;
import javafx.scene.control.Label;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import javafx.scene.input.MouseEvent;
import javafx.stage.Modality;
import javafx.stage.Stage;
import javafx.stage.WindowEvent;
import javafx.util.Duration;
import net.sf.cuf.csvviewfx.AppData;
import net.sf.cuf.csvviewfx.Version;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Presentation controller for the about dialog.
 */
public class AboutPc implements Pc
{
    /** the fade in and out duriation */
    private static final Duration FADE_DURATION = Duration.seconds(3);

    /** our app icon, never null */
    @FXML
    private Image mAppIcon;
    /** our about image, never null */
    @FXML
    private Image mAboutImage;
    /** our image ;-), never null */
    @FXML
    private Image mAuthorImage;
    /** our about label, never null */
    @FXML
    private Label mAboutLabel;
    /** the view containing our about logo, never null */
    @FXML
    private ImageView mAboutLogo;
    /** our root container, never null */
    @FXML
    public StackPane mRootContainer;
    /** the version label, never null */
    @FXML
    private Label mVersion;
    /** the JDK label, never null */
    @FXML
    private Label mJdk;
    /** the modal dialog we show, null when no dialog is shown */
    private Stage mStage;
    /** the main stage we belong to, never null after init was called */
    private Stage mMainStage;
    /** our fade transitions, never null */
    private List<Transition> mTransitions= new ArrayList<Transition>();

    /** {@inheritDoc} */
    @Override
    public void init(final Dc pDc, final Map<String, ? super Object> pArgs)
    {
        mMainStage= (Stage)pArgs.get(AppData.MAIN_STAGE);

        // init our labels
        StringBuilder versionText = new StringBuilder(mVersion.getText());
        versionText.append(", Version ");
        versionText.append(Version.VERSION);
        versionText.append(" (");
        versionText.append(Version.DATE);
        versionText.append(" )");
        mVersion.setText(versionText.toString());

        StringBuilder jdkText = new StringBuilder("Java VM from ");
        jdkText.append(System.getProperty("java.vendor"));
        jdkText.append(", Version ");
        jdkText.append(System.getProperty("java.version"));
        mJdk.setText(jdkText.toString());

        // fix the size of the logo area
        double maxWidth= Math.max(mAuthorImage.getWidth(), mAboutImage.getWidth());
        double maxHeight= Math.max(mAuthorImage.getHeight(), mAboutImage.getHeight());
        mAboutLabel.setMinHeight(maxHeight);
        mAboutLabel.setMinWidth(maxWidth);

        // init our transition and attach it to a mouse click
        final Transition transitionForward = createTransition(mAboutLogo, mAuthorImage);
        final Transition transitionBackward = createTransition(mAboutLogo, mAboutImage);
        mTransitions.add(transitionBackward);
        mTransitions.add(transitionBackward);
        mAboutLabel.setOnMouseClicked(new EventHandler<MouseEvent>()
        {
            @Override
            public void handle(final MouseEvent pMouseEvent)
            {
                if (mAboutLogo.getImage().equals(mAboutImage))
                {
                    transitionForward.play();
                }
                else
                {
                    transitionBackward.play();
                }
            }
        });
    }

    /**
     * Create a transition from the current image in the image view to the new image
     * by fading the current image out and fading the nem image in.
     * Code inspired by http://java-buddy.blogspot.de/2012/02/implement-fade-infade-out-animation.html
     * @param pImageView the image view, must not be null
     * @param pNewImage the new image, must not be null
     * @return the transtion, never null
     */
    private Transition createTransition(final ImageView pImageView, final Image pNewImage)
    {
        FadeTransition fadeOutTransition = new FadeTransition(FADE_DURATION, pImageView);
        fadeOutTransition.setFromValue(1.0);
        fadeOutTransition.setToValue(0.0);
        fadeOutTransition.setOnFinished(new EventHandler<ActionEvent>()
        {
            @Override
            public void handle(final ActionEvent pActionEvent)
            {
                pImageView.setImage(pNewImage);
            }
        });

        FadeTransition fadeInTransition = new FadeTransition(FADE_DURATION, pImageView);
        fadeInTransition.setFromValue(0.0);
        fadeInTransition.setToValue(1.0);

        SequentialTransition sequentialTransition = SequentialTransitionBuilder.create()
                .children(fadeOutTransition, fadeInTransition)
                .build();

        return sequentialTransition;
    }

    /**
     * Called by our dialog peer to show our dialog, will block until the
     * user closes the dialog.
     */
    void show()
    {
        mStage = new Stage();
        mStage.initOwner(mMainStage);
        Scene scene = new Scene(mRootContainer);
        mStage.setScene(scene);
        mStage.setTitle("Über");
        mStage.initModality(Modality.WINDOW_MODAL);
        mStage.setOnCloseRequest(new EventHandler<WindowEvent>()
        {
            @Override
            public void handle(final WindowEvent pWindowEvent)
            {
                pWindowEvent.consume();
                ok();
            }
        });
        mStage.getIcons().add(mAppIcon);
        mAboutLogo.setImage(mAboutImage);
        mStage.showAndWait();
        mStage= null;
    }

    /**
     * Called if the user presses ok or closes the dialog.
     */
    public void ok()
    {
        mStage.hide();

        // stop any animation
        for (final Transition transition : mTransitions)
        {
            transition.stop();
        }
    }
}
