package net.sf.cuf.csvviewfx.main;

import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.scene.layout.BorderPane;
import net.sf.cuf.fw.Application;
import net.sf.cuf.fw.Dc;
import net.sf.cuf.fw.Pc;
import net.sf.cuf.xfer.Response;
import javafx.application.Platform;
import javafx.beans.property.BooleanProperty;
import javafx.collections.ListChangeListener;
import javafx.event.EventHandler;
import javafx.fxml.FXML;
import javafx.scene.Scene;
import javafx.scene.control.CheckBox;
import javafx.scene.control.Label;
import javafx.scene.control.SelectionMode;
import javafx.scene.control.TableColumn;
import javafx.scene.control.TableView;
import javafx.scene.control.TextField;
import javafx.scene.control.Tooltip;
import javafx.scene.control.cell.MapValueFactory;
import javafx.scene.image.Image;
import javafx.scene.input.KeyEvent;
import javafx.stage.DirectoryChooser;
import javafx.stage.FileChooser;
import javafx.stage.Stage;
import javafx.stage.Window;
import javafx.stage.WindowEvent;
import net.sf.cuf.csvviewfx.AppData;
import net.sf.cuf.csvviewfx.data.FilteredObservableList;
import net.sf.cuf.csvviewfx.util.TableCopyAction;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Presentation controller for the main window.
 */
public class MainPc implements Pc
{
    /** our app icon, never null */
    @FXML
    private Image               mAppIcon;
    /** our auto filter checkbox, never null */
    @FXML
    private CheckBox            mAutoFilter;
    /** our status field, never null */
    @FXML
    private Label               mStatusText;
    /** our input field, never null */
    @FXML
    private TextField           mFilter;
    /** our Table, never null */
    @FXML
    private TableView           mCsvTable;
    /** our scene, never null */
    private Scene               mScene;
    /** our root container, never null */
    @FXML
    public BorderPane           mRootContainer;
    /** our application state, never null after init was called */
    private Application         mApplication;
    /** our dialog peer, never null after init was called */
    private MainDc              mDc;
    /** null or the last filtered expression */
    private String              mOldFilterText;
    /** null or the old tool tip of our filter */
    private Tooltip             mOldTooltip;
    /** null/empty or the old style of our filter */
    private String              mOldBackground;
    /** our model containing all data, never null after init was called */
    private FilteredObservableList mModel;

    /** {@inheritDoc} */
    @Override
    public void init(final Dc pDc, final Map<String, ? super Object> pArgs)
    {
        // set up link to our app and dialog peer
        mApplication= (Application) pArgs.get(Application.APPLICATION_KEY);
        mDc= (MainDc)pDc;

        // get data from map and show it, must be done before we filter
        setModel((FilteredObservableList)pArgs.get(MainDc.TABLE_ARG));

        // we do manage the close ourself
        Platform.setImplicitExit(false);

        // set the title, our scene and connect the close event
        Stage stage= (Stage)pArgs.get(AppData.MAIN_STAGE);
        stage.setTitle("CsvView: die CSV CUF Testanwendung");
        mScene = new Scene(mRootContainer);
        stage.setScene(mScene);
        stage.setOnCloseRequest(new EventHandler<WindowEvent>()
        {
            @Override
            public void handle(final WindowEvent pWindowEvent)
            {
                pWindowEvent.consume();
                quit();
            }
        });
        stage.getIcons().add(mAppIcon);

        // restore the layout postion
        double x     = Double.parseDouble(AppData.DEFAULT_X);
        double y     = Double.parseDouble(AppData.DEFAULT_Y);
        double width = Double.parseDouble(AppData.DEFAULT_WIDTH);
        double height= Double.parseDouble(AppData.DEFAULT_WIDTH);
        try
        {
            x     = Double.parseDouble(mApplication.getProperty(AppData.DEFAULT_X_KEY, AppData.DEFAULT_X));
            y     = Double.parseDouble(mApplication.getProperty(AppData.DEFAULT_Y_KEY, AppData.DEFAULT_Y));
            width = Double.parseDouble(mApplication.getProperty(AppData.DEFAULT_WIDTH_KEY, AppData.DEFAULT_WIDTH));
            height= Double.parseDouble(mApplication.getProperty(AppData.DEFAULT_HEIGHT_KEY, AppData.DEFAULT_HEIGHT));
        }
        catch (NumberFormatException ignored)
        {
        }
        stage.setX(x);
        stage.setY(y);
        stage.setWidth(width);
        stage.setHeight(height);

        // adjust the filter check box and text and filter the initial model
        boolean doAutofilter = Boolean.valueOf(mApplication.getProperty(AppData.DEFAULT_AUTOFILTER_KEY, AppData.DEFAULT_AUTOFILTER));
        mAutoFilter.setSelected(doAutofilter);
        String filterText= mApplication.getProperty(AppData.DEFAULT_FILTER_KEY, null);
        if (filterText!=null)
        {
            mFilter.setText(filterText);
            mFilter.selectAll();
            mFilter.requestFocus();
            doFilter(filterText);
        }

        // remember the textfield background+tooltip for our error-handling
        mOldFilterText= null;
        mOldTooltip   = mFilter.getTooltip();
        mOldBackground= mFilter.getStyle();

        // attach the on-the-fly callback we use if the user wants auto filter
        mFilter.textProperty().addListener(new ChangeListener<String>()
        {
            @Override
            public void changed(ObservableValue<? extends String> pObservableValue, String pOldValue, String pNewValue)
            {
                if (mAutoFilter.isSelected() && !pNewValue.equals(mOldFilterText))
                {
                    doFilter(pNewValue);
                }
            }
        });

        // enable multi-select and attach our clip board handler and our selection listener
        mCsvTable.getSelectionModel().setSelectionMode(SelectionMode.MULTIPLE);
        BooleanProperty copyRow= (BooleanProperty)mApplication.getAppModel().get(AppData.COPY_WHOLE_ROW_PROPERTY);
        mCsvTable.setOnKeyReleased(new TableCopyAction(copyRow));
        mCsvTable.getSelectionModel().getSelectedIndices().addListener(new ListChangeListener()
        {
            @Override
            public void onChanged(final Change pChange)
            {
                updateStatusLabel();
            }
        });

        // set initial status text line
        updateStatusLabel();

        // TODO see Swing MainPc for missing stuff

        // go live!
        stage.show();
    }

    /**
     * Helper method that does the filtering. If the filter failed it marks
     * the textfield with a color/tooltip.
     * @param pFilterText the new filter text
     */
    private void doFilter(final String pFilterText)
    {
        if (pFilterText.isEmpty())
        {
            mModel.filterReset();
        }
        else
        {
            boolean filterOK= mModel.filter(pFilterText);
            if (filterOK)
            {
                mFilter.setTooltip(mOldTooltip);
                mFilter.setStyle(mOldBackground);
            }
            else
            {
                mFilter.setStyle("-fx-background-color: cyan;");
                mFilter.setTooltip(new Tooltip("Eingabe ist keine gültiger regulärer Ausdruck"));
            }
        }
        updateStatusLabel();

        mOldFilterText= pFilterText;

    }

    /**
     * Small helper method that updates the status label.
     */
    private void updateStatusLabel()
    {
        int    allRows = mModel.getAllRowsCount();
        int    rows    = mModel.size();
        int    selected= mCsvTable.getSelectionModel().getSelectedIndices().size();
        mStatusText.setText(allRows + " Zeilen, davon " + rows + " gefiltered" + " und " + selected + " selektiert");
    }

    /**
     * Callback from the file open menu.
     * We show a file chooser and forward the open to our dialog component, we also store
     * the current directory in our application properties.
     */
    public void openFile()
    {
        FileChooser fileChooser= new FileChooser();
        try
        {
            String dir= mApplication.getProperty(AppData.DEFAULT_DIR_KEY, AppData.DEFAULT_DIR);
            File initialDir = new File(dir).getCanonicalFile();
            if (initialDir.isDirectory())
            {
                fileChooser.setInitialDirectory(initialDir);
            }
        }
        catch (IOException ignored)
        {
        }
        fileChooser.getExtensionFilters().add(AppData.CSV_EXTENSIONS);

        File file= fileChooser.showOpenDialog(mScene.getWindow());
        if (file!=null && file.isFile())
        {
            mDc.openFile(file);
            // update settings
            mApplication.setProperty(AppData.DEFAULT_DIR_KEY, file.getParent());
            mApplication.setProperty(AppData.FILE_NAME_KEY, file.getAbsolutePath());
            mApplication.setProperty(AppData.INPUT_SELECTION_KEY, AppData.INPUT_SELECTION_FILE);
        }
    }

    /**
     * Callback from the tree open menu.
     */
    public void openTree()
    {
        DirectoryChooser directoryChooser= new DirectoryChooser();
        try
        {
            String dir= mApplication.getProperty(AppData.DEFAULT_DIR_KEY, AppData.DEFAULT_DIR);
            File initialDir = new File(dir).getCanonicalFile();
            if (initialDir.isDirectory())
            {
                directoryChooser.setInitialDirectory(initialDir);
            }
        }
        catch (IOException ignored)
        {
        }

        File file= directoryChooser.showDialog(mScene.getWindow());
        if (file!=null && file.isDirectory())
        {
            mDc.openTree(file);
            // update settings
            mApplication.setProperty(AppData.DEFAULT_DIR_KEY, file.getAbsolutePath());
            mApplication.setProperty(AppData.TREE_NAME_KEY, file.getAbsolutePath());
            mApplication.setProperty(AppData.INPUT_SELECTION_KEY, AppData.INPUT_SELECTION_TREE);
        }
    }

    /**
     * Callback from the file quit menu, will save our properties and quit.
     */
    public void quit()
    {
        // save current size
        Window window = mScene.getWindow();
        double width = window.getWidth();
        double height= window.getHeight();
        double x     = window.getX();
        double y     = window.getY();
        mApplication.setProperty(AppData.DEFAULT_X_KEY,      Double.toString(x));
        mApplication.setProperty(AppData.DEFAULT_Y_KEY,      Double.toString(y));
        mApplication.setProperty(AppData.DEFAULT_WIDTH_KEY,  Double.toString(width));
        mApplication.setProperty(AppData.DEFAULT_HEIGHT_KEY, Double.toString(height));

        // save the filter settings
        mApplication.setProperty(AppData.DEFAULT_AUTOFILTER_KEY, Boolean.toString(mAutoFilter.isSelected()));
        mApplication.setProperty(AppData.DEFAULT_FILTER_KEY, mFilter.getText());

        // TODO: use a ApplicationFx method instead
        Platform.exit();
    }

    /**
     * Callback from the show options menu.
     */
    public void showOptions()
    {
        mDc.showOptions();
    }

    /**
     * Callback from the show about menu.
     */
    public void showAbout()
    {
        mDc.showAbout();
    }

    /**
     * Callback from the auto filter checkbox.
     */
    public void autoFilterChanged()
    {
        if (mAutoFilter.isSelected())
        {
            doFilter(mFilter.getText());
        }
    }

    /**
     * Callback from the search button and the filter textfield.
     */
    public void search()
    {
        doFilter(mFilter.getText());
    }

    /**
     * Called from our Dc to set a new table model, will also filter.
     * @param pModel the table model to set, must not be null
     */
    public void setModel(final FilteredObservableList pModel)
    {
        List<TableColumn<Map, String>> columns= new ArrayList<TableColumn<Map, String>>(pModel.getColumnCount());
        for (int i = 0, n = pModel.getColumnCount(); i < n; i++)
        {
            String columnName=  pModel.getColumnName(i);
            TableColumn<Map, String> tableColumn = new TableColumn<Map, String>(columnName);
            tableColumn.setCellValueFactory(new MapValueFactory<String>(columnName));
            columns.add(tableColumn);
        }
        mCsvTable.getColumns().clear();
        mCsvTable.getColumns().addAll(columns);
        mCsvTable.setItems(pModel);
        mModel = pModel;
        if (mAutoFilter.isSelected())
        {
            doFilter(mFilter.getText());
        }
    }

    /**
     * Called from our Dc if loading failed.
     * @param pResponse the response containing more information about the failure, never null
     */
    public void showLoadError(final Response pResponse)
    {
        StringBuilder sb = new StringBuilder();
        sb.append("BAD NEWS!\n");
        sb.append("Während des Ladens ging etwas schief:\n");
        if (pResponse.getError()==null)
        {
            sb.append("Aber wir wissen nicht was :-(");
        }
        else
        {
            sb.append(pResponse.getError());
        }

        // http://stackoverflow.com/questions/11794987/what-is-replacement-of-alert-in-javafx-2-1-alert-is-in-javafx-1-3-but-not-in-j?lq=1
        // http://stackoverflow.com/questions/8309981/how-to-create-and-show-common-dialog-error-warning-confirmation-in-javafx-2
        // https://github.com/4ntoine/JavaFxDialog/wiki
        // TODO
        System.out.println("showLoadError: "+sb);
    }
}
