package net.sf.cuf.csvviewfx.option;

import javafx.scene.layout.StackPane;
import net.sf.cuf.appevent.AppEvent;
import net.sf.cuf.appevent.AppEventSupport;
import net.sf.cuf.appevent.AppEventUtil;
import net.sf.cuf.fw.Application;
import net.sf.cuf.fw.Dc;
import net.sf.cuf.fw.Pc;
import javafx.beans.property.BooleanProperty;
import javafx.event.EventHandler;
import javafx.fxml.FXML;
import javafx.scene.Scene;
import javafx.scene.control.Button;
import javafx.scene.control.CheckBox;
import javafx.scene.control.ComboBox;
import javafx.scene.control.RadioButton;
import javafx.scene.control.TextField;
import javafx.scene.image.Image;
import javafx.stage.DirectoryChooser;
import javafx.stage.FileChooser;
import javafx.stage.Modality;
import javafx.stage.Stage;
import javafx.stage.WindowEvent;
import net.sf.cuf.csvviewfx.AppData;
import net.sf.cuf.csvviewfx.DataSourceChanged;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * Presentation controller for the option dialog.
 */
public class OptionPc implements Pc, AppEventSupport
{
    /** our app icon, never null */
    @FXML
    private Image mAppIcon;
    /** our root container, never null */
    @FXML
    public StackPane mRootContainer;
    /** text field for the file */
    @FXML
    private TextField mFileText;
    /** radio buttion for the tree */
    @FXML
    private RadioButton mTree;
    /** text field for the tree */
    @FXML
    private TextField mTreeText;
    /** checkbox if the first row is a header */
    @FXML
    private CheckBox mFirstRow;
    /** text field for the separator */
    @FXML
    private TextField mCSVSeparator;
    /** combobox for the encoding */
    @FXML
    private ComboBox<String> mEncoding;
    /** radio button for cell only copy */
    @FXML
    private RadioButton mCell;
    /** radio button for whole row copy */
    @FXML
    private RadioButton mRow;
    /** our default button, never null */
    @FXML
    public Button mDefault;
    /** our cancel button, never null */
    @FXML
    public Button mCancel;
    /** our OK button, never null */
    @FXML
    public Button mOK;
    /** the main stage we belong to, never null after init was called */
    private Stage mMainStage;
    /** our application, never null after init was called */
    private Application mApplication;
    /** the modal dialog we show, null when no dialog is shown */
    private Stage mStage;
    /** our dialog peer, never null after init was called */
    private OptionDc mDc;
    /** our mapping between input toogle group index and input selection */
    private static final List<String> INPUT_MAPPING= Arrays.asList(AppData.INPUT_SELECTION_FILE, AppData.INPUT_SELECTION_TREE);

    /** {@inheritDoc} */
    @Override
    public void init(final Dc pDc, final Map<String, ? super Object> pArgs)
    {
        // set up link to our app, dialog peer and visual parent
        mApplication= (Application) pArgs.get(Application.APPLICATION_KEY);
        mDc= (OptionDc)pDc;
        mMainStage= (Stage)pArgs.get(AppData.MAIN_STAGE);
    }

    /**
     * Process an AppEvent.
     * @param pAppEvent event that should be routed/processes
     */
    @Override
    public void postAppEvent(final AppEvent pAppEvent)
    {
        AppEventUtil.postAppEvent(mDc, pAppEvent);
    }

    /**
     * Called by our dialog peer to show our dialog, will block until the
     * user closes the dialog.
     */
    void show()
    {
        mStage = new Stage();
        mStage.initOwner(mMainStage);
        Scene scene= new Scene(mRootContainer);
        mStage.setScene(scene);
        mStage.setTitle("Optionen");
        mStage.initModality(Modality.WINDOW_MODAL);
        mStage.setOnCloseRequest(new EventHandler<WindowEvent>()
        {
            @Override
            public void handle(final WindowEvent pWindowEvent)
            {
                pWindowEvent.consume();
                cancel();
            }
        });
        mStage.getIcons().add(mAppIcon);
        initData();
        mStage.showAndWait();
        mStage= null;
    }

    /**
     * Called before we show the data, used to set the UI to the current values.
     */
    private void initData()
    {
        int index= INPUT_MAPPING.indexOf(mApplication.getProperty(AppData.INPUT_SELECTION_KEY, AppData.INPUT_SELECTION_FILE));
        if (index<0) index= INPUT_MAPPING.indexOf(AppData.INPUT_SELECTION_FILE);
        mTree.getToggleGroup().getToggles().get(index).setSelected(true);
        mFileText.setText(mApplication.getProperty(AppData.FILE_NAME_KEY, AppData.FILE_NAME_DEFAULT));
        mTreeText.setText(mApplication.getProperty(AppData.TREE_NAME_KEY, AppData.TREE_NAME_DEFAULT));

        boolean firstRowIsHeader= Boolean.valueOf(mApplication.getProperty(AppData.FIRST_ROW_IS_HEADER_KEY, AppData.FIRST_ROW_IS_HEADER_DEFAULT));
        mFirstRow.setSelected(firstRowIsHeader);
        String seperator= mApplication.getProperty(AppData.CSV_SEPARATOR_KEY, AppData.CSV_SEPARATOR_DEFAULT);
        mCSVSeparator.setText(seperator);
        String encoding= mApplication.getProperty(AppData.ENCODING_NAME_KEY, AppData.ENCODING_NAME_DEFAULT);
        mEncoding.getSelectionModel().select(encoding);

        BooleanProperty copyRow= (BooleanProperty)mApplication.getAppModel().get(AppData.COPY_WHOLE_ROW_PROPERTY);
        mRow.setSelected(copyRow.getValue());
        mCell.setSelected(!copyRow.getValue());
    }

    /**
     * Callback from the search csv file button.
     */
    public void searchFile()
    {
        FileChooser fileChooser= new FileChooser();
        try
        {
            String file= mFileText.getText();
            File initialDir = new File(file).getCanonicalFile().getParentFile();
            if (initialDir.isDirectory())
            {
                fileChooser.setInitialDirectory(initialDir);
            }
        }
        catch (IOException ignored)
        {
        }
        fileChooser.getExtensionFilters().add(AppData.CSV_EXTENSIONS);

        File file= fileChooser.showOpenDialog(mStage.getScene().getWindow());
        if (file!=null && file.isFile())
        {
            mFileText.setText(file.getPath());
        }
    }


    /**
     * Callback from the search dir tree button.
     */
    public void searchDir()
    {
        DirectoryChooser directoryChooser= new DirectoryChooser();
        try
        {
            String dir= mTreeText.getText();
            File initialDir = new File(dir).getCanonicalFile();
            if (initialDir.isDirectory())
            {
                directoryChooser.setInitialDirectory(initialDir);
            }
        }
        catch (IOException ignored)
        {
        }

        File file= directoryChooser.showDialog(mStage.getScene().getWindow());
        if (file!=null && file.isDirectory())
        {
            mFileText.setText(file.getPath());
        }
    }

    /**
     * Callback from our ok button, we update the app state and fire an DataSourceChanged
     * app event if something was changed.
     */
    public void ok()
    {
        boolean sourceChanged= false;
        int oldIndex= INPUT_MAPPING.indexOf(mApplication.getProperty(AppData.INPUT_SELECTION_KEY, AppData.INPUT_SELECTION_FILE));
        int newIndex= mTree.getToggleGroup().getToggles().indexOf(mTree.getToggleGroup().getSelectedToggle());
        if (oldIndex!=newIndex)
        {
            mApplication.setProperty(AppData.INPUT_SELECTION_KEY, INPUT_MAPPING.get(newIndex));
            sourceChanged= true;
        }
        String newFileText= mFileText.getText();
        String oldFileText= mApplication.getProperty(AppData.FILE_NAME_KEY, AppData.FILE_NAME_DEFAULT);
        if (!oldFileText.equals(newFileText))
        {
            mApplication.setProperty(AppData.DEFAULT_DIR_KEY, new File(newFileText).getParent());
            mApplication.setProperty(AppData.FILE_NAME_KEY, newFileText);
            sourceChanged= sourceChanged || (newIndex==INPUT_MAPPING.indexOf(AppData.INPUT_SELECTION_FILE));
        }
        String newTreeText= mTreeText.getText();
        String oldTreeText= mApplication.getProperty(AppData.TREE_NAME_KEY, AppData.TREE_NAME_DEFAULT);
        if (!oldTreeText.equals(newTreeText))
        {
            mApplication.setProperty(AppData.DEFAULT_DIR_KEY, newTreeText);
            mApplication.setProperty(AppData.TREE_NAME_KEY, newTreeText);
            sourceChanged= sourceChanged || (newIndex==INPUT_MAPPING.indexOf(AppData.INPUT_SELECTION_TREE));
        }

        boolean oldFirstRowIsHeader= Boolean.valueOf(mApplication.getProperty(AppData.FIRST_ROW_IS_HEADER_KEY, AppData.FIRST_ROW_IS_HEADER_DEFAULT));
        boolean newFirstRowIsHeader= mFirstRow.isSelected();
        if (oldFirstRowIsHeader!=newFirstRowIsHeader)
        {
            mApplication.setProperty(AppData.FIRST_ROW_IS_HEADER_KEY, Boolean.toString(newFirstRowIsHeader));
            sourceChanged= sourceChanged || (newIndex==INPUT_MAPPING.indexOf(AppData.INPUT_SELECTION_FILE));
        }
        String oldSeperator= mApplication.getProperty(AppData.CSV_SEPARATOR_KEY, AppData.CSV_SEPARATOR_DEFAULT);
        String newSeperator= mCSVSeparator.getText();
        if (!oldSeperator.equals(newSeperator))
        {
            mApplication.setProperty(AppData.CSV_SEPARATOR_KEY, newSeperator);
            sourceChanged= sourceChanged || (newIndex==INPUT_MAPPING.indexOf(AppData.INPUT_SELECTION_FILE));
        }
        String oldEncoding= mApplication.getProperty(AppData.ENCODING_NAME_KEY, AppData.ENCODING_NAME_DEFAULT);
        String newEncoding= mEncoding.getSelectionModel().getSelectedItem();
        if (!oldEncoding.equals(newEncoding))
        {
            mApplication.setProperty(AppData.ENCODING_NAME_KEY, newEncoding);
            sourceChanged= sourceChanged || (newIndex==INPUT_MAPPING.indexOf(AppData.INPUT_SELECTION_FILE));
        }

        BooleanProperty copyRow= (BooleanProperty)mApplication.getAppModel().get(AppData.COPY_WHOLE_ROW_PROPERTY);
        if (mRow.isSelected() != copyRow.getValue())
        {
            copyRow.setValue(mRow.isSelected());
        }

        // notify the display if something changed
        if (sourceChanged)
        {
            postAppEvent(new DataSourceChanged(this));
        }
        mStage.hide();
    }

    /**
     * Callback from our cancel button or when the dialog is closed, we just hide the
     * stage.
     */
    public void cancel()
    {
        mStage.hide();
    }

    /**
     * Callback from our defaults button, we set all settings to default values.
     */
    public void defaults()
    {
        int index= INPUT_MAPPING.indexOf(AppData.INPUT_SELECTION_FILE);
        mTree.getToggleGroup().getToggles().get(index).setSelected(true);
        mFileText.setText(AppData.FILE_NAME_DEFAULT);
        mTreeText.setText(AppData.TREE_NAME_DEFAULT);
        mFirstRow.setSelected(Boolean.valueOf(AppData.FIRST_ROW_IS_HEADER_DEFAULT));
        mCSVSeparator.setText(AppData.CSV_SEPARATOR_DEFAULT);
        mEncoding.getSelectionModel().select(AppData.ENCODING_NAME_DEFAULT);
        mRow.setSelected(Boolean.valueOf(AppData.COPY_WHOLE_ROW_DEFAULT));
        mCell.setSelected(!Boolean.valueOf(AppData.COPY_WHOLE_ROW_DEFAULT));
    }
}
