package net.sf.cuf.csvviewfx.main;

import net.sf.cuf.appevent.AppEvent;
import net.sf.cuf.appevent.AppEventManager;
import net.sf.cuf.appevent.AppEventSupport;
import net.sf.cuf.appevent.AppEventUtil;
import net.sf.cuf.appevent.BindEvent;
import net.sf.cuf.fw.Application;
import net.sf.cuf.fw.Dc;
import net.sf.cuf.xfer.JavaFXDispatcher;
import net.sf.cuf.xfer.Dispatch;
import net.sf.cuf.xfer.Request;
import net.sf.cuf.xfer.Response;
import net.sf.cuf.csvviewfx.AppData;
import net.sf.cuf.csvviewfx.about.AboutDc;
import net.sf.cuf.csvviewfx.option.OptionDc;
import net.sf.cuf.csvviewfx.data.CsvFileredObservableList;
import net.sf.cuf.csvviewfx.DataSourceChanged;
import net.sf.cuf.csvviewfx.data.FilteredObservableList;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

/**
 * Dialog controller for the main window.
 */
public class MainDc implements Dc, AppEventSupport
{
    /** argument key for init holding the data */
    static final String TABLE_ARG= "tableModel";

    /** helper for the AppEvent stuff, never null  */
    private AppEventManager     mAppEventManger;
    /** our Dispatcher, never null */
    private Dispatch            mDispatcher;
    /** our presentation controller, never null */
    private MainPc              mPc;
    /** our about dialog controller, never null after init was called */
    private AboutDc             mAboutDc;
    /** our option dialog controller, never null after init was called */
    private OptionDc            mOptionDc;
    /** our application, never null after init was called */
    private Application         mApplication;

    /**
     * Create a new dialog controller.
     * @param pPc our presentation controller peer, must not be null
     */
    public MainDc(final MainPc pPc)
    {
        mAppEventManger= new AppEventManager();
        mDispatcher= new JavaFXDispatcher();
        mPc = pPc;
    }

    /**
     * This method initializes the AppEvent handling, loads the first
     * data, initializes  all sub-dialog components and then initializes
     * its presentation component.
     * @param pParent our parent dialog component, is null because we have no parent
     * @param pArgs the arguments
     */
    @Override
    public void init(final Dc pParent, final Map<String, ? super Object> pArgs)
    {
        // set up link to our app
        mApplication= (Application) pArgs.get(Application.APPLICATION_KEY);

        // we want to get noticed if the data source is changed
        BindEvent bindEvent= new BindEvent(this, "dataSourceChanged", DataSourceChanged.class, 0);
        postAppEvent(bindEvent);

        // first create and initialize all our sub-Dc's
        mAboutDc= new AboutDc();
        mAboutDc.init(this, pArgs);
        mOptionDc= new OptionDc();
        mOptionDc.init(this, pArgs);

        // load first data
        Request  loadDataRequest= getCurrentRequest();
        Response response       = mDispatcher.syncExecute(loadDataRequest);
        FilteredObservableList firstData;
        if (response.isValid())
        {
            firstData= (FilteredObservableList) response.getResult();
        }
        else
        {
            firstData= new CsvFileredObservableList();
        }

        // now initialize our Pc
        Map<String, ? super Object> args= new HashMap<String, Object>(pArgs);
        args.put(TABLE_ARG, firstData);
        mPc.init(this, args);
    }

    /**
     * Process an AppEvent.
     * @param pAppEvent event that should be routed/processes
     */
    public void postAppEvent(final AppEvent pAppEvent)
    {
        AppEventUtil.postAppEvent(mAppEventManger, pAppEvent);
    }

    /**
     * Small helper to generate the load request depending on the current input selection.
     * @return the current load request, never null
     */
    private Request getCurrentRequest()
    {
        Request request;
        String inputSelection= mApplication.getProperty(AppData.INPUT_SELECTION_KEY, AppData.INPUT_SELECTION_FILE);
        if (AppData.INPUT_SELECTION_FILE.equals(inputSelection))
        {
            File   file             = new File(mApplication.getProperty(AppData.FILE_NAME_KEY, AppData.FILE_NAME_DEFAULT));
            boolean firstRowIsHeader= Boolean.valueOf(mApplication.getProperty(AppData.FIRST_ROW_IS_HEADER_KEY, AppData.FIRST_ROW_IS_HEADER_DEFAULT));
            String seperator        = mApplication.getProperty(AppData.CSV_SEPARATOR_KEY, AppData.CSV_SEPARATOR_DEFAULT);
            String encoding         = mApplication.getProperty(AppData.ENCODING_NAME_KEY, AppData.ENCODING_NAME_DEFAULT);
            request= new LoadFileRequest(file, firstRowIsHeader, seperator, encoding);


        }
        else
        {
            File tree= new File(mApplication.getProperty(AppData.TREE_NAME_KEY, AppData.TREE_NAME_DEFAULT));
            request= new LoadTreeRequest(tree);
        }

        return request;
    }

    /**
     * AppEvent callback for the DataSourceChanged event, we re-load
     * the current data.
     * @param pDataSourceChanged not used
     */
    @SuppressWarnings({"UnusedParameters"})
    public void dataSourceChanged(final DataSourceChanged pDataSourceChanged)
    {
        Request loadDataRequest= getCurrentRequest();
        loadDataRequest.setDispatchTarget(this, "loadResponse");
        mDispatcher.asyncDispatchInEDT(loadDataRequest);
    }

    /**
     * Called from our presentation peer to load a file.
     * @param pFile the file we should load, must not be null
     */
    void openFile(final File pFile)
    {
        boolean firstRowIsHeader= Boolean.valueOf
                (mApplication.getProperty(AppData.FIRST_ROW_IS_HEADER_KEY, AppData.FIRST_ROW_IS_HEADER_DEFAULT));
        String seperator=
                mApplication.getProperty(AppData.CSV_SEPARATOR_KEY, AppData.CSV_SEPARATOR_DEFAULT);
        String encoding=
                mApplication.getProperty(AppData.ENCODING_NAME_KEY, AppData.ENCODING_NAME_DEFAULT);
        Request request= new LoadFileRequest(pFile, firstRowIsHeader, seperator, encoding);
        request.setDispatchTarget(this, "loadResponse");
        mDispatcher.asyncDispatchInEDT(request);
    }

    /**
     * Called from our presentation peer to load a tree, we also set the
     * file name as the new default file name.
     * @param pDirectory the dir we should load, must not be null
     */
    public void openTree(final File pDirectory)
    {
        Request request= new LoadTreeRequest(pDirectory);
        request.setDispatchTarget(this, "loadResponse");
        mDispatcher.asyncDispatchInEDT(request);
    }

    /**
     * Called by the Dispatcher after the request dispatched in openFile,
     * openTree or dataSourceChanged finished.
     * @param pResponse the response, containing either a FilteredObservableList as result
     *                  or an Exception as an error.
     */
    public void loadResponse(final Response pResponse)
    {
        if (pResponse.isValid())
        {
            FilteredObservableList newModel= (FilteredObservableList)pResponse.getResult();
            mPc.setModel(newModel);
        }
        else
        {
            mPc.showLoadError(pResponse);
        }
    }

    /**
     * Called from our presentation controller when the user wants
     * to show the about dialog
     */
    void showAbout()
    {
        mAboutDc.show();
    }

    /**
     * Called from our presentation controller when the user wants
     * to show the options dialog
     */
    void showOptions()
    {
        mOptionDc.show();
    }
}
