package net.sf.cuf.csvview.util;

import javax.swing.table.AbstractTableModel;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * A CSVTableModel generates its data from parsing a CSV input stream.
 */
public class CSVTableModel extends AbstractTableModel implements FilteredTableModel
{
    /** total number of rows */
    private int        mRows;
    /** total number of columns (some rows may have less columns) */
    private int        mColumns;
    /** the data */
    private String[][] mData;
    /** number of rows after applying the filter */
    private int        mFilteredRows;
    /** mapper from filtered rows to rows in mData */
    private int[]      mFilteredRowsIndexMapper;
    /** column header names */
    private String[]   mRowHeaders;
    /** null or last filter expression */
    private String     mLastFilter;
    /** true if the first row contains the header */
    private boolean    mFirstRowIsHeader;
    /** the separator for our "comma" separated values */
    private String     mCSVSeparator;
    /** default separator */
    private static final String CSV_SEPARATOR= ",";

    /** helper for all empty strings found during split-up */
    private static final String   EMPTY            = "";
    /** helper to get an String array from a list */
    private static final String[] DUMMY            = new String[0];
    /** static pattern for a quote */
    private static final Pattern  CSV_QUOTE_PATTERN= Pattern.compile("\"\"");

    /**
     * Create an empty table model (no rows, no columns).
     */
    public CSVTableModel()
    {
        init();
    }

    /**
     * Create a new table model by parsing the input stream as CSV data.
     * @param pCSVData the input we read our data from, may be null
     * @param pFirstRowIsHeader true if the first row contains the header
     * @param pCSVSeparator "comma" separator, may be null (then "," is taken)
     * @param pEncoding the encoding pf pCSVData, may be null if pCSVData is null
     * @throws IOException if we have IO problems
     */
    public CSVTableModel(final InputStream pCSVData, final boolean pFirstRowIsHeader, final String pCSVSeparator, final String pEncoding) throws IOException
    {
        if (pCSVData==null)
        {
            init();
        }
        else
        {
            init(new InputStreamReader(pCSVData, pEncoding), pFirstRowIsHeader, pCSVSeparator);
        }
    }

    /**
     * Create a new table model by parsing the CSV data from the input reader.
     * @param pCSVData the reader we read the CSV data from, may be null
     * @param pFirstRowIsHeader true if the first row contains the header
     * @param pCSVSeparator "comma" separator, may be null
     * @throws IOException if we have IO problems
     */
    public CSVTableModel(final Reader pCSVData, final boolean pFirstRowIsHeader, final String pCSVSeparator) throws IOException
    {
        if (pCSVData==null)
        {
            init();
        }
        else
        {
            init(pCSVData, pFirstRowIsHeader, pCSVSeparator);
        }
    }

    /**
     * Small helper method doing common constructor stuff.
     */
    private void init()
    {
        mRows                   = 0;
        mColumns                = 0;
        mData                   = new String[mRows][mColumns];
        mFilteredRows           = 0;
        mFilteredRowsIndexMapper= new int[mRows];
        mRowHeaders             = new String[mColumns];
        mLastFilter             = null;
        mFirstRowIsHeader       = false;
        mCSVSeparator           = CSV_SEPARATOR;
    }

    /**
     * Small helper method doing common constructor stuff.
     * @param pCSVData the reader we get the data from, must not be null
     * @param pFirstRowIsHeader flag if the first row should be treated as a header
     * @param pCSVSeparator the separator of the data, may be null
     * @throws IOException on I/O problems
     */
    private void init(final Reader pCSVData, final boolean pFirstRowIsHeader, final String pCSVSeparator) throws IOException
    {
        if (pCSVData==null)
        {
            init();
            return;
        }
        if (pCSVSeparator==null)
        {
            mCSVSeparator= CSV_SEPARATOR;
        }
        else
        {
            mCSVSeparator= pCSVSeparator;
        }

        LineNumberReader reader         = new LineNumberReader(pCSVData);
        List<String[]>   data           = new ArrayList<>();
        List<String>     row            = new ArrayList<>();
        String           csvMainString  =
                "  \\G(?:^|"+mCSVSeparator+")\n"+
                "  (?:\n"+
                "     # Either a double-quoted field...\n"+
                "     \" # field's opening quote\n"+
                "      (  (?> [^\"]*+ ) (?> \"\" [^\"]*+ )*+  )\n"+
                "     \" # field's closing quote\n"+
                "   # ... or ...\n"+
                "   |\n"+
                "     # ... some non-quote/non-comma text ...\n"+
                "     ( [^\""+mCSVSeparator+"]*+ )\n"+
                "  )\n";
        Pattern csvMainPattern= Pattern.compile(csvMainString, Pattern.COMMENTS);
        Matcher csvQuote      = CSV_QUOTE_PATTERN.matcher(EMPTY);
        Matcher csvMain       = csvMainPattern. matcher(EMPTY);

        String line= reader.readLine();
        while (line!=null)
        {
            // parse line
            row.clear();
            csvMain.reset(line);
            while ( csvMain.find() )
            {
                // we'll fill this with $1 or $2 . . .
                String field;
                String first = csvMain.group(2);
                if ( first != null )
                {
                    field = first;
                }
                else
                {
                    // if $1, must replace paired double-quotes with one double quote
                    csvQuote.reset(csvMain.group(1));
                    field = csvQuote.replaceAll("\"");
                }
                // we can now work with field, but we optimize empty fields first
                if (EMPTY.equals(field))
                {
                    field= EMPTY;
                }
                row.add(field);
            }
            data.add(row.toArray(new String[row.size()]));
            mColumns= Math.max(mColumns, row.size());

            // get next line
            mRows++;
            line= reader.readLine();
        }

        // remove first row for header (if applicable)
        if (pFirstRowIsHeader && mRows>1)
        {
            mFirstRowIsHeader= true;
            mRows--;
            mRowHeaders= data.remove(0);
        }
        else
        {
            mRowHeaders      = DUMMY;
            mFirstRowIsHeader= false;
        }

        // re-fit data into simpler (faster) data structures
        mData= new String[mRows][0];
        for (int i = 0; i < mRows; i++)
        {
            String[] rowContent= data.get(i);
            mData[i]= rowContent;
        }
        mFilteredRows           = mRows;
        mFilteredRowsIndexMapper= new int[mRows];
        for (int i = 0; i < mRows; i++)
        {
            mFilteredRowsIndexMapper[i]= i;
        }

        mLastFilter= null;
    }

    public int getRowCount()
    {
        return mFilteredRows;
    }

    public int getAllRowsCount()
    {
        return mRows;
    }

    public int getColumnCount()
    {
        return mColumns;
    }

    public Object getValueAt(final int pRowIndex, final int pColumnIndex)
    {
        int      realIndex= mFilteredRowsIndexMapper[pRowIndex];
        String[] row      = mData[realIndex];
        if (row.length>pColumnIndex)
            return row[pColumnIndex];
        else
            return "";
    }

    public String getColumnName(final int pColumnIndex)
    {
        if (mRowHeaders.length>pColumnIndex)
            return mRowHeaders[pColumnIndex];
        else
            return Integer.toString(pColumnIndex);
    }

    public boolean filter(final String pFilterExpression)
    {
        Pattern filter;
        try
        {
            filter= Pattern.compile(pFilterExpression,Pattern.CASE_INSENSITIVE);
        }
        catch (PatternSyntaxException ignored)
        {
            return false;
        }
        Matcher match= filter.matcher("");

        mFilteredRows= 0;
        mLastFilter  = pFilterExpression;
        for (int i = 0; i < mData.length; i++)
        {
            String[] row = mData[i];
            for (final String field : row)
            {
                match.reset(field);
                if (match.find())
                {
                    mFilteredRowsIndexMapper[mFilteredRows] = i;
                    mFilteredRows++;
                    break;
                }
            }
        }
        fireTableDataChanged();

        return true;
    }

    public void filterReset()
    {
        mFilteredRows= mRows;
        mLastFilter  = null;
        for (int i = 0; i < mRows; i++)
        {
            mFilteredRowsIndexMapper[i]= i;
        }
        fireTableDataChanged();
    }

    public void setFirstRowIsHeader(final boolean pFirstRowIsHeader)
    {
        // check if something really changed
        if (mFirstRowIsHeader==pFirstRowIsHeader)
        {
            return;
        }

        // check if we have enough data
        if (pFirstRowIsHeader && mRows<1)
        {
            return;
        }

        // copy stuff
        if (pFirstRowIsHeader)
        {
            mRows--;
            String[][] data= new String[mRows][0];
            System.arraycopy(mData, 1, data, 0, mRows);
            mRowHeaders= mData[0];
            mData      = data;
        }
        else
        {
            mRows++;
            String[][] data= new String[mRows][0];
            System.arraycopy(mData, 0, data, 1, mData.length);
            mData      = data;
            mData[0]   = mRowHeaders;
            mRowHeaders= DUMMY;
        }
        mFirstRowIsHeader= pFirstRowIsHeader;

        // re-filter if needed
        mFilteredRowsIndexMapper= new int[mRows];
        if (mLastFilter!=null)
        {
            filter(mLastFilter);
        }
        else
        {
            filterReset();
        }
    }
}
