package net.sf.cuf.csvview;

import com.jgoodies.looks.Options;
import net.sf.cuf.fw.AbstractSwingApplication;
import net.sf.cuf.model.ValueHolder;
import net.sf.cuf.model.ValueModel;
import net.sf.cuf.ui.SwingDecorator;
import net.sf.cuf.fw2.UserProperties;
import net.sf.cuf.ui.builder.SwingXMLBuilder;
import net.sf.cuf.fw.Application;
import net.sf.cuf.fw.Dc;
import net.sf.cuf.fw.EDTExceptionUtil;

import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Main class of the csvview application. This class manages
 * the start/stop process and is a creator and storage of the
 * application models and properties.
 */
public class Main extends AbstractSwingApplication
{
    /**
     * Private constructor, only main should create us.
     */
    private Main()
    {
        super();
    }

    /**
     * Start the application, first load properties, then I18N stuff,
     * then initialize the models and finally init the root Dc in the
     * EDT.
     * @param pArgs the arguments
     */
    public void doStart(final String[] pArgs)
    {
        super.doStart(pArgs);

        String version = "error report for csvview" + ", Version " +
                         Version.VERSION +
                         " (" +
                         Version.DATE +
                         ')';
        EDTExceptionUtil.setVersion('.' +AppData.APP_NAME, version);

        // load properties
        loadProperties();

        // init SwingDecorator (only needed by SortingTable)
        SwingDecorator.addBundle("i18n/sortingtable");


        /*
         * The following tweaks were taken from
         * http://wiki.java.net/bin/view/Javadesktop/SecretSwingProperties and
         * http://www.javadesktop.org/forums/thread.jspa?threadID=7311&tstart=255
         */

        // re-layout if component size changes
        Toolkit.getDefaultToolkit().setDynamicLayout(true);

        // do not erase the background during moving windows "gray fog"
        System.setProperty("sun.awt.noerasebackground", "true");

        // keep awt working set in memory (only relevant on Windows, needs Java 5.0)
        // for details see http://support.microsoft.com/default.aspx?scid=kb;en-us;293215
        // and http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=5043070
        System.setProperty("sun.awt.keepWorkingSetOnMinimize", "true");

        // use anti-alias text on Linux
        if (System.getProperty("os.name", "unknown").toLowerCase().contains("linux"))
        {
            System.setProperty("swing.aatext", "true");
        }


        /*
         * init our application domain models
         */

        // the PLAF is modeled with a LOV that maps a list of display items, to a list
        // of class names

        List<String> plafDisplayNames= new ArrayList<>();
        List<String> plafClassnNames = new ArrayList<>();

        plafDisplayNames.add("System");
        plafClassnNames.add(UIManager.getSystemLookAndFeelClassName());

        plafDisplayNames.add("Swing Metal");
        plafClassnNames.add("javax.swing.plaf.metal.MetalLookAndFeel");

        plafDisplayNames.add("JGoodies Windows");
        plafClassnNames.add("com.jgoodies.looks.windows.WindowsLookAndFeel");

        plafDisplayNames.add("JGoodies Plastic XP");
        plafClassnNames.add("com.jgoodies.looks.plastic.PlasticXPLookAndFeel");

        plafDisplayNames.add("JGoodies Plastic");
        plafClassnNames.add("com.jgoodies.looks.plastic.PlasticLookAndFeel");

        plafDisplayNames.add("JGoodies Plastic 3D");
        plafClassnNames.add("com.jgoodies.looks.plastic.Plastic3DLookAndFeel");

        String plafClassName= getProperty(AppData.PLAF_NAME_KEY, AppData.PLAF_DEFAULT_NAME);
        if (!plafClassnNames.contains(plafClassName))
        {
            plafClassName= AppData.PLAF_DEFAULT_NAME;
        }
        ValueModel<String> selectedPlafClassName= new ValueHolder<>(plafClassName);

        // store the PLAF item/key list as well as the current selection holder in our app model
        getAppModel().put(AppData.PLAF_ITEMS,         plafDisplayNames);
        getAppModel().put(AppData.PLAF_KEYS,          plafClassnNames);
        getAppModel().put(AppData.PLAF_SELECTION_KEY, selectedPlafClassName);

        // for JGoodies
        UIManager.put(Options.USE_SYSTEM_FONTS_APP_KEY, Boolean.TRUE);
        Options.setDefaultIconSize(new Dimension(18, 18));

        // set initial PLAF
        //noinspection EmptyCatchBlock
        try { UIManager.setLookAndFeel(selectedPlafClassName.getValue()); }
        catch (Exception ignored) { }

        // setup input source model
        List<String>  inputKeys= new ArrayList<>();
        inputKeys.add(AppData.CSV_SOURCE_FILE);
        inputKeys.add(AppData.CSV_SOURCE_URL);
        inputKeys.add(AppData.CSV_SOURCE_TREE);

        String csvSource = getProperty(AppData.CSV_SOURCE_KEY, AppData.CSV_SOURCE_FILE);
        if (!inputKeys.contains(csvSource))
        {
            csvSource= AppData.CSV_SOURCE_FILE;
        }
        ValueModel<String> inputSelection= new ValueHolder<>(csvSource);

        // store the input item/key list as well as the current selection holder in our app model
        // because we have no real items to display, we use the keys as items
        getAppModel().put(AppData.INPUT_ITEMS,     inputKeys);
        getAppModel().put(AppData.INPUT_KEYS,      inputKeys);
        getAppModel().put(AppData.INPUT_SELECTION, inputSelection);

        // setup encoding source model
        List<String> encodingKeys= Arrays.asList(AppData.ENCODING_DEFAULT_NAME, "UTF-8", "ISO-8859-1");
        String encoding = getProperty(AppData.ENCODING_NAME_KEY, AppData.ENCODING_DEFAULT_NAME);
        if (!encodingKeys.contains(encoding))
        {
            encoding= AppData.ENCODING_DEFAULT_NAME;
        }
        ValueModel<String> encodingSelection= new ValueHolder<>(encoding);

        // store the encoding item/key list as well as the current selection holder in our app model
        // because we have no real items to display, we use the keys as items
        getAppModel().put(AppData.ENCODING_ITEMS,     encodingKeys);
        getAppModel().put(AppData.ENCODING_KEYS,      encodingKeys);
        getAppModel().put(AppData.ENCODING_SELECTION, encodingSelection);

        // we need the application properties in the XML to create
        // the PropertiesAdapter ValueModels, we also merge in all AppModel's
        Map<String, ? super Object> nonVisual= new HashMap<>();
        nonVisual.put("AppProperties", getProperties());
        nonVisual.putAll(getAppModel());

        /*
         * load GUI and other objects
         */
        String      fileName= AppData.XML_UI_NAME;
        ClassLoader loader  = getClass().getClassLoader();
        if (loader==null)
        {
            loader= ClassLoader.getSystemClassLoader();
        }
        InputStream     input  = loader.getResourceAsStream(fileName);
        SwingXMLBuilder builder= SwingXMLBuilder.create(input, nonVisual);

        // extract value model's that we need in our AppModel Map
        getAppModel().put(AppData.FIRST_ROW_IS_HEADER_VM,
                          builder.getNonVisualObject(AppData.FIRST_ROW_IS_HEADER_VM));
        getAppModel().put(AppData.CSV_SEPARATOR_VM,
                          builder.getNonVisualObject(AppData.CSV_SEPARATOR_VM));
        getAppModel().put(AppData.ENCODING_SELECTION,
                          builder.getNonVisualObject(AppData.ENCODING_SELECTION));
        getAppModel().put(AppData.COPY_WHOLE_ROW_VM,
                          builder.getNonVisualObject(AppData.COPY_WHOLE_ROW_VM));

        /*
         * init Dc's/Pc's
         */
        final Dc  dc  = (Dc)builder.getNonVisualObject("MainDc");
        final Map<String, ? super  Object> args= new HashMap<>();
        args.put(Application.APPLICATION_KEY, this);

        // to be consistent with the default thread behaviour, we init the Dc in the EDT
        SwingUtilities.invokeLater(() -> dc.init(null, args));
    }

    /**
     * Stop the application, saves the properties and then calls System.exit.
     */
    public void doStop()
    {
        saveProperties();
        //noinspection CallToSystemExit
        System.exit(0);
    }

    /**
     * Small helper method to load the properties.
     */
    private void loadProperties()
    {
        try
        {
            mAppProperties = UserProperties.read(null, null, AppData.CSVVIEW_USER_PROPERTIES);
        }
        catch (Exception e)
        {
            //noinspection UseOfSystemOutOrSystemErr
            System.err.println("will ignore loadProperties problems: " + e);
        }
    }

    /**
     * Save all properties.
     */
    private void saveProperties()
    {
        try
        {
            /*
             * persist our application domain models, the rest are handled via
             * PropertiesAdapter's
             */
            ValueModel<String> inputSelection= (ValueModel<String>)getAppModel().get(AppData.INPUT_SELECTION);
            String             csvSource     = inputSelection.getValue();
            setProperty(AppData.CSV_SOURCE_KEY, csvSource);

            ValueModel<String> plafSelection= (ValueModel<String>)getAppModel().get(AppData.PLAF_SELECTION_KEY);
            String             defaultLAF   = plafSelection.getValue();
            setProperty(AppData.PLAF_NAME_KEY, defaultLAF);

            ValueModel<String> encodingSelection= (ValueModel<String>)getAppModel().get(AppData.ENCODING_SELECTION);
            String             encoding   = encodingSelection.getValue();
            setProperty(AppData.ENCODING_NAME_KEY, encoding);

            // do the save
            UserProperties.write(mAppProperties, null, AppData.CSVVIEW_USER_PROPERTIES);
        }
        catch (Exception e)
        {
            //noinspection UseOfSystemOutOrSystemErr
            System.err.println("will ignore saveProperties problems: " + e);
        }
    }

    /**
     * Main entry point to the application.
     * @param pArgs the arguments
     */
    @SuppressWarnings({"MethodNamesDifferingOnlyByCase"})
    public static void main(final String[] pArgs)
    {
        Application app= new Main();
        app.doStart(pArgs);
    }
}
