package net.sf.cuf.csvview.browse;

import net.sf.cuf.csvview.AppData;
import net.sf.cuf.csvview.util.CSVFileFilter;
import net.sf.cuf.csvview.util.FilteredTableModel;
import net.sf.cuf.csvview.util.TableRowCopyAction;
import net.sf.cuf.model.ValueModel;
import net.sf.cuf.ui.DispatcherAction;
import net.sf.cuf.ui.builder.SwingXMLBuilder;
import net.sf.cuf.fw.Application;
import net.sf.cuf.fw.Dc;
import net.sf.cuf.fw.Pc;
import net.sf.cuf.ui.table.ContextMenuAction;
import net.sf.cuf.ui.table.ContextMenuActionHideColumn;
import net.sf.cuf.ui.table.ContextMenuActionOptimalColumnWidth;
import net.sf.cuf.ui.table.ContextMenuActionSortColumn;
import net.sf.cuf.ui.table.ContextMenuActionSortDialog;
import net.sf.cuf.ui.table.ContextMenuActionVisibilityDialog;
import net.sf.cuf.ui.table.OptimalColumnWidthSupport;
import net.sf.cuf.ui.table.TableContextMenu;
import net.sf.cuf.ui.table.TableProperties;
import net.sf.cuf.xfer.Response;

import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.TransferHandler;
import javax.swing.UIManager;
import javax.swing.event.ChangeEvent;
import javax.swing.table.TableModel;
import java.awt.Color;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import java.util.Map;

/**
 * Presentation component of the main panel.
 */
public class MainPc implements Pc, SwingXMLBuilder.Backlink
{
    /** argument key for init */
    static final String TABLE_ARG= "tableModel";

    /** our mapping helper */
    private SwingXMLBuilder     mBuilder;
    /** our dialog component peer */
    private MainDc              mDc;
    /** our application */
    private Application         mApp;
    /** the data we are displaying */
    private FilteredTableModel  mTableModel;
    /** null or the last filtered expression */
    private String              mOldFilterText;
    /** null or the "normal" background of our filter input */
    private Color               mOldBackground;
    /** null or the "normal" tooltip of our filter input */
    private String              mOldTooltip;
    /** flag if we are in auto filter mode */
    private boolean             mAutoFilter;
    /** replace row marker for external program execution */
    private static final String REPLACE_ROW = "%r";
    /** replace cell marker for external program execution */
    private static final String REPLACE_CELL = "%s";

    /**
     * When a SwingXMLBuilder creates an object of ourselves, we want to
     * know that builder.
     * @param pBuilder the builder that created this object
     */
    public void setSwingXMLBuilder(final SwingXMLBuilder pBuilder)
    {
        mBuilder= pBuilder;
    }

    /**
     * This method connects the widgets to models and uses states/state adapters
     * and ValueModel adapters to minimize the event handling code.
     * @param pDc our peer dialog component
     * @param pArgs the arguments
     */
    public void init(final Dc pDc, final Map<String, ? super Object> pArgs)
    {
        mDc        = (MainDc) pDc;
        mTableModel= (FilteredTableModel)pArgs.get(TABLE_ARG);
        mApp       = (Application)       pArgs.get(Application.APPLICATION_KEY);

        // bind ourselves to the first row is header ValueModel
        ValueModel model= (ValueModel)mApp.getAppModel().get(AppData.FIRST_ROW_IS_HEADER_VM);
        model.onChangeSend(this, "setFirstRowIsHeader");

        // bind ourselves to the PLAF changes
        ValueModel plafSelection= (ValueModel)mApp.getAppModel().get(AppData.PLAF_SELECTION_KEY);
        plafSelection.onChangeSend(this, "plafChanged");


        // set our table model, but don't filter (yet)
        setModelIntern(mTableModel, false);

        // add excel-like width adjustment and various other cool stuff
        JTable table= (JTable)mBuilder.getComponentByName("Frame/Panel/ScrollPane/CSVTabelle");
        new OptimalColumnWidthSupport(table);
        ContextMenuAction[] actions =
                {
                    new ContextMenuActionSortColumn(true),
                    new ContextMenuActionSortColumn(false),
                    new ContextMenuActionHideColumn(),
                    new ContextMenuActionOptimalColumnWidth(),
                    TableContextMenu.SEPARATOR,
                    new ContextMenuActionSortDialog(),
                    new ContextMenuActionVisibilityDialog()
                };
        new TableContextMenu(table, TableContextMenu.CONTEXT_MENU_IN_TABLE_AND_PARENT, actions);
        // we're inside a scrollbar, so don't auto resize
        table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);

        // mouse listener for external program execution
        table.addMouseListener(new MouseAdapter()
        {
            public void mouseClicked(final MouseEvent e)
            {
                if (e.getClickCount()>1)
                {
                    doubleClicked();
                }
            }
        });

        // add our own copy handler for the table
        ActionMap  map       = table.getActionMap();
        ValueModel copyHolder= (ValueModel) mApp.getAppModel().get(AppData.COPY_WHOLE_ROW_VM);
        map.put(TransferHandler.getCopyAction().getValue(Action.NAME),
                new TableRowCopyAction(copyHolder));

        // set initial status text line
        updateStatusLabel();

        // set filter status
        mAutoFilter= Boolean.valueOf(
                (mApp.getProperty(AppData.DEFAULT_AUTOFILTER_KEY, "true")));
        DispatcherAction autoFilter= (DispatcherAction)mBuilder.getNonVisualObject("AutoFilter");
        autoFilter.setSelected(mAutoFilter);

        // remember the textfield background+tooltip for our error-handling
        mOldFilterText= null;
        JTextField filter= (JTextField)mBuilder.getComponentByName("Frame/Panel/Input");
        mOldBackground= filter.getBackground();
        mOldTooltip   = filter.getToolTipText();
        String filterText= mApp.getProperty(AppData.DEFAULT_FILTER_KEY, "");
        if (!"".equals(filterText))
        {
            filter.setText(filterText);
            filter.selectAll();
        }

        // make the frame visible
        JFrame frame= (JFrame)mBuilder.getContainerByName("Frame");
        int x     = Integer.parseInt(AppData.DEFAULT_X);
        int y     = Integer.parseInt(AppData.DEFAULT_Y);
        int width = Integer.parseInt(AppData.DEFAULT_WIDTH);
        int height= Integer.parseInt(AppData.DEFAULT_WIDTH);
        //noinspection EmptyCatchBlock
        try
        {
            x     = Integer.parseInt(mApp.getProperty(AppData.DEFAULT_X_KEY,      AppData.DEFAULT_X     ));
            y     = Integer.parseInt(mApp.getProperty(AppData.DEFAULT_Y_KEY,      AppData.DEFAULT_Y     ));
            width = Integer.parseInt(mApp.getProperty(AppData.DEFAULT_WIDTH_KEY,  AppData.DEFAULT_WIDTH ));
            height= Integer.parseInt(mApp.getProperty(AppData.DEFAULT_HEIGHT_KEY, AppData.DEFAULT_HEIGHT));
        }
        catch (NumberFormatException ignored)
        {
        }
        frame.setLocation(x, y);
        frame.setSize    (width, height);
        frame.setVisible (true);
    }

    /**
     * Callback for quitting the application. We store the current frame size/position
     * and the table sorting/ordering/column width in the applications properties and
     * ask our dialog component to quit.
     */
    public void quit()
    {
        // save current size etc.
        JFrame frame = (JFrame)mBuilder.getContainerByName("Frame");
        int    width = frame.getWidth();
        int    height= frame.getHeight();
        int    x     = frame.getX();
        int    y     = frame.getY();
        mApp.setProperty(AppData.DEFAULT_X_KEY,      Integer.toString(x));
        mApp.setProperty(AppData.DEFAULT_Y_KEY,      Integer.toString(y));
        mApp.setProperty(AppData.DEFAULT_WIDTH_KEY,  Integer.toString(width));
        mApp.setProperty(AppData.DEFAULT_HEIGHT_KEY, Integer.toString(height));

        JTable table= (JTable)mBuilder.getComponentByName("Frame/Panel/ScrollPane/CSVTabelle");
        TableProperties.store(table, mApp.getProperties(), AppData.DEFAULT_TABLE_SETTINGS);

        mDc.quit();
    }

    /**
     * Callback for opening a file. We show a file chooser and forward the
     * open to our dialog component, we also store the current directory
     * in our application properties.
     */
    public void openFile()
    {
        JFrame       frame      = (JFrame)mBuilder.getContainerByName("Frame");
        String       dir        = mApp.getProperty(AppData.DEFAULT_DIR_KEY,
                                                   AppData.DEFAULT_DIR);
        JFileChooser fileChooser= new JFileChooser(dir);
        fileChooser.addChoosableFileFilter(new CSVFileFilter());

        int back= fileChooser.showOpenDialog(frame);
        if (back== JFileChooser.APPROVE_OPTION)
        {
            File file= fileChooser.getSelectedFile();
            mDc.openFile(file);
            mApp.setProperty(AppData.DEFAULT_DIR_KEY, file.getParent());
        }
    }

    /**
     * Callback for opening a tree. We show a file chooser and forward the
     * open to our dialog component, we also store the current directory
     * in our application properties.
     */
    public void openTree()
    {
        JFrame       frame      = (JFrame)mBuilder.getContainerByName("Frame");
        String       dir        = mApp.getProperty(AppData.DEFAULT_DIR_KEY,
                                                   AppData.DEFAULT_DIR);
        JFileChooser fileChooser= new JFileChooser(dir);
        fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);

        int back= fileChooser.showOpenDialog(frame);
        if (back== JFileChooser.APPROVE_OPTION)
        {
            File file= fileChooser.getSelectedFile();
            mDc.openTree(file);
            mApp.setProperty(AppData.DEFAULT_DIR_KEY, file.getParent());
        }
    }

    /**
     * Called by our dialog component to publish new table data.
     * @param pModel the new table model
     */
    public void setModel(final FilteredTableModel pModel)
    {
        setModelIntern(pModel, true);
    }

    /**
     * Small helper to set the model and optionally filter. We
     * also load the table sorting etc. from our application properties.
     * @param pModel the new table model
     * @param pDoFilter true if we should filter
     */
    private void setModelIntern(final FilteredTableModel pModel, final boolean pDoFilter)
    {
        mTableModel= pModel;
        JTable table= (JTable)mBuilder.getComponentByName("Frame/Panel/ScrollPane/CSVTabelle");
        table.setModel(mTableModel);
        if (pDoFilter)
        {
            doFilter();
        }
        TableProperties.load(table, mApp.getProperties(), AppData.DEFAULT_TABLE_SETTINGS);
    }

    /**
     * Callback of the text field on each caret movement. Depending if we
     * are in auto filter mode and if the filter text changed we re-filter.
     */
    public void filter()
    {
        JTextField filter    = (JTextField)mBuilder.getComponentByName("Frame/Panel/Input");
        String     filterText= filter.getText();
        if (!mAutoFilter || filterText.equals(mOldFilterText))
        {
            return;
        }
        doFilter();
    }

    /**
     * Callback from our text field or the search button. We re-filter the data.
     */
    public void search()
    {
        doFilter();
    }

    /**
     * Helper method that does the filtering. If the filter failed it marks
     * the textfield with a color/tooltip.
     */
    private void doFilter()
    {
        JTextField filter    = (JTextField)mBuilder.getComponentByName("Frame/Panel/Input");
        String     filterText= filter.getText();
        mOldFilterText= filterText;

        if ("".equals(filterText))
        {
            mTableModel.filterReset();
        }
        else
        {
            boolean filterOK= mTableModel.filter(filterText);
            if (filterOK)
            {
                filter.setBackground (mOldBackground);
                filter.setToolTipText(mOldTooltip);
            }
            else
            {
                filter.setBackground(Color.cyan);
                filter.setToolTipText("Eingabe ist keine gueltiger regulaerer Ausdruck");
            }
        }
        mApp.setProperty(AppData.DEFAULT_FILTER_KEY, filterText);

        updateStatusLabel();
    }

    /**
     * Small helper method that updates the status label.
     */
    public void updateStatusLabel()
    {
        JLabel label   = (JLabel)mBuilder.getComponentByName("Frame/Panel/Status");
        int    allRows = mTableModel.getAllRowsCount();
        int    rows    = mTableModel.getRowCount();
        JTable table   = (JTable)mBuilder.getComponentByName("Frame/Panel/ScrollPane/CSVTabelle");
        int    selected= table.getSelectedRowCount();
        label.setText(allRows + " Zeilen, davon " + rows + " gefiltered" + " und " + selected + " selektiert");
    }

    /**
     * Callback from the auto-filter checkbox. We filter if it was switched
     * on and store the state in the application properties.
     */
    public void autoFilter()
    {
        mOldFilterText= null;
        DispatcherAction autoFilter= (DispatcherAction)mBuilder.getNonVisualObject("AutoFilter");
        boolean          isSelected= autoFilter.isSelected();
        if (isSelected)
        {
            mAutoFilter= true;
            doFilter();
        }
        else
        {
            mAutoFilter= false;
        }
        mApp.setProperty(AppData.DEFAULT_AUTOFILTER_KEY, Boolean.toString(isSelected));
    }

    /**
     * Callback method from our dialog component to show an error.
     * @param pResponse the response containing the error
     */
    public void showLoadError(final Response<?> pResponse)
    {
        StringBuilder sb = new StringBuilder();
        sb.append("BAD NEWS!\n");
        sb.append("Während des Ladens ging etwas schief:\n");
        if (pResponse.getError()==null)
        {
            sb.append("Aber wir wissen nicht was :-(");
        }
        else
        {
            sb.append(pResponse.getError());
        }

        JFrame frame= (JFrame)mBuilder.getContainerByName("Frame");
        JOptionPane.showMessageDialog(frame,
                                      sb.toString(),
                                      "ERROR",
                                      JOptionPane.ERROR_MESSAGE);
    }

    /**
     * ValueModel callback if the first-row-is-header value changed.
     * @param e event containing the ValueModel
     */
    public void setFirstRowIsHeader(final ChangeEvent e)
    {
        ValueModel model           = ((ValueModel)e.getSource());
        boolean    firstRowIsHeader= model.booleanValue();

        mTableModel.setFirstRowIsHeader(firstRowIsHeader);
        JTable table= (JTable)mBuilder.getComponentByName("Frame/Panel/ScrollPane/CSVTabelle");
        table.createDefaultColumnsFromModel();
    }

    /**
     * Table callback if a cell is double clicked, may trigger the execution
     * of an external program.
     */
    private void doubleClicked()
    {
        JTable table= (JTable)mBuilder.getComponentByName("Frame/Panel/ScrollPane/CSVTabelle");
        int     selectedRow   = table.getSelectedRow();
        int     selectedColumn= table.getSelectedColumn();
        boolean exeActive     = mApp.getProperty(AppData.EXE_PROGRAM_ACTIVE_KEY,
                                                 AppData.EXE_PROGRAM_ACTIVE).
                                equals(Boolean.TRUE.toString());

        if (exeActive && (selectedRow>-1) && (selectedColumn>-1))
        {
            int realRow   = selectedRow;
            int realColumn= table.convertColumnIndexToModel(selectedColumn);
            String     cellData= table.getModel().getValueAt(realRow, realColumn).toString();
            String     template= mApp.getProperty(AppData.EXE_PROGRAM_NAME_KEY, "");
            String     command = replaceAll(template, REPLACE_CELL, cellData);

            if (command.contains(REPLACE_ROW))
            {
                TableModel    model= table.getModel();
                StringBuilder sb = new StringBuilder();
                int           n    = model.getColumnCount();
                for (int i= 0; i < n; i++)
                {
                    sb.append(table.getValueAt(realRow,i));
                    if (i<n-1)
                    {
                        sb.append(',');
                    }
                }
                command= replaceAll(command, REPLACE_ROW, sb.toString());
            }
            mDc.exeProgram(command);
        }
    }

    /**
     * Replace all pReplaceWhat Strings in pData with pReplaceWith and return
     * the result. Don't uses the regexp support, because this would mean that
     * we have to escape pReplaceWith!
     * @param pData data to operate on
     * @param pReplaceWhat what to replace
     * @param pReplaceWith replacement
     * @return replaced string
     */
    private static String replaceAll(final String pData, final String pReplaceWhat, final String pReplaceWith)
    {
        StringBuilder sb = new StringBuilder();
        int           start= 0;
        int           index= pData.indexOf(pReplaceWhat, start);
        while (index>=0)
        {
            sb.append(pData.substring(start, index));
            sb.append(pReplaceWith);
            start= index+pReplaceWhat.length();
            index= pData.indexOf(pReplaceWhat, start);
        }
        if (start<pData.length())
        {
            sb.append(pData.substring(start));
        }
        return sb.toString();
    }

    /**
     * ValueModel callback if the user changed the wanted look-and-feel.
     * We update all dialogs/our frame after setting the new look-and-feel.
     * @param e not used
     */
    public void plafChanged(final ChangeEvent e)
    {
        ValueModel plafSelection= (ValueModel)mApp.getAppModel().get(AppData.PLAF_SELECTION_KEY);
        String plafName= (String)plafSelection.getValue();
        //noinspection EmptyCatchBlock
        try
        {
            UIManager.setLookAndFeel(plafName);
            SwingUtilities.updateComponentTreeUI(mBuilder.getContainerByName("Frame"));
            SwingUtilities.updateComponentTreeUI(mBuilder.getContainerByName("Frame/OptionsDialog"));
            SwingUtilities.updateComponentTreeUI(mBuilder.getContainerByName("Frame/AboutDialog"));
        }
        catch (Exception ignored)
        {
        }
    }
}
