package net.sf.cuf.csvview.browse;

import net.sf.cuf.csvview.AppData;
import net.sf.cuf.csvview.util.CSVTableModel;
import net.sf.cuf.csvview.util.DataSourceChanged;
import net.sf.cuf.csvview.util.FilteredTableModel;
import net.sf.cuf.appevent.AppEvent;
import net.sf.cuf.appevent.AppEventManager;
import net.sf.cuf.appevent.AppEventSupport;
import net.sf.cuf.appevent.AppEventUtil;
import net.sf.cuf.appevent.BindEvent;
import net.sf.cuf.model.ValueModel;
import net.sf.cuf.ui.builder.SwingXMLBuilder;
import net.sf.cuf.fw.Application;
import net.sf.cuf.fw.Dc;
import net.sf.cuf.xfer.SwingDispatcher;
import net.sf.cuf.xfer.Dispatch;
import net.sf.cuf.xfer.Response;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

/**
 * Dialog component of the main panel.
 */
public class MainDc implements Dc, AppEventSupport, SwingXMLBuilder.Backlink
{
    /** our presentation component peer */
    private MainPc          mPc;
    /** our mapping helper  */
    private SwingXMLBuilder mBuilder;
    /** helper for the AppEvent stuff  */
    private AppEventManager mAppEventManger;
    /** our application  */
    private Application     mApp;
    /** our Dispatcher */
    private Dispatch        mDispatcher;

    /**
     * When a SwingXMLBuilder creates an object of ourselves, we want to
     * know that builder.
     * @param pBuilder the builder that created this object
     */
    public void setSwingXMLBuilder(final SwingXMLBuilder pBuilder)
    {
        mBuilder= pBuilder;
    }

    /**
     * Process an AppEvent.
     * @param pAppEvent event that should be routed/processes
     */
    public void postAppEvent(final AppEvent pAppEvent)
    {
        AppEventUtil.postAppEvent(mAppEventManger, pAppEvent);
    }

    /**
     * This method initializes the AppEvent handling, loads the first
     * data, initializes  all sub-dialog components and then initializes
     * its presentation component.
     * @param pParent our parent dialog component, is null because we have no parent
     * @param pArgs the arguments
     */
    public void init(final Dc pParent, final Map<String, ? super Object>  pArgs)
    {
        // create a dispatcher
        mDispatcher= new SwingDispatcher();

        // setup app event handling
        mAppEventManger= new AppEventManager();

        BindEvent bindEvent= new BindEvent(this, "dataSourceChanged", DataSourceChanged.class, 0);
        postAppEvent(bindEvent);

        // set up link to our app
        mApp= (Application) pArgs.get(Application.APPLICATION_KEY);

        // first initialize all our sub-Dc's
        Dc dc= (Dc) mBuilder.getNonVisualObject("OptionDc");
        dc.init(this, pArgs);
        dc= (Dc) mBuilder.getNonVisualObject("AboutDc");
        dc.init(this, pArgs);

        // load first data
        LoadDataRequest    loadDataRequest= getCurrentRequest();
        Response           response       = mDispatcher.syncExecute(loadDataRequest);
        FilteredTableModel tableModel;
        if (response.isValid())
        {
            tableModel= (FilteredTableModel) response.getResult();
        }
        else
        {
            tableModel= new CSVTableModel();
        }

        // now initialize our Pc
        mPc= (MainPc)mBuilder.getNonVisualObject("MainPc");
        Map<String, ? super Object> map= new HashMap<String, Object>(pArgs);
        map.put(MainPc.TABLE_ARG, tableModel);
        mPc.init(this, map);
    }

    /**
     * Small helper to generate the load request depending on the current input selection.
     * @return the current load request, never null
     */
    private LoadDataRequest getCurrentRequest()
    {
        ValueModel<?> firstRowIsHeaderModel = (ValueModel<?>)mApp.getAppModel().get(AppData.FIRST_ROW_IS_HEADER_VM);
        boolean    firstRowIsHeader         = firstRowIsHeaderModel.booleanValue();
        ValueModel<String> csvSeparatorModel= (ValueModel<String>)mApp.getAppModel().get(AppData.CSV_SEPARATOR_VM);
        String     csvSeparator             = csvSeparatorModel.getValue();
        ValueModel<String> encodingModel    = (ValueModel<String>)mApp.getAppModel().get(AppData.ENCODING_SELECTION);
        String     encoding                 = encodingModel.getValue();
        ValueModel<String> inputSelection   = (ValueModel<String>)mApp.getAppModel().get(AppData.INPUT_SELECTION);
        String     csvSource                = inputSelection.getValue();

        // generate request depending on the chosen csv source
        LoadDataRequest loadDataRequest;
        if (csvSource.equals(AppData.CSV_SOURCE_URL))
        {
            String urlName = mApp.getProperty(AppData.DEFAULT_URL_NAME_KEY,
                                              AppData.DEFAULT_URL_NAME);
            loadDataRequest= new LoadURLRequest(urlName, firstRowIsHeader, csvSeparator, encoding);
        }
        else if (csvSource.equals(AppData.CSV_SOURCE_TREE))
        {
            String treeName = mApp.getProperty(AppData.DEFAULT_TREE_NAME_KEY,
                                               AppData.DEFAULT_TREE_NAME);
            loadDataRequest= new LoadTreeRequest(treeName);
        }
        else
        {
            String fileName= mApp.getProperty(AppData.DEFAULT_FILE_NAME_KEY,
                                              AppData.DEFAULT_FILE_NAME);
            loadDataRequest = new LoadFileRequest(new File(fileName), firstRowIsHeader, csvSeparator, encoding);
        }
        return loadDataRequest;
    }

    /**
     * Called by our presentation component to load a file, we also set the
     * file name as the new default file name.
     * @param pFile the file we should load
     */
    public void openFile(final File pFile)
    {
        mApp.setProperty(AppData.DEFAULT_FILE_NAME_KEY, pFile.getPath());

        ValueModel<String> inputSelection= (ValueModel<String>)mApp.getAppModel().get(AppData.INPUT_SELECTION);
        inputSelection.setValue(AppData.CSV_SOURCE_FILE);

        LoadDataRequest loadDataRequest= getCurrentRequest();
        loadDataRequest.setDispatchTarget(this, "loadResponse");
        mDispatcher.asyncDispatchInEDT(loadDataRequest);
    }

    /**
     * Called by our presentation component to load a tree, we also set the
     * tree name as the new default tree name.
     * @param pFile the tree root we should load
     */
    public void openTree(final File pFile)
    {
        mApp.setProperty(AppData.DEFAULT_TREE_NAME_KEY, pFile.getPath());

        ValueModel<String> inputSelection= (ValueModel<String>)mApp.getAppModel().get(AppData.INPUT_SELECTION);
        inputSelection.setValue(AppData.CSV_SOURCE_TREE);

        LoadDataRequest loadDataRequest= getCurrentRequest();
        loadDataRequest.setDispatchTarget(this, "loadResponse");
        mDispatcher.asyncDispatchInEDT(loadDataRequest);
    }

    /**
     * Called by the Dispatcher after the request dispatched in openFile,
     * openTree or dataSourceChanged finished.
     * @param pResponse the response, containing either a FilteredTableModel as result
     *                  or an Exception as an error.
     */
    public void loadResponse(final Response pResponse)
    {
        if (pResponse.isValid())
        {
            FilteredTableModel newModel= (FilteredTableModel)pResponse.getResult();
            mPc.setModel(newModel);
        }
        else
        {
            mPc.showLoadError(pResponse);
        }
    }

    /**
     * Called by our presentation component to quit the application.
     */
    public void quit()
    {
        mApp.doStop();
    }

    /**
     * AppEvent callback for the DataSourceChanged event, we re-load
     * the current data.
     * @param pDataSourceChanged not used
     */
    @SuppressWarnings({"UnusedParameters"})
    public void dataSourceChanged(final DataSourceChanged pDataSourceChanged)
    {
        LoadDataRequest loadDataRequest= getCurrentRequest();
        loadDataRequest.setDispatchTarget(this, "loadResponse");
        mDispatcher.asyncDispatchInEDT(loadDataRequest);
    }

    /**
     * Called by our presentation component to execute an external command.
     * @param pCommand the command string to execute
     */
    public void exeProgram(final String pCommand)
    {
        //noinspection EmptyCatchBlock
        try
        {
            //noinspection CallToRuntimeExec
            Runtime.getRuntime().exec(pCommand);
        }
        catch (Exception ignored)
        {
        }
    }
}
