package net.sf.cuf.csvview.option;

import net.sf.cuf.csvview.AppData;
import net.sf.cuf.csvview.util.CSVFileFilter;
import net.sf.cuf.csvview.util.DataSourceChanged;
import net.sf.cuf.csvview.util.EXEFileFilter;
import net.sf.cuf.appevent.AppEvent;
import net.sf.cuf.appevent.AppEventSupport;
import net.sf.cuf.appevent.AppEventUtil;
import net.sf.cuf.model.BufferedValueHolder;
import net.sf.cuf.model.ValueModel;
import net.sf.cuf.ui.builder.SwingXMLBuilder;
import net.sf.cuf.fw.Application;
import net.sf.cuf.fw.Dc;
import net.sf.cuf.fw.Pc;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JTextField;
import javax.swing.filechooser.FileFilter;
import java.awt.Rectangle;
import java.io.File;
import java.io.IOException;
import java.util.Map;

/**
 * OptionPc is the presentation component for the options panel
 * of our application.
 * Most of the work is done via connecting ValueModel's in the init
 * method to the widgets, the ok, cancel and defaults method mainly
 * act on the ValueModel's.
 */
public class OptionPc implements Pc, AppEventSupport, SwingXMLBuilder.Backlink
{
    /** our mapping helper */
    private SwingXMLBuilder     mBuilder;
    /** our application */
    private Application         mApp;
    /** our dialog component peer */
    private OptionDc            mDc;
    /** trigger for the ValueModels */
    private ValueModel          mTrigger;
    /** buffers the input selection */
    private BufferedValueHolder mInputSelectionBuffer;
    /** buffers the file name */
    private BufferedValueHolder mFileNameBuffer;
    /** buffers the URL name */
    private BufferedValueHolder mURLNameBuffer;
    /** buffers the tree name */
    private BufferedValueHolder mTreeNameBuffer;
    /** buffers the first-row-is-header model */
    private BufferedValueHolder mFirstRowIsHeaderBuffer;
    /** buffers the first-row-is-header model */
    private BufferedValueHolder mCSVSeparatorBuffer;
    /** buffers if a double click executes a program */
    private ValueModel          mExeActiveBuffer;
    /** buffers the program name */
    private ValueModel          mExeNameBuffer;
    /** buffers the PLAF selection */
    private ValueModel          mPLAFSelectionBuffer;
    /** buffers the encoding selection */
    private BufferedValueHolder mEncodingSelectionBuffer;
    /** buffers the copy row selection */
    private ValueModel          mCopyRowSelectionBuffer;

    /** base name of the options */
    private static final String BASE= "Frame/OptionsDialog/Panel/Options/";

    /**
     * When a SwingXMLBuilder creates an object of ourselves, we want to
     * know that builder.
     * @param pBuilder the builder that created this object
     */
    public void setSwingXMLBuilder(final SwingXMLBuilder pBuilder)
    {
        mBuilder= pBuilder;
    }

    /**
     * Process an AppEvent.
     * @param pAppEvent event that should be routed/processes
     */
    public void postAppEvent(final AppEvent pAppEvent)
    {
        AppEventUtil.postAppEvent(mDc,pAppEvent);
    }

    /**
     * This method connects the widgets to ValueModel's, all states/state adapter
     * init was already done in the XML.
     * @param pDc our peer dialog component
     * @param pArgs the arguments
     */
    public void init(final Dc pDc, final Map<String, ? super Object>  pArgs)
    {
        mDc = (OptionDc)pDc;
        mApp= (Application)pArgs.get(Application.APPLICATION_KEY);

        // get our value models from the builder
        mTrigger         = (ValueModel)         mBuilder.getNonVisualObject("OptionPc/Trigger");
        mFileNameBuffer  = (BufferedValueHolder)mBuilder.getNonVisualObject("OptionPc/FileNameBuffer");
        mURLNameBuffer   = (BufferedValueHolder)mBuilder.getNonVisualObject("OptionPc/URLNameBuffer");
        mTreeNameBuffer  = (BufferedValueHolder)mBuilder.getNonVisualObject("OptionPc/TreeNameBuffer");
        mFirstRowIsHeaderBuffer = (BufferedValueHolder)mBuilder.getNonVisualObject("OptionPc/FirstRowIsHeaderBuffer");
        mCSVSeparatorBuffer     = (BufferedValueHolder)mBuilder.getNonVisualObject("OptionPc/CSVSeparatorBuffer");
        mExeActiveBuffer        = (ValueModel)mBuilder.getNonVisualObject("OptionPc/ExeActiveBuffer");
        mExeNameBuffer          = (ValueModel)mBuilder.getNonVisualObject("OptionPc/ExeNameBuffer");
        mCopyRowSelectionBuffer = (ValueModel)mBuilder.getNonVisualObject("OptionPc/CopyRowSelectionBuffer");
        mPLAFSelectionBuffer    = (ValueModel)mBuilder.getNonVisualObject("OptionPc/PlafSelectionBuffer");
        mEncodingSelectionBuffer= (BufferedValueHolder)mBuilder.getNonVisualObject("OptionPc/EncodingSelectionBuffer");
        mInputSelectionBuffer   = (BufferedValueHolder)mBuilder.getNonVisualObject("OptionPc/InputSelectionBuffer");
    }

    /**
     * Show an options dialog.
     */
    public void show()
    {
        // we want a modal dialog
        JDialog optionsDialog= (JDialog) mBuilder.getContainerByName("Frame/OptionsDialog");
        optionsDialog.setModal(true);

        // set OK as default button
        JButton okButton= (JButton) mBuilder.getComponentByName("Frame/OptionsDialog/Panel/ButtonBar/OK");
        optionsDialog.getRootPane().setDefaultButton(okButton);

        // fill in data (must be done before pack!) by triggering the read of
        // the subjects in our buffering value models
        mTrigger.setValueForced(Boolean.FALSE);

        // calculate size
        optionsDialog.pack();

        // center dialog with respect to its parent
        Rectangle parentBounds= optionsDialog.getParent().getBounds();
        int deltaX = (parentBounds.width  - optionsDialog.getBounds().width) / 2;
        int deltaY = (parentBounds.height - optionsDialog.getBounds().height) / 2;
        optionsDialog.setLocation(parentBounds.x + deltaX, parentBounds.y + deltaY);

        // make dialog visible
        optionsDialog.setVisible(true);
    }

    /**
     * Act on "OK" in the options dialog.
     */
    public void ok()
    {
        // hide dialog
        JDialog optionsDialog= (JDialog) mBuilder.getContainerByName("Frame/OptionsDialog");
        optionsDialog.setVisible(false);

        /*
         * map our dialog state memory to the application state
         */

        // depending if the source or the details of a source changed, trigger re-load
        boolean sourceChanged= false;
        if (mInputSelectionBuffer.isBufferEqual() && mEncodingSelectionBuffer.isBufferEqual())
        {
            // the type didn't changed, check the details
            ValueModel<String> inputSelection= (ValueModel<String>)mApp.getAppModel().get(AppData.INPUT_SELECTION);
            String             csvSource     = inputSelection.getValue();

            if (csvSource.equals(AppData.CSV_SOURCE_FILE))
            {
                sourceChanged= !mFileNameBuffer.isBufferEqual();
                sourceChanged= sourceChanged || !mFirstRowIsHeaderBuffer.isBufferEqual();
                sourceChanged= sourceChanged || !mCSVSeparatorBuffer.isBufferEqual();
            }
            else if (csvSource.equals(AppData.CSV_SOURCE_URL))
            {
                sourceChanged= !mURLNameBuffer.isBufferEqual();
                sourceChanged= sourceChanged || !mFirstRowIsHeaderBuffer.isBufferEqual();
                sourceChanged= sourceChanged || !mCSVSeparatorBuffer.isBufferEqual();
            }
            else if (csvSource.equals(AppData.CSV_SOURCE_TREE))
            {
                sourceChanged= !mTreeNameBuffer.isBufferEqual();
            }
        }
        else
        {
            // the type or the encoding changed
            sourceChanged= true;
        }

        // write the buffered models back to the real models
        mTrigger.setValueForced(Boolean.TRUE);

        // notify the display if something changed
        if (sourceChanged)
        {
            postAppEvent(new DataSourceChanged(this));
        }
    }

    /**
     * Act on "Cancel" or "ESCAPE" in the options dialog, we just hide the dialog.
     */
    public void cancel()
    {
        // hide dialog
        JDialog optionsDialog= (JDialog) mBuilder.getContainerByName("Frame/OptionsDialog");
        optionsDialog.setVisible(false);
    }

    /**
     * Set default values.
     */
    public void defaults()
    {
        // set the buffering value models to the defaults
        mInputSelectionBuffer   .setValue(AppData.CSV_SOURCE_FILE);
        mFileNameBuffer         .setValue(AppData.DEFAULT_FILE_NAME);
        mURLNameBuffer          .setValue(AppData.DEFAULT_URL_NAME);
        mTreeNameBuffer         .setValue(System.getProperty("user.home"));
        mFirstRowIsHeaderBuffer .setValue(Boolean.valueOf(AppData.FIRST_ROW_IS_HEADER));
        mCSVSeparatorBuffer     .setValue(AppData.CSV_SEPARATOR);
        mExeActiveBuffer        .setValue(AppData.EXE_PROGRAM_ACTIVE);
        mExeNameBuffer          .setValue(AppData.EXE_PROGRAM_NAME);
        mPLAFSelectionBuffer    .setValue(AppData.PLAF_DEFAULT_NAME);
        mEncodingSelectionBuffer.setValue(AppData.ENCODING_DEFAULT_NAME);
        mCopyRowSelectionBuffer .setValue(Boolean.valueOf(AppData.COPY_WHOLE_ROW));
    }

    /**
     * Fill in default file field via a file chooser.
     */
    public void fileSearch()
    {
        showChooser(new CSVFileFilter(), JFileChooser.FILES_ONLY, BASE+"Source/FileName");
    }

    /**
     * Fill in default tree field via a file chooser.
     */
    public void treeSearch()
    {
        showChooser(null, JFileChooser.DIRECTORIES_ONLY, BASE+"Source/TreeName");
    }

    /**
     * Fill in default exe field via a file chooser.
     */
    public void exeSearch()
    {
        showChooser(new EXEFileFilter(), JFileChooser.FILES_ONLY, BASE+"Action/EXEName");
    }

    /**
     * Shows a file dialog for CSV files, and sets the text field if one
     * was chosen. We also remember the current dir in that case.
     * @param pFileFilter filter for the file dialog, may be null
     * @param pFileSelectionMode  JFileChooser.FILES_AND_DIRECTORIES,
     * JFileChooser.FILES_ONLY, JFileChooser.DIRECTORIES_ONLY
     * @param pTextFieldName name of the text field we should update
     */
    private void showChooser(final FileFilter pFileFilter, final int pFileSelectionMode, final String pTextFieldName)
    {
        JFrame       frame      = (JFrame)mBuilder.getContainerByName("Frame");
        String       dir        = mApp.getProperty(AppData.DEFAULT_DIR_KEY,
                                                   AppData.DEFAULT_DIR);
        JFileChooser fileChooser= new JFileChooser(dir);
        if (pFileFilter!=null)
        {
            fileChooser.addChoosableFileFilter(pFileFilter);
        }
        fileChooser.setFileSelectionMode(pFileSelectionMode);

        int back= fileChooser.showOpenDialog(frame);
        if (back== JFileChooser.APPROVE_OPTION)
        {
            File   file    = fileChooser.getSelectedFile();
            String fileName= null;
            //noinspection EmptyCatchBlock
            try
            {
                fileName= file.getCanonicalPath();
            }
            catch (IOException ignored)
            {
            }

            if (fileName!=null)
            {
                // set the text field
                JTextField textField= (JTextField) mBuilder.getComponentByName(pTextFieldName);
                textField.setText(fileName);

                // remember our directory
                mApp.setProperty(AppData.DEFAULT_DIR_KEY, file.getParent());
            }
        }
    }
}
