package net.sf.cuf.csvview.util;

import javax.swing.table.AbstractTableModel;
import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * A TreeTableModel generates its data from recursivly parsing a
 * directory tree.
 * Its rows are all found files/directories, its columns are path,
 * name, size and last modification date of the file/directory.
 */
public class TreeTableModel extends AbstractTableModel implements FilteredTableModel
{
    /** total number of rows */
    private int mRowCount;
    /** number of rows after applying the filter */
    private int        mFilteredRows;
    /** mapper from filtered rows to rows in mData */
    private int[]      mFilteredRowsIndexMapper;
    /** data of our model */
    private Object[][] mData;

    /** our table headers. */
    public static final String[] HEADER     = {"path",       "name",       "size",     "last modified"};
    /** our column classes. */
    public static final Class<?> [] HEADERCLASS= {String.class, String.class, Long.class, Date.class};

    /**
     * Creates a new TreeTableModel by parsing the directories starting at the
     * handed tree.
     * @param pRootNode the root node, must not be null
     * @throws IllegalArgumentException if pRootNode is null
     */
    public TreeTableModel(final File pRootNode)
    {
        if (pRootNode==null)
        {
            throw new IllegalArgumentException("root file node must not be null");
        }

        // traverseTree adds all data to the list
        List<Object[]> rows= new ArrayList<Object[]>();
        traverseTree(pRootNode, rows);

        // convert the list to faster data structures
        mRowCount               = rows.size();
        mFilteredRows           = mRowCount;
        mData                   = new Object[mRowCount][HEADER.length];
        mFilteredRowsIndexMapper= new int[mRowCount];
        for (int i = 0; i < mRowCount; i++)
        {
            mFilteredRowsIndexMapper[i]= i;
            Object[] row=  rows.get(i);
            System.arraycopy(row, 0, mData[i], 0, row.length);
        }
    }

    /**
     * Small helper to traverse a tree.
     * @param pCurrentNode the current node
     * @param pRows the list we add our file information to
     */
    private void traverseTree(final File pCurrentNode, final List<Object[]> pRows)
    {
        if (!pCurrentNode.canRead() || !pCurrentNode.exists())
            return;

        Object[] row= new Object[HEADER.length];
        row[0]= pCurrentNode.getPath();
        row[1]= pCurrentNode.getName();
        row[2]= pCurrentNode.length();
        row[3]= new Date(pCurrentNode.lastModified());
        pRows.add(row);

        if (pCurrentNode.isDirectory())
        {
            File[] children= pCurrentNode.listFiles();
            if (children!=null)
            {
                for (final File child : children)
                {
                    traverseTree(child, pRows);
                }
            }
        }
    }

    public int getAllRowsCount()
    {
        return mRowCount;
    }

    public int getColumnCount()
    {
        return HEADER.length;
    }

    public Class<?> getColumnClass(final int pColumnIndex)
    {
        return HEADERCLASS[pColumnIndex];
    }

    public int getRowCount()
    {
        return mFilteredRows;
    }

    public Object getValueAt(final int pRowIndex, final int pColumnIndex)
    {
        int      realIndex= mFilteredRowsIndexMapper[pRowIndex];
        Object[] row      = mData[realIndex];
        return row[pColumnIndex];
    }

    public String getColumnName(final int pColumnIndex)
    {
        return HEADER[pColumnIndex];
    }

    public boolean filter(final String pFilterExpression)
    {
        Pattern filter;
        try
        {
            filter= Pattern.compile(pFilterExpression,Pattern.CASE_INSENSITIVE);
        }
        catch (PatternSyntaxException ignored)
        {
            return false;
        }
        Matcher match= filter.matcher("");

        mFilteredRows= 0;
        for (int i = 0; i < mData.length; i++)
        {
            Object[] row = mData[i];
            for (final Object column : row)
            {
                match.reset(column.toString());
                if (match.find())
                {
                    mFilteredRowsIndexMapper[mFilteredRows] = i;
                    mFilteredRows++;
                    break;
                }
            }
        }
        fireTableDataChanged();
        return true;
    }

    public void filterReset()
    {
        mFilteredRows= mRowCount;
        for (int i = 0; i < mRowCount; i++)
        {
            mFilteredRowsIndexMapper[i]= i;
        }
        fireTableDataChanged();
    }

    /**
     * We simply ignore the setFirstRowIsHeader call.
     * @param pFirstRowIsHeader not used
     */
    public void setFirstRowIsHeader(final boolean pFirstRowIsHeader)
    {
    }
}
