package net.sf.aguacate.swagger.configuration.spi;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.IOUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;

import net.sf.aguacate.configuration.ConfigurationCoupling;
import net.sf.aguacate.swagger.configuration.SwaggerConfiguration;
import net.sf.aguacate.swagger.configuration.SwaggerConfigurationCoupling;
import net.sf.aguacate.util.codec.bridge.CodecCoupling;
import net.sf.aguacate.util.dynamic.bridge.Dynamic;
import net.sf.aguacate.util.dynamic.bridge.DynamicFactoryCoupling;
import net.sf.aguacate.util.resource.ResourceLocator;
import net.sf.aguacate.util.resource.impl.ResourceLocatorClassImpl;
import net.sf.aguacate.util.resource.impl.ResourceLocatorFileImpl;

public class SwaggerConfigurationSpi implements SwaggerConfiguration {

	private static final Logger LOGGER = LogManager.getFormatterLogger(SwaggerConfigurationCoupling.class);

	private static final ObjectMapper YAML_MAPPER = new ObjectMapper(new YAMLFactory());

	private static final String JS_FILENAME = "convert.js";

	private static final String EXTENSION = "js";

	private static final String SUFFIX = ".json";

	private static final String ENVIRONMENT = "DIRECTORY_SWAGGER";

	private static final File DIRECTORY;

	private static final Dynamic DYNAMIC;

	private final ResourceLocator locator;

	static {
		assert JS_FILENAME.endsWith("." + EXTENSION);
		String temp = System.getProperty(ENVIRONMENT);
		if (temp == null || temp.isEmpty()) {
			temp = System.getenv(ENVIRONMENT);
			if (temp == null || temp.isEmpty()) {
				LOGGER.info("No " + ENVIRONMENT + " defined, using default");
				DIRECTORY = null;
			} else {
				LOGGER.info("using " + ENVIRONMENT + " (env): {}", temp);
				DIRECTORY = new File(temp);
			}
		} else {
			LOGGER.info("using " + ENVIRONMENT + " (prop): {}", temp);
			DIRECTORY = new File(temp);
		}
		InputStream inputStream = SwaggerConfigurationSpi.class.getResourceAsStream(JS_FILENAME);
		if (inputStream == null) {
			throw new IllegalStateException("File " + JS_FILENAME + " not found");
		} else {
			try {
				DYNAMIC = DynamicFactoryCoupling.getByExtension(EXTENSION)
						.createFrom(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
			} finally {
				try {
					inputStream.close();
				} catch (IOException e) {
					LOGGER.warn("On resource close", e);
				}
			}
		}
	}

	public SwaggerConfigurationSpi() {
		if (DIRECTORY == null) {
			this.locator = new ResourceLocatorClassImpl(SwaggerConfigurationCoupling.class);
		} else {
			this.locator = new ResourceLocatorFileImpl(DIRECTORY);
		}
	}

	@Override
	public String getConfiguration(String name) {
		try {
			String swagger = loadSwagger(name);
			if (swagger == null) {
				return null;
			} else {
				Object[] parameters = parameters(swagger);
				LOGGER.trace("before script");
				String result = DYNAMIC.typedExec("convert", swagger, parameters);
				LOGGER.trace("after script");
				return YAML_MAPPER.writeValueAsString(CodecCoupling.jsonCodecBridge().decodeMap(result));
			}
		} catch (IOException e) {
			throw new IllegalStateException(e);
		}
	}

	String[] parameters(String swagger) throws IOException {
		Map<String, Object> conf = CodecCoupling.jsonCodecBridge().decodeMap(swagger);
		@SuppressWarnings("unchecked")
		List<String> configuration = (List<String>) conf.get("configurations");
		int size = configuration.size();
		String[] params = new String[size];
		for (int i = 0; i < size; i++) {
			String name = configuration.get(i);
			String string = loadConfiguration(name);
			if (string == null) {
				throw new IllegalStateException(name);
			} else {
				params[i] = string;
			}
		}
		return params;
	}

	String toString(String filename) throws IOException {
		InputStream inputStream = locator.open(filename);
		if (inputStream == null) {
			throw new IllegalStateException("not found ".concat(filename));
		} else {
			return toString(inputStream);
		}
	}

	String loadSwagger(String name) throws IOException {
		String filename = name.concat(SUFFIX);
		LOGGER.debug(filename);
		return load0(locator, filename);
	}

	String loadConfiguration(String name) throws IOException {
		String filename = name.concat(SUFFIX);
		LOGGER.debug(filename);
		return load0(ConfigurationCoupling.get().getLocator(), filename);
	}

	String load0(ResourceLocator localLocator, String filename) throws IOException {
		InputStream inputStream = localLocator.open(filename);
		if (inputStream == null) {
			return null;
		} else {
			return toString(inputStream);
		}
	}

	String toString(InputStream inputStream) throws IOException {
		try {
			return IOUtils.toString(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
		} finally {
			try {
				inputStream.close();
			} catch (IOException e) {
				LOGGER.warn("On resource close", e);
			}
		}
	}

}
