'use strict';
(function() {

	function toSwaggerSchemaBoolean() {
		return {
			type : 'boolean',
			nullable : false
		};
	}

	function toSwaggerSchemaString(name, attributes) {
		return {
			type : 'string',
			nullable : false,
			minLength : attributes['minlen'],
			maxLength : attributes['maxlen'],
			pattern : attributes['regex']
		};
	}

	function toSwaggerSchemaI64(name, attributes) {
		// exclusiveMinimum
		// exclusiveMaximum
		// multipleOf
		return {
			type : 'integer',
			nullable : false,
			format : 'int64',
			minimum : parseInt(attributes['minval']),
			maximum : parseInt(attributes['maxval'])
		};
	}

	function toSwaggerSchemaDouble(name, attributes) {
		// exclusiveMinimum
		// exclusiveMaximum
		// multipleOf
		return {
			type : 'number',
			nullable : false,
			format : 'double',
			minimum : parseFloat(attributes['minval']),
			maximum : parseFloat(attributes['maxval'])
		};
	}

	function toSwaggerSchemaDate(name, attributes) {
		return {
			type : 'string',
			nullable : false,
			format : 'date',
			minLength : 10,
			maxLength : 10
		};
	}

	function toSwaggerSchemaTime(name, attributes) {
		return {
			type : 'string',
			nullable : false,
			format : 'time',
			minLength : 10,
			maxLength : 10
		};
	}

	function toSwaggerSchemaDateTime(name, attributes) {
		return {
			type : 'string',
			nullable : false,
			format : 'date-time',
			minLength : 10,
			maxLength : 10
		};
	}

	function toSwaggerSchemaArray(name, attributes) {
		return {
			type : 'array',
			nullable : false,
			items : []
		};
	}

	function toSwaggerParameter(name, required, attributes) {
		var type = attributes['type'];
		var schema;
		switch (type) {
		case 'TIME':
			schema = toSwaggerSchemaTime(name, attributes);
			break;
		case 'DATETIME':
			schema = toSwaggerSchemaDateTime(name, attributes);
			break;
		case 'DYNAMIC_DATE':
		case 'DATE':
			schema = toSwaggerSchemaDate(name, attributes);
			break;
		case 'STRING':
			schema = toSwaggerSchemaString(name, attributes);
			break;
		case 'INTEGER':
			schema = toSwaggerSchemaI64(name, attributes);
			break;
		case 'FLOAT':
			schema = toSwaggerSchemaDouble(name, attributes);
			break;
		case 'BOOLEAN':
			schema = toSwaggerSchemaBoolean(name, attributes);
			break;
		case 'STRUCTURE_ARRAY':
		default:
			throw 'Illegal type: ' + type;
		}
		return {
			name : name,
			required : required,
			'in' : 'path',
			schema : schema
		};
	}

	function toProperty(required, attributes) {
		var type = attributes['type'];
		var data;
		switch (type) {
		case 'TIME':
			data = {
				type : 'string',
				format : 'time'
			};
			break;
		case 'DATETIME':
			data = {
				type : 'string',
				format : 'date-time'
			};
			break;
		case 'DYNAMIC_DATE':
		case 'DATE':
			data = {
				type : 'string',
				format : 'date'
			};
			break;
		case 'STRING':
			data = {
				type : 'string'
			};
			break;
		case 'INTEGER':
			data = {
				type : 'integer',
				format : 'int64'
			};
			break;
		case 'FLOAT':
			data = {
				type : 'number',
				format : 'double'
			};
			break;
		case 'BOOLEAN':
			data = {
				type : 'boolean'
			};
			break;
		case 'ARRAY': {
			var temp = attributes['data'];
			var subtype;
			switch (temp['type']) {
			case 'INTEGER':
				subtype = 'integer';
				break;
			case 'STRING':
				subtype = 'string';
				break;
			default:
				throw new 'Not supported item array type: ' + temp['type'];
			}
			data = {
				type : 'array',
				items : {
					type : subtype
				}
			};
			break;
		}
		case 'STRUCTURE_ARRAY': {
			var temp = {
				type : 'object',
				properties : {}
			};
			var properties = temp['properties'];
			var d = attributes['data'];
			for ( var key in d) {
				properties[key] = toProperty(false, d[key]);
			}
			data = {
				type : 'array',
				items : temp
			};
			break;
		}
		case 'STRUCTURE': {
			var temp = {
				type : 'object',
				properties : {}
			};
			var properties = temp['properties'];
			var d = attributes['data'];
			for ( var key in d) {
				properties[key] = toProperty(false, d[key]);
			}
			data = temp;
			break;
		}
		case 'ARRAY':
		default:
			throw 'Illegal type: ' + type;
		}
		return data;
	}

	function normalizeMethods(methods) {
		var normalized = {};
		for ( var methodName in methods) {
			var newName;
			if ('GET0' === methodName) {
				newName = 'get';
			} else {
				newName = methodName.toLowerCase();
			}
			normalized[newName] = methods[methodName];
		}
		return normalized;
	}

	function selectMethods(methodName, methods0, methods1) {
		var method;
		switch (methodName) {
		case 'GET0':
		case 'PUT':
			method = methods0[methodName];
			if (method === undefined) {
				method = {};
				methods0[methodName] = method;
			}
			break;
		case 'GET':
		case 'DELETE':
		case 'POST':
		case 'PATCH':
			method = methods1[methodName];
			if (method === undefined) {
				method = {};
				methods1[methodName] = method;
			}
			break;
		default:
			throw 'Illegal method: ' + methodName;
		}
		return method;
	}

	function processInput(entity, primary, fieldName, attributes, methods0,
			methods1) {
		var input = attributes['input'];
		for ( var methodName in input) {
			var method = selectMethods(methodName, methods0, methods1);
			if (primary === fieldName) {
				var parameters = method['parameters'];
				if (parameters == undefined) {
					parameters = [];
					method['parameters'] = parameters;
				}
				var required = input[methodName]['mandatory'];
				var parameter = toSwaggerParameter(fieldName, required,
						attributes);
				parameters.push(parameter);
			} else {
				var operationName = entity + '-' + methodName;
				var requestBody = method['requestBody'];
				if (requestBody === undefined) {
					requestBody = {
						required : true,
						content : {
							'application/json; charset=utf-8' : {
								schema : {
									type : 'object',
									properties : {}
								}
							}
						}
					};
					method['requestBody'] = requestBody;
				}
				requestBody['content']['application/json; charset=utf-8']['schema']['properties'][fieldName] = toProperty(
						required, attributes);
			}
		}
	}

	function processOutput(entity, primary, fieldName, attributes, methods0,
			methods1) {
		var output = attributes['output'];
		for (var i = 0; i < output.length; i++) {
			var methodName = output[i];
			var method = selectMethods(methodName, methods0, methods1);
			var responses = method['responses'];
			if (responses == undefined) {
				responses = {
					'200' : {
						description : 'OK.',
						content : {
							'application/json' : {
								schema : {
									type : 'object',
									properties : {}
								}
							}
						}
					},
					'400' : {
						description : 'Bad request.'
					},
					'404' : {
						description : 'Data was not found.'
					},
					'5XX' : {
						description : 'Unexpected error.'
					}
				};
				method['responses'] = responses;
			}
			responses['200']['content']['application/json']['schema']['properties'][fieldName] = toProperty(
					false, attributes);
		}
	}

	function processFields(entity, fields, primary) {
		var methods1 = {};
		var methods0 = {};
		for ( var fieldName in fields) {
			var attributes = fields[fieldName];
			processInput(entity, primary, fieldName, attributes, methods0,
					methods1);
			processOutput(entity, primary, fieldName, attributes, methods0,
					methods1);
		}
		var method = methods0['PUT'];
		if (method !== undefined) {
			var responses = method['responses'];
			if (responses === undefined) {
				responses = {
					'201' : {
						description : 'Created.'
					},
					'400' : {
						description : 'Bad request.'
					},
					'5XX' : {
						description : 'Unexpected error.'
					}
				};
				method['responses'] = responses;
			}
		}
		var method = methods1['DELETE'];
		if (method !== undefined) {
			var responses = method['responses'];
			if (responses === undefined) {
				responses = {
					'200' : {
						description : 'OK.'
					},
					'400' : {
						description : 'Bad request.'
					},
					'5XX' : {
						description : 'Unexpected error.'
					}
				};
				method['responses'] = responses;
			}
		}
		var method = methods1['PATCH'];
		if (method !== undefined) {
			var responses = method['responses'];
			if (responses === undefined) {
				responses = {
					'200' : {
						description : 'OK.'
					},
					'400' : {
						description : 'Bad request.'
					},
					'404' : {
						description : 'Data was not found.'
					},
					'5XX' : {
						description : 'Unexpected error.'
					}
				};
				method['responses'] = responses;
			}
		}
		var method = methods0['GET0'];
		if (method !== undefined) {
			var parameters = method['parameters'];
			if (parameters === undefined) {
				parameters = [];
				method['parameters'] = parameters;
			}
			var parameter = toSwaggerParameter('Page number', false, {
				minval : '0',
				maxval : '100000',
				type : 'INTEGER'
			});
			parameter['in'] = 'query';
			parameter['description'] = 'page';
			parameters.push(parameter);
			var parameter = toSwaggerParameter('size', false, {
				minval : '0',
				maxval : '100000',
				type : 'INTEGER'
			});
			parameter['in'] = 'query';
			parameter['description'] = 'Elements per page';
			parameters.push(parameter);
		}
		return [ normalizeMethods(methods0), normalizeMethods(methods1) ];
	}

	function addProps(source, target) {
		for ( var method in target) {
			var map0 = source[method];
			if (map0 !== undefined) {
				var map1 = target[method];
				for ( var name in map0) {
					map1[name] = map0[name];
				}
			}
		}
	}

	function process(paths, name, data, info) {
		var source = JSON.parse(data);
		var fields = source['fields'];
		var primary = source['primary'];
		var methods = processFields(name, fields, primary);
		var methods0 = methods[0];
		addProps(info[''], methods0);
		var methods1 = methods[1];
		var temp1 = '{' + primary + '}';
		addProps(info[temp1], methods1);
		var path0 = '/' + name;
		var path = path0 + '/' + temp1;
		paths[path0] = methods0;
		paths[path] = methods1;
	}

	return {
		convert : function(swagger, data) {
			var s = JSON.parse(swagger);
			var configurations = s['configurations'];
			var pathsInfo = s['paths'];
			var paths = {};
			var length = configurations.length;
			for (var i = 0; i < length; i++) {
				var configuration = configurations[i];
				var info = pathsInfo[configuration];
				process(paths, configuration, data[i], info);
			}
			var response = s['metadata'];
			response['openapi'] = '3.0.1';
			response['paths'] = paths;
			return JSON.stringify(response);
		}
	}

})();
